<?php

namespace tasks\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use tasks\Models\DatabaseTableTasks;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Vrsta extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTasks $vrsteTable;
    private Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTasks $vrsteTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->vrsteTable = $vrsteTable;
        $this->validator = $validator;
    }

    public function edit(Request $request): Response 
    {
        try {
            $vrsta = null;
            $editToken = null;

            if ($id = $request->getParam('id')) {
                $vrsta = $this->vrsteTable->findById($id);
                if ($vrsta) {
                    $editToken = $this->createEditToken($vrsta->id, 'vrsta');
                }
            }

            return $this->response()->render(
                'tasks/resources/views/vrsta/edit.html.php',
                [
                    'vrsta' => $vrsta,
                    'edit_token' => $editToken
                ],
                'Edit Vrsta'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading task type edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load task type edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $vrsta = $request->getPost('vrsta');
            $vrstaId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'vrsta');
                if (isset($tokenResult['error'])) {
                    return $this->renderVrstaForm(
                        $vrsta, 
                        $editToken, 
                        $tokenResult
                    );
                }
                // Token je valjan
                $vrstaId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $vrsta['id'] = $vrstaId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'vrsta[name]' => ['required', 'alpha_spaces', ['max', 255]],
                'vrsta[orgjedinica]' => ['required', 'plain_text', ['max', 255]]
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($vrstaId) {
                    if ($validToken) {
                        $newEditToken = $this->refreshEditToken($validToken, 'vrsta');
                    } else {
                        $newEditToken = $this->createEditToken($vrstaId, 'vrsta');
                    }
                }

                return $this->renderVrstaForm(
                    $vrsta, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 5. SPREMANJE PODATAKA
            $this->vrsteTable->save($vrsta);
            
            // 6. PREUSMJERAVANJE
            return Response::redirect('tasktype/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving task type', [
                'message' => $e->getMessage(),
                'vrsta' => $vrsta ?? null
            ]);
            return $this->response()->renderError('Failed to save task type');
        }
    }

    public function list(): Response 
    {
        try {
            $vrste = $this->vrsteTable->findAll();
            $this->sessionManager->set('valid_vrste_ids', array_map(fn($vrsta) => $vrsta->id, $vrste));

            return $this->response()->render(
                'tasks/resources/views/vrsta/list.html.php',
                [
                    'user' => $this->authentication->getUser(),
                    'vrste' => $vrste
                ],
                'Sve Vrste'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading task types list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load task types list');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $vrstaId = $request->getPost('id');
            
            // Sigurnosna provjera da li je ID na listi valjanih ID-ova
            $validVrsteIds = $this->sessionManager->get('valid_vrste_ids', []);
            if (!in_array($vrstaId, $validVrsteIds ?? [])) {
                $this->logger->security('Unauthorized task type deletion attempt', [
                    'id' => $vrstaId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Brisanje vrste
            $this->vrsteTable->delete($vrstaId);
            $this->removeFromSession($vrstaId);

            return Response::redirect('tasktype/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting task type', [
                'message' => $e->getMessage(),
                'id' => $vrstaId ?? null
            ]);
            return $this->response()->renderError('Failed to delete task type');
        }
    }

    private function removeFromSession(int $vrstaId): void 
    {
        $validIds = $this->sessionManager->get('valid_vrste_ids', []);
        $validIds = array_filter($validIds, fn($id) => $id != $vrstaId);
        $this->sessionManager->set('valid_vrste_ids', $validIds);
    }

    /**
     * Helper method to render the vrsta form with errors
     */
    private function renderVrstaForm(array $vrsta, ?string $editToken, array $errors): Response
    {
        return $this->response()->render(
            'tasks/resources/views/vrsta/edit.html.php',
            [
                'vrsta' => (object)$vrsta,
                'edit_token' => $editToken,
                'errors' => $errors
            ],
            'Edit Vrsta'
        );
    }
}