<?php

namespace tasks\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use tasks\Models\DatabaseTableTasks;
use tasks\Models\WorkPerformedTable;

class WorkPerformed extends BaseController
{
    protected DatabaseTable $korisniciTable;
    protected DatabaseTableTasks $zadatciTable;
    protected WorkPerformedTable $workPerformedTable;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTasks $zadatciTable,
        WorkPerformedTable $workPerformedTable,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->zadatciTable = $zadatciTable;
        $this->workPerformedTable = $workPerformedTable;
    }

    public function index(Request $request): Response
    {
        $zadatakId = (int)$request->getParam('id');
        
        if (!$zadatakId) {
            return Response::redirect('tasks/list');
        }
        
        $zadatak = $this->zadatciTable->findById($zadatakId);
        
        if (!$zadatak) {
            return Response::redirect('tasks/list');
        }
        
        $workItems = $this->workPerformedTable->getWorkPerformedByZadatak($zadatakId);
        $summary = $this->workPerformedTable->getWorkPerformedSummary($zadatakId);
        
        return $this->response()->render(
            'tasks/resources/views/workperformed/index.html.php',
            [
                'zadatak' => $zadatak,
                'workItems' => $workItems,
                'summary' => $summary,
                'title' => 'Work Performed - Task #' . $zadatakId
            ],
            'Work Performed - Task #' . $zadatakId
        );
    }
    
    public function search(Request $request): Response
    {
        $type = $request->getPost('type');
        $query = $request->getPost('query');
        $category = $request->getPost('category') ?? '';
        
        if (empty($type) || empty($query) || strlen($query) < 2) {
            return Response::json(['results' => []]);
        }
        
        $results = $this->workPerformedTable->searchCatalogItems($type, $query, $category);
        
        return Response::json(['results' => $results]);
    }
    
    public function addItem(Request $request): Response
    {
        $zadatakId = (int)$request->getPost('zadatak_id');
        $itemId = (int)$request->getPost('item_id');
        $type = $request->getPost('type');
        $quantity = (float)$request->getPost('quantity');
        
        if (!$zadatakId || !$itemId || empty($type) || $quantity <= 0) {
            return Response::json([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
        }
        
        $catalogItem = $this->workPerformedTable->getCatalogItem($type, $itemId);
        
        if (!$catalogItem) {
            return Response::json([
                'success' => false,
                'message' => 'Item not found'
            ]);
        }
        
        $itemData = [
            'zadatak_id' => $zadatakId,
            'item_id' => $itemId,
            'description' => $catalogItem['description'],
            'category' => $catalogItem['category'],
            'quantity' => $quantity,
            'unit' => $catalogItem['unit'],
            'price' => $catalogItem['price'],
            'total' => $quantity * $catalogItem['price'],
            'type' => $type,
            'date_added' => date('Y-m-d H:i:s')
        ];
        
        $newItemId = $this->workPerformedTable->addWorkItem($itemData);
        
        // Update the main task's ticket_work value with total work performed
        $summary = $this->workPerformedTable->getWorkPerformedSummary($zadatakId);
        $this->zadatciTable->save([
            'id' => $zadatakId,
            'ticket_work' => $summary['total_amount']
        ]);
        
        return Response::json([
            'success' => true,
            'item' => array_merge($itemData, ['id' => $newItemId]),
            'summary' => $summary
        ]);
    }
    
    public function updateItem(Request $request): Response
    {
        $itemId = (int)$request->getPost('id');
        $zadatakId = (int)$request->getPost('zadatak_id');
        $quantity = (float)$request->getPost('quantity');
        
        if (!$itemId || !$zadatakId || $quantity <= 0) {
            return Response::json([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
        }
        
        $item = $this->workPerformedTable->findById($itemId);
        
        if (!$item || $item->zadatak_id != $zadatakId) {
            return Response::json([
                'success' => false,
                'message' => 'Item not found'
            ]);
        }
        
        $updatedData = [
            'id' => $itemId,
            'quantity' => $quantity,
            'total' => $quantity * $item->price
        ];
        
        $this->workPerformedTable->updateWorkItem($itemId, $updatedData);
        
        // Update the main task's ticket_work value with total work performed
        $summary = $this->workPerformedTable->getWorkPerformedSummary($zadatakId);
        $this->zadatciTable->save([
            'id' => $zadatakId,
            'ticket_work' => $summary['total_amount']
        ]);
        
        return Response::json([
            'success' => true,
            'item' => array_merge((array)$item, $updatedData),
            'summary' => $summary
        ]);
    }
    
    public function deleteItem(Request $request): Response
    {
        $itemId = (int)$request->getPost('id');
        $zadatakId = (int)$request->getPost('zadatak_id');
        
        if (!$itemId || !$zadatakId) {
            return Response::json([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
        }
        
        $item = $this->workPerformedTable->findById($itemId);
        
        if (!$item || $item->zadatak_id != $zadatakId) {
            return Response::json([
                'success' => false,
                'message' => 'Item not found'
            ]);
        }
        
        $this->workPerformedTable->delete($itemId);
        
        // Update the main task's ticket_work value with total work performed
        $summary = $this->workPerformedTable->getWorkPerformedSummary($zadatakId);
        $this->zadatciTable->save([
            'id' => $zadatakId,
            'ticket_work' => $summary['total_amount'] ?? 0
        ]);
        
        return Response::json([
            'success' => true,
            'summary' => $summary
        ]);
    }
}