<?php

namespace tasks\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;

class DatabaseTableTasks extends DatabaseTable
{
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor with logger support
     * 
     * @param \baseKRIZAN\Database\DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        \baseKRIZAN\Database\DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableTasks initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all required tables exist
     * 
     * @return void
     */
    private function ensureTablesExist(): void
    {
        $this->ensureZadatakTableExists();
        $this->ensureZadatakCommentsTableExists();
        $this->ensureZadatakOsobaTableExists();
        $this->ensureZadatakOsobeTableExists();
        $this->ensureZadatakVrstaTableExists();
        $this->ensureZadatakVrsteTableExists();
    }
    
    /**
     * Ensure zadatak (main tasks) table exists
     * 
     * @return void
     */
    private function ensureZadatakTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `user_id` int(11) DEFAULT NULL,
                    `ticket_date` datetime DEFAULT NULL,
                    `ticket_type` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_number` bigint(20) DEFAULT NULL,
                    `ticket_connumb` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_address` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_contact` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_priority` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_arrdate` datetime DEFAULT NULL,
                    `ticket_comment` text COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_taken` datetime DEFAULT NULL,
                    `ticket_resolved` datetime DEFAULT NULL,
                    `ticket_h` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_day` int(11) DEFAULT NULL,
                    `ticket_month` int(11) DEFAULT NULL,
                    `ticket_year` int(11) DEFAULT NULL,
                    `ticket_work` float DEFAULT NULL,
                    `ticket_sharelink` text COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_vrsta` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_osoba` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `ticket_brojdj` int(11) DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created zadatak table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTasks', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureZadatakTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure zadatak_comments table exists
     * 
     * @return void
     */
    private function ensureZadatakCommentsTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak_comments' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak_comments` (
                    `zadatakId` int(11) NOT NULL,
                    `comment_id` int(11) NOT NULL AUTO_INCREMENT,
                    `parent_comment_id` int(11) NOT NULL,
                    `comment_sender_name` varchar(50) COLLATE utf8_general_ci NOT NULL,
                    `comment` varchar(255) COLLATE utf8_general_ci NOT NULL,
                    `date` datetime NOT NULL,
                    PRIMARY KEY (`comment_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created zadatak_comments table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTasks', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureZadatakCommentsTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure zadatak_osoba table exists (relation table)
     * 
     * @return void
     */
    private function ensureZadatakOsobaTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak_osoba' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak_osoba` (
                    `zadatakId` int(11) NOT NULL,
                    `osobaId` int(11) NOT NULL,
                    PRIMARY KEY (`zadatakId`,`osobaId`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created zadatak_osoba table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTasks', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureZadatakOsobaTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure zadatak_osobe table exists (persons table)
     * 
     * @return void
     */
    private function ensureZadatakOsobeTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak_osobe' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak_osobe` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `name` varchar(255) COLLATE utf8_general_ci NOT NULL,
                    `radnomjesto` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `tvrtka` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `area` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `orgjednica` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    `vidljivost` int(11) NOT NULL DEFAULT 1,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created zadatak_osobe table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTasks', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureZadatakOsobeTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure zadatak_vrsta table exists (relation table)
     * 
     * @return void
     */
    private function ensureZadatakVrstaTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak_vrsta' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak_vrsta` (
                    `zadatakId` int(11) NOT NULL,
                    `vrstaId` int(11) NOT NULL,
                    PRIMARY KEY (`zadatakId`,`vrstaId`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created zadatak_vrsta table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTasks', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureZadatakVrstaTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    /**
     * Ensure zadatak_vrste table exists (task types table)
     * 
     * @return void
     */
    private function ensureZadatakVrsteTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak_vrste' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak_vrste` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `name` varchar(255) COLLATE utf8_general_ci NOT NULL,
                    `orgjednica` varchar(255) COLLATE utf8_general_ci DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created zadatak_vrste table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTasks', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureZadatakVrsteTableExists'
                ]);
            }
            throw $e;
        }
    }

    //mytaskscontroller
        public function findZadatciByFilter(string $menuItem, ?string $tip = '', int $limit = 5): array
        {
            $query = 'SELECT ' . $this->table . '.* FROM ' . $this->table;
            $parameters = [];
        
            if (empty($tip)) {
                if (is_numeric($menuItem)) {
                    $query .= " INNER JOIN zadatak_osoba ON {$this->table}.id = zadatak_osoba.zadatakId";
                    $query .= " INNER JOIN zadatak_vrsta ON {$this->table}.id = zadatak_vrsta.zadatakId";
                    $query .= " WHERE zadatak_osoba.osobaId = :menuItem AND zadatak_vrsta.vrstaId = 2";
                    $parameters['menuItem'] = (int)$menuItem;
                }
            } else {
                switch ($tip) {
                    case 'vrsta':
                        $query .= " INNER JOIN zadatak_vrsta ON {$this->table}.id = zadatak_vrsta.zadatakId";
                        $query .= " WHERE zadatak_vrsta.vrstaId = :menuItem";
                        $parameters['menuItem'] = (int)$menuItem;
                        break;
                    case 'osoba':
                        $query .= " INNER JOIN zadatak_osoba ON {$this->table}.id = zadatak_osoba.zadatakId";
                        $query .= " WHERE zadatak_osoba.osobaId = :menuItem";
                        $parameters['menuItem'] = (int)$menuItem;
                        break;
                }
            }
        
            // Direktno ubacujemo limit u query string
            $query .= " ORDER BY {$this->table}.ticket_date DESC, {$this->table}.id DESC LIMIT " . (int)$limit;
        
            return $this->dbConnection->queryAndFetchAllObjects($query, $parameters, $this->className, $this->constructorArgs);
        }

        public function countZadatciByFilter(string $menuItem, ?string $tip = ''): int
        {
            $query = "SELECT COUNT(*) as total FROM {$this->table}";
            $parameters = [];

            if (empty($tip)) {
                if (is_numeric($menuItem)) {
                    $query .= " INNER JOIN zadatak_osoba ON {$this->table}.id = zadatak_osoba.zadatakId";
                    $query .= " INNER JOIN zadatak_vrsta ON {$this->table}.id = zadatak_vrsta.zadatakId";
                    $query .= " WHERE zadatak_osoba.osobaId = :menuItem AND zadatak_vrsta.vrstaId = 2";
                    $parameters['menuItem'] = $menuItem;
                }
            } else {
                switch ($tip) {
                    case 'vrsta':
                        $query .= " INNER JOIN zadatak_vrsta ON {$this->table}.id = zadatak_vrsta.zadatakId";
                        $query .= " WHERE zadatak_vrsta.vrstaId = :menuItem";
                        $parameters['menuItem'] = $menuItem;
                        break;
                    case 'osoba':
                        $query .= " INNER JOIN zadatak_osoba ON {$this->table}.id = zadatak_osoba.zadatakId";
                        $query .= " WHERE zadatak_osoba.osobaId = :menuItem";
                        $parameters['menuItem'] = $menuItem;
                        break;
                }
            }

            return (int)$this->dbConnection->querySingleValue($query, $parameters);
        }

        public function findOsobeByZadatak(int $zadatakId): array
        {
            $query = "SELECT o.id, o.name 
                    FROM zadatak_osoba zo 
                    INNER JOIN zadatak_osobe o ON zo.osobaId = o.id 
                    WHERE zo.zadatakId = :zadatakId";

            return $this->dbConnection->queryAndFetchAllAssoc($query, ['zadatakId' => $zadatakId]);
        }

        public function findVrsteByZadatak(int $zadatakId): array
        {
            $query = "SELECT v.id, v.name 
                    FROM zadatak_vrsta zv 
                    INNER JOIN zadatak_vrste v ON zv.vrstaId = v.id 
                    WHERE zv.zadatakId = :zadatakId";

            return $this->dbConnection->queryAndFetchAllAssoc($query, ['zadatakId' => $zadatakId]);
        }

        public function findOsobeByNames(array $names): array
        {
            if (empty($names)) {
                return [];
            }

            $placeholders = str_repeat('?,', count($names) - 1) . '?';
            $query = "SELECT id, name FROM zadatak_osobe WHERE name IN ($placeholders)";

            return $this->dbConnection->queryAndFetchAllAssoc($query, $names);
        }

        public function findVrsteByNames(array $names): array
        {
            if (empty($names)) {
                return [];
            }

            $placeholders = str_repeat('?,', count($names) - 1) . '?';
            $query = "SELECT id, name FROM zadatak_vrste WHERE name IN ($placeholders)";

            return $this->dbConnection->queryAndFetchAllAssoc($query, $names);
        }

    //comments controller
        public function addComment(array $commentData): void 
        {
            $query = "INSERT INTO zadatak_comments 
                    (parent_comment_id, comment, comment_sender_name, zadatakId, date) 
                    VALUES (:parent_comment_id, :comment, :comment_sender_name, :zadatakId, :date)";
        
            $this->dbConnection->execute($query, [
                ':parent_comment_id' => $commentData['parent_comment_id'],
                ':comment' => $commentData['comment'],
                ':comment_sender_name' => $commentData['comment_sender_name'],
                ':zadatakId' => $commentData['zadatakId'],
                ':date' => $commentData['date']
            ]);
        }
        
        public function findTaskComments(int $zadatakId): array 
        {
            $query = "SELECT * FROM zadatak_comments 
                    WHERE parent_comment_id = '0' AND zadatakId = :zadatakId 
                    ORDER BY date DESC";
        
            return $this->dbConnection->queryAndFetchAllAssoc($query, [':zadatakId' => $zadatakId]);
        }
        
        public function findCommentReplies(int $parentId): array 
        {
            $query = "SELECT * FROM zadatak_comments 
                    WHERE parent_comment_id = :parent_id
                    ORDER BY date ASC";
        
            return $this->dbConnection->queryAndFetchAllAssoc($query, [':parent_id' => $parentId]);
        }
        
        // New method to count replies for a comment
        public function countReplies(int $commentId): int 
        {
            $query = "SELECT COUNT(*) as count FROM zadatak_comments 
                    WHERE parent_comment_id = :comment_id";

            return (int)$this->dbConnection->querySingleValue($query, [':comment_id' => $commentId]);
        }


        /**
         * Rekurzivno broji sve odgovore na komentar (uključujući odgovore na odgovore)
         * 
         * @param int $commentId ID komentara za koji brojimo odgovore
         * @return int Ukupan broj svih odgovora na svim razinama
         */
        public function countAllReplies(int $commentId): int 
        {
            // Prvo dohvati direktne odgovore
            $query = "SELECT comment_id FROM zadatak_comments 
                    WHERE parent_comment_id = :comment_id";
                    
            $replies = $this->dbConnection->queryAndFetchAllAssoc($query, [':comment_id' => $commentId]);
            
            // Inicijalni broj je broj direktnih odgovora
            $count = count($replies);
            
            // Za svaki direktni odgovor, rekurzivno zbroji i njegove odgovore
            foreach ($replies as $reply) {
                $count += $this->countAllReplies($reply['comment_id']);
            }
            
            return $count;
        }
        
    // Admin statistike
        public function getTaskTypesCount(): int 
        {
            $query = "SELECT COUNT(*) as count FROM zadatak_vrste";
            return (int)$this->dbConnection->querySingleValue($query);
        }
        
        public function getUsersCount(): int 
        {
            $query = "SELECT COUNT(*) as count FROM zadatak_osobe WHERE vidljivost = 1";
            return (int)$this->dbConnection->querySingleValue($query);
        }
        
        public function getCompletedTasksCount(): int 
        {
            $query = "SELECT COUNT(*) as count FROM {$this->table} WHERE ticket_vrsta = 'Realizirano'";
            return (int)$this->dbConnection->querySingleValue($query);
        }
        
        public function getInProgressTasksCount(): int 
        {
            $query = "SELECT COUNT(*) as count FROM {$this->table} WHERE ticket_vrsta = 'Preuzeto'";
            return (int)$this->dbConnection->querySingleValue($query);
        }
        
        public function getTasksByStatus(): array 
        {
            $query = "SELECT ticket_vrsta as status, COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE ticket_vrsta IS NOT NULL 
                    GROUP BY ticket_vrsta 
                    ORDER BY count DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        }
        
        public function getTasksByType(): array 
        {
            $query = "SELECT ticket_type as type, COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE ticket_type IS NOT NULL 
                    GROUP BY ticket_type 
                    ORDER BY count DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        }
        
        public function getTopPerformers(int $limit = 5): array 
        {
            $query = "SELECT ticket_osoba as name, COUNT(*) as task_count, 
                    SUM(ticket_work) as total_work 
                    FROM {$this->table} 
                    WHERE ticket_osoba IS NOT NULL 
                    GROUP BY ticket_osoba 
                    ORDER BY task_count DESC 
                    LIMIT " . (int)$limit;
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        }
        
        public function getMonthlyTaskCounts(): array 
        {
            $query = "SELECT ticket_month as month, COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE ticket_year = YEAR(CURRENT_DATE()) 
                    AND ticket_month IS NOT NULL 
                    GROUP BY ticket_month 
                    ORDER BY ticket_month ASC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        }
        
        public function getTotalRevenue(): float 
        {
            $query = "SELECT SUM(ticket_work) as total FROM {$this->table} WHERE ticket_work IS NOT NULL";
            return (float)$this->dbConnection->querySingleValue($query);
        }
        
        public function getAverageResolutionTime(): float 
        {
            $query = "SELECT AVG(ticket_h) as average 
                    FROM {$this->table} 
                    WHERE ticket_h > 0";
            return (float)$this->dbConnection->querySingleValue($query);
        }
        
        public function getTasksByPriority(): array 
        {
            $query = "SELECT ticket_priority as priority, COUNT(*) as count 
                    FROM {$this->table} 
                    WHERE ticket_priority IS NOT NULL 
                    GROUP BY ticket_priority 
                    ORDER BY count DESC";
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        }
        
        public function getRecentTasks(int $limit = 5): array 
        {
            $query = "SELECT id, ticket_date, ticket_type, ticket_address, ticket_vrsta 
                    FROM {$this->table} 
                    ORDER BY ticket_date DESC 
                    LIMIT " . (int)$limit;
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        }
}