<?php

namespace tasks\Models;

use baseKRIZAN\Database\DatabaseConnection;
use baseKRIZAN\Error\Logger;

class WorkPerformedTable extends DatabaseTableTasks
{
    // Add a protected logger property that we can access from this class
    protected ?Logger $wpLogger;
    
    // Add mapping for ticket types to table names
    private array $catalogTableMapping = [
        'A1' => 'zadatak_troskovnika1',
        'HT' => 'zadatak_troskovnikht', 
        'TMHR' => 'zadatak_troskovniktmhr'
    ];
    
    public function __construct(
        DatabaseConnection $dbConnection,
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, 'zadatak_wp', 'id', 'stdClass', [], $logger);
        
        // Store the logger in our own property so we can access it
        $this->wpLogger = $logger;
        
        // Ensure all required tables exist
        $this->ensureWorkPerformedTableExists();
        $this->ensureCatalogTablesExist();
    }
    
    /**
     * Get the actual table name for a given ticket type
     */
    private function getCatalogTableName(string $type): string
    {
        return $this->catalogTableMapping[$type] ?? 'zadatak_troskovnik' . strtolower($type);
    }
    
    private function ensureWorkPerformedTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'zadatak_wp' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `zadatak_wp` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `zadatak_id` int(11) NOT NULL,
                    `item_id` int(11) NOT NULL,
                    `description` varchar(255) COLLATE utf8_general_ci NOT NULL,
                    `category` varchar(50) COLLATE utf8_general_ci NOT NULL,
                    `quantity` decimal(10,2) NOT NULL DEFAULT '1.00',
                    `unit` varchar(20) COLLATE utf8_general_ci DEFAULT NULL,
                    `price` decimal(10,2) NOT NULL DEFAULT '0.00',
                    `total` decimal(10,2) NOT NULL DEFAULT '0.00',
                    `type` varchar(10) COLLATE utf8_general_ci NOT NULL,
                    `date_added` datetime NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `zadatak_id` (`zadatak_id`),
                    CONSTRAINT `fk_zadatak_wp_zadatak` FOREIGN KEY (`zadatak_id`) 
                    REFERENCES `zadatak` (`id`) ON DELETE CASCADE
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                
                $this->dbConnection->execute($sql);
                
                // Use our own logger property instead of parent's private logger
                if ($this->wpLogger) {
                    $this->wpLogger->modules('Created zadatak_wp table');
                }
            }
        } catch (\PDOException $e) {
            // Use our own logger property instead of parent's private logger
            if ($this->wpLogger) {
                $this->wpLogger->error('Database error in WorkPerformedTable', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureWorkPerformedTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    private function ensureCatalogTablesExist(): void
    {
        try {
            // Use the mapping to get correct table names
            foreach ($this->catalogTableMapping as $type => $tableName) {
                // Check if table exists
                $tableExists = $this->dbConnection->querySingleValue(
                    "SELECT 1 FROM information_schema.tables WHERE table_name = '{$tableName}' AND table_schema = DATABASE()"
                );
                
                if ($tableExists === null) {
                    // Create table
                    $sql = "CREATE TABLE `{$tableName}` (
                        `id` int(11) NOT NULL AUTO_INCREMENT,
                        `description` varchar(255) COLLATE utf8_general_ci NOT NULL,
                        `category` varchar(50) COLLATE utf8_general_ci NOT NULL,
                        `unit` varchar(20) COLLATE utf8_general_ci NOT NULL,
                        `price` decimal(10,2) NOT NULL,
                        `code` varchar(50) COLLATE utf8_general_ci DEFAULT NULL,
                        PRIMARY KEY (`id`),
                        KEY `category` (`category`),
                        KEY `description` (`description`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_general_ci";
                    
                    $this->dbConnection->execute($sql);
                    
                    if ($this->wpLogger) {
                        $this->wpLogger->modules("Created {$tableName} table for {$type} catalog");
                    }
                }
            }
        } catch (\PDOException $e) {
            if ($this->wpLogger) {
                $this->wpLogger->error('Database error in WorkPerformedTable', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureCatalogTablesExist'
                ]);
            }
            throw $e;
        }
    }
    
    public function getWorkPerformedByZadatak(int $zadatakId): array
    {
        $query = "SELECT * FROM {$this->table} WHERE zadatak_id = :zadatak_id ORDER BY id ASC";
        $params = [':zadatak_id' => $zadatakId];
        return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
    }
    
    public function getWorkPerformedSummary(int $zadatakId): array
    {
        $query = "SELECT 
                    SUM(total) as total_amount,
                    COUNT(*) as items_count,
                    SUM(CASE WHEN category = 'labor' THEN 1 ELSE 0 END) as labor_count,
                    SUM(CASE WHEN category = 'material' THEN 1 ELSE 0 END) as material_count
                FROM {$this->table} 
                WHERE zadatak_id = :zadatak_id";
                
        $params = [':zadatak_id' => $zadatakId];
        return $this->dbConnection->queryAndFetchAssoc($query, $params) ?? [];
    }
    
    public function addWorkItem(array $data): int
    {
        // Calculate total if not provided
        if (!isset($data['total'])) {
            $data['total'] = $data['quantity'] * $data['price'];
        }
        
        // Set date added if not provided
        if (!isset($data['date_added'])) {
            $data['date_added'] = date('Y-m-d H:i:s');
        }
        
        // Use save method instead of insert directly
        $this->save($data);
        
        // Use DatabaseConnection's lastInsertId method instead of PDO
        return (int)$this->dbConnection->lastInsertId();
    }
    
    public function updateWorkItem(int $id, array $data): void
    {
        // Calculate total if not provided
        if (!isset($data['total']) && isset($data['quantity']) && isset($data['price'])) {
            $data['total'] = $data['quantity'] * $data['price'];
        }
        
        // Include primary key in the data array
        $data[$this->primaryKey] = $id;
        
        // Use save instead of update
        $this->save($data);
    }
    
    public function searchCatalogItems(string $type, string $query, string $category = ''): array
    {
        // Use the mapping to get the correct table name
        $tableName = $this->getCatalogTableName($type);
        $searchQuery = "%$query%";
        
        $sql = "SELECT * FROM {$tableName} WHERE description LIKE :query";
        $params = [':query' => $searchQuery];
        
        if (!empty($category)) {
            $sql .= " AND category = :category";
            $params[':category'] = $category;
        }
        
        $sql .= " LIMIT 15";
        
        return $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
    }
    
    public function getCatalogItem(string $type, int $itemId): ?array
    {
        // Use the mapping to get the correct table name
        $tableName = $this->getCatalogTableName($type);
        
        $sql = "SELECT * FROM {$tableName} WHERE id = :id";
        $params = [':id' => $itemId];
        
        $item = $this->dbConnection->queryAndFetchAssoc($sql, $params);
        return $item ?: null;
    }
}