/**
 * Twitter-style Comments System with Responsive Design
 */

// Hardcoded routes for comments
const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
const fetchCommentsRoute = `${publicUrl}/comments/api/fetch-comments`;
const addCommentRoute = `${publicUrl}/comments/api/add-comments`;
const fetchRepliesRoute = `${publicUrl}/comments/api/fetch-replies`;

document.addEventListener('DOMContentLoaded', function() {
    // Initialize comments
    initializeComments();
    
    // Load comments on page load
    if (document.querySelector('.task-header[data-task-id]')) {
        setTimeout(load_comment, 500);
    }

    // Setup event delegation for dynamically added elements
    setupEventDelegation();
});

function setupEventDelegation() {
    // Using event delegation for ALL dynamically added elements
    document.addEventListener('click', function(event) {
        // Handle reply count badge clicks
        if (event.target.closest('.reply-count')) {
            const badge = event.target.closest('.reply-count');
            const commentId = badge.closest('.comment').dataset.commentId;
            const container = badge.closest('.comments-container') || badge.closest('.overlay-body');
            
            // Find the specific reply container in this context
            toggleRepliesInContext(commentId, badge, container);
            
            // Prevent default browser actions
            event.preventDefault();
            return false;
        }
        
        // Handle reply buttons
        if (event.target.closest('.comment-reply-btn')) {
            const button = event.target.closest('.comment-reply-btn');
            const commentId = button.dataset.commentId;
            openCommentModal(commentId, true);
            
            event.preventDefault();
            return false;
        }
        
        // Handle comments floating button (mobile)
        if (event.target.closest('#comments_floating_btn')) {
            toggleCommentsOverlay(true);
            
            event.preventDefault();
            return false;
        }
        
        // Handle close overlay button
        if (event.target.closest('#close_comments_overlay')) {
            toggleCommentsOverlay(false);
            
            event.preventDefault();
            return false;
        }
        
        // Handle add comment buttons
        if (event.target.closest('.add-comment-btn')) {
            openCommentModal();
            
            event.preventDefault();
            return false;
        }
    });
}

// Initialize comments functionality
function initializeComments() {
    // Setup comment form submission
    const commentForm = document.getElementById('comment_form');
    if (commentForm) {
        $('#comment_form').off('submit').on('submit', function(event) {
            event.preventDefault();
            submitComment();
        });
    }
    
    // Fix modal backdrop issues
    $('#commentModal').on('hidden.bs.modal', function() {
        $('.modal-backdrop').remove();
        $('body').removeClass('modal-open').css('padding-right', '');
    });
    
    // Update comment count
    setInterval(updateCommentCount, 1000);
}

// Toggle comments overlay for mobile
function toggleCommentsOverlay(show) {
    const overlay = document.getElementById('comments_overlay');
    if (!overlay) return;
    
    if (show) {
        overlay.classList.add('active');
        document.body.classList.add('comments-open');
    } else {
        overlay.classList.remove('active');
        document.body.classList.remove('comments-open');
    }
}

// Open comment modal
function openCommentModal(commentId = 0, isReply = false) {
    // Reset form
    $('#comment_form')[0].reset();
    
    // Set title and comment ID
    document.getElementById('comment-title').textContent = isReply ? 'Reply to Comment' : 'Add Comment';
    document.getElementById('comment_id').value = commentId;
    
    // Clean up any existing backdrops
    $('.modal-backdrop').remove();
    $('body').removeClass('modal-open').css('padding-right', '');
    
    // Show modal safely
    setTimeout(() => {
        $('#commentModal').modal('show');
        
        // Focus textarea
        setTimeout(() => $('#comment_content').focus(), 300);
    }, 100);
    
    // Explicitly return false to prevent any form submissions
    return false;
}

// Update comment counter
function updateCommentCount() {
    // Count comments in any comments container
    const mainContainer = document.querySelector('.comments-card .comments-container');
    let count = 0;
    
    if (mainContainer) {
        count = mainContainer.querySelectorAll('.comment-thread').length;
    }
    
    // Update counter badge
    const counter = document.getElementById('comments_count');
    if (counter) {
        counter.textContent = count;
        counter.style.display = count > 0 ? 'flex' : 'none';
    }
}

// Submit a comment
function submitComment() {
    const form = document.getElementById('comment_form');
    
    if (!form) return;
    
    // Using jQuery AJAX for sending comments
    $.ajax({
        url: addCommentRoute,
        method: "POST",
        data: new FormData(form),
        processData: false,
        contentType: false,
        dataType: "JSON",
        success: function(data) {
            if (data.error) {
                document.getElementById('comment_message').innerHTML = data.error;
            }
            form.reset();
            $('#commentModal').modal('hide');
            
            // Reload comments
            load_comment();
            
            setTimeout(() => {
                document.getElementById('comment_message').innerHTML = '';
            }, 3000);
        },
        error: function(xhr, status, error) {
            console.error('Error submitting comment:', error);
            $('#commentModal').modal('hide');
        }
    });
}

// Toggle replies visibility in a specific context
function toggleRepliesInContext(commentId, element, container) {
    if (!container) return;
    
    // Find the reply container within this specific container context
    const repliesContainer = container.querySelector(`#replies-${commentId}`);
    
    if (!repliesContainer) return;
    
    // Check if replies are already loaded
    if (repliesContainer.innerHTML.trim() === '') {
        // Show loading state
        element.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        element.classList.add('loading');
        
        // Fetch replies
        $.ajax({
            url: fetchRepliesRoute,
            method: "POST",
            data: { "commentId": commentId },
            dataType: "JSON",
            success: function(response) {
                if (response.error) {
                    console.error(response.error);
                    element.innerHTML = `<i class="fas fa-comment-dots"></i> ${element.dataset.count}`;
                    element.classList.remove('loading');
                    return;
                }
                
                // Add replies to container
                repliesContainer.innerHTML = response.replies;
                
                // Show replies
                repliesContainer.style.display = 'block';
                
                // Update badge state
                element.innerHTML = `<i class="fas fa-comment-dots"></i> ${element.dataset.count}`;
                element.classList.remove('loading');
                element.classList.add('active');
                
                // Also update the same reply container in other views
                updateRepliesInOtherViews(commentId, response.replies);
            },
            error: function(xhr, status, error) {
                console.error('Error fetching replies:', error);
                element.innerHTML = `<i class="fas fa-comment-dots"></i> ${element.dataset.count}`;
                element.classList.remove('loading');
            }
        });
    } else {
        // Toggle visibility
        const isVisible = repliesContainer.style.display !== 'none';
        
        repliesContainer.style.display = isVisible ? 'none' : 'block';
        
        // Update badge state
        element.classList.toggle('active');
    }
}

// Update replies in other views (for sync between mobile/desktop)
function updateRepliesInOtherViews(commentId, replies) {
    // Find all reply containers with this ID
    const allReplyContainers = document.querySelectorAll(`#replies-${commentId}`);
    
    allReplyContainers.forEach(container => {
        if (container.innerHTML.trim() === '') {
            container.innerHTML = replies;
        }
    });
    
    // Update all badges
    const allBadges = document.querySelectorAll(`.comment[data-comment-id="${commentId}"] .reply-count`);
    allBadges.forEach(badge => {
        badge.classList.add('active');
    });
}

// Load comments via AJAX
function load_comment() {
    try {
        const taskHeader = document.querySelector('.task-header');
        if (!taskHeader) return;
        
        const taskId = taskHeader.getAttribute('data-task-id');
        if (!taskId) return;
  
        $.ajax({
            url: fetchCommentsRoute,
            method: "POST",
            data: {"myData": taskId},
            success: function(data) {
                // Update all comments containers
                const commentsContainers = document.querySelectorAll('.comments-container');
                commentsContainers.forEach(container => {
                    container.innerHTML = data;
                });
                
                // Update comment count
                updateCommentCount();
            },
            error: function(xhr, status, error) {
                console.error('Error loading comments:', error);
            }
        });
    } catch (error) {
        console.error("Error loading comments:", error);
    }
}

// Handle theme changes
document.addEventListener('themechange', function(e) {
    const root = document.documentElement;
    const theme = e.detail.theme;
    
    if (theme === 'dark') {
        root.style.setProperty('--accent-primary-rgb', '255, 107, 0');
        root.style.setProperty('--accent-secondary-rgb', '138, 43, 226');
    } else {
        root.style.setProperty('--accent-primary-rgb', '52, 152, 219');
        root.style.setProperty('--accent-secondary-rgb', '46, 204, 113');
    }
});