/**
 * Real Data Tasks Admin Dashboard Charts
 * Handles chart creation functionality with real database data
 */
document.addEventListener('DOMContentLoaded', function() {
  // Get data elements
  const chartsData = document.getElementById('task-charts-data');
  
  if (!chartsData) {
    console.error('Tasks charts data element not found!');
    return;
  }
  
  try {
    // Parse real data from the backend
    const statusData = JSON.parse(chartsData.getAttribute('data-status') || '[]');
    const typeData = JSON.parse(chartsData.getAttribute('data-type') || '[]');
    const monthlyData = JSON.parse(chartsData.getAttribute('data-monthly') || '[]');
    
    // Create charts with real data
    createTasksByStatusChart(statusData);
    createTasksByTypeChart(typeData);
    createMonthlyTasksChart(monthlyData);
    
  } catch (error) {
    console.error('Error initializing tasks admin charts:', error);
  }
});

/**
 * Create the tasks by status chart
 * @param {Array} statusData - Real status data from database
 */
function createTasksByStatusChart(statusData) {
  if (!document.getElementById('tasksByStatusChart')) {
    console.error('Tasks by status chart element not found');
    return;
  }
  
  if (!statusData || statusData.length === 0) {
    console.warn('No status data available');
    return;
  }
  
  const statusColors = {
    'Realizirano': '#4caf50',
    'Preuzeto': '#2196f3',
    'Novo': '#ff9800',
    'Stornirano': '#f44336',
    'Vratiti na doradu': '#9c27b0'
  };
  
  const labels = statusData.map(item => item.status || 'Nepoznato');
  const values = statusData.map(item => parseInt(item.count) || 0);
  const colors = labels.map(label => statusColors[label] || '#757575');
  
  const ctx = document.getElementById('tasksByStatusChart').getContext('2d');
  new Chart(ctx, {
    type: 'doughnut',
    data: {
      labels: labels,
      datasets: [{
        data: values,
        backgroundColor: colors,
        borderColor: '#ffffff',
        borderWidth: 2,
        hoverBorderWidth: 3
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '50%',
      plugins: {
        legend: {
          position: 'right',
          labels: {
            boxWidth: 12,
            font: {
              size: 11
            },
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
                  return {
                    text: `${label} (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
              return `${context.label}: ${value} zadataka (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the tasks by type chart
 * @param {Array} typeData - Real type data from database
 */
function createTasksByTypeChart(typeData) {
  if (!document.getElementById('tasksByTypeChart')) {
    console.error('Tasks by type chart element not found');
    return;
  }
  
  if (!typeData || typeData.length === 0) {
    console.warn('No type data available');
    return;
  }
  
  const typeColors = {
    'TMHR': '#ff5722',
    'A1': '#3f51b5',
    'HT': '#009688',
    'Other': '#795548'
  };
  
  const labels = typeData.map(item => item.type || 'Nepoznato');
  const values = typeData.map(item => parseInt(item.count) || 0);
  const colors = labels.map(label => typeColors[label] || '#757575');
  
  const ctx = document.getElementById('tasksByTypeChart').getContext('2d');
  new Chart(ctx, {
    type: 'doughnut',
    data: {
      labels: labels,
      datasets: [{
        data: values,
        backgroundColor: colors,
        borderColor: '#ffffff',
        borderWidth: 2,
        hoverBorderWidth: 3
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '50%',
      plugins: {
        legend: {
          position: 'right',
          labels: {
            boxWidth: 12,
            font: {
              size: 11
            },
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
                  return {
                    text: `${label} (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
              return `${context.label}: ${value} zadataka (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the monthly tasks chart
 * @param {Array} monthlyData - Real monthly data from database
 */
function createMonthlyTasksChart(monthlyData) {
  if (!document.getElementById('monthlyTasksChart')) {
    console.error('Monthly tasks chart element not found');
    return;
  }
  
  if (!monthlyData || monthlyData.length === 0) {
    console.warn('No monthly data available');
    return;
  }
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  
  // Create array for all months with 0 counts
  let monthlyChartData = Array(12).fill(0);
  
  // Fill in actual counts
  monthlyData.forEach(item => {
    if (item.month > 0 && item.month <= 12) {
      monthlyChartData[item.month - 1] = parseInt(item.count) || 0;
    }
  });
  
  const ctx = document.getElementById('monthlyTasksChart').getContext('2d');
  
  // Create gradient
  const gradient = ctx.createLinearGradient(0, 0, 0, 400);
  gradient.addColorStop(0, 'rgba(54, 162, 235, 0.8)');
  gradient.addColorStop(1, 'rgba(54, 162, 235, 0.2)');
  
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: monthNames,
      datasets: [{
        label: 'Zadaci',
        data: monthlyChartData,
        backgroundColor: gradient,
        borderColor: '#36a2eb',
        borderWidth: 2,
        borderRadius: 8,
        borderSkipped: false,
        hoverBackgroundColor: 'rgba(54, 162, 235, 0.9)',
        hoverBorderColor: '#2196f3',
        hoverBorderWidth: 3
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      scales: {
        x: {
          grid: {
            display: false
          },
          ticks: {
            font: {
              size: 11
            }
          }
        },
        y: {
          beginAtZero: true,
          ticks: {
            precision: 0,
            font: {
              size: 11
            }
          }
        }
      },
      plugins: {
        legend: {
          display: false
        },
        tooltip: {
          callbacks: {
            title: function(context) {
              return monthNames[context[0].dataIndex];
            },
            label: function(context) {
              return `Zadaci: ${context.parsed.y}`;
            }
          }
        }
      },
      animation: {
        duration: 2000,
        easing: 'easeInOutQuart',
        delay: (context) => context.dataIndex * 100
      }
    }
  });
}