/**
 * Work Performed JavaScript
 * Handles search, add, edit, and delete functionality for work items
 */
$(function() {
    // API Routes
    const ROUTES = {
        SEARCH: '/zaobracun/api/search',
        ADD_ITEM: '/zaobracun/api/add-item',
        UPDATE_ITEM: '/zaobracun/api/update-item',
        DELETE_ITEM: '/zaobracun/api/delete-item'
    };

    // Get base path from app config element
    const appConfig = document.getElementById('app-config');
    const basePath = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';
    
    // Cache DOM elements
    const $searchQuery = $('#search_query');
    const $category = $('#category');
    const $searchResults = $('#search_results');
    const $selectedItemContainer = $('#selected_item_container');
    const $zadatakId = $('#zadatak_id');
    const $zadatakType = $('#zadatak_type');
    const $clearSearch = $('#clear_search');
    const $quantity = $('#quantity');
    const $addItemBtn = $('#add_item_btn');
    const $workItemsContainer = $('#work_items_container');
    const $totalAmount = $('#total_amount');
    const $itemsCount = $('#items_count');
    
    // Selected item data
    let selectedItem = null;

    // Initialize
    init();

    function init() {
        // Initialize event listeners
        initEventListeners();

        // Initialize existing work items event listeners
        initWorkItemsListeners();
    }

    function initEventListeners() {
        // Search input event
        $searchQuery.on('keyup', debounce(handleSearch, 300));
        
        // Category change event
        $category.on('change', function() {
            if ($searchQuery.val().length >= 3) {
                handleSearch();
            }
        });

        // Clear search button
        $clearSearch.on('click', clearSearch);

        // Quantity buttons
        $('#increase_quantity').on('click', function() {
            updateQuantity(parseFloat($quantity.val()) + 0.1);
        });

        $('#decrease_quantity').on('click', function() {
            updateQuantity(parseFloat($quantity.val()) - 0.1);
        });

        // Add item button
        $addItemBtn.on('click', addWorkItem);

        // Edit quantity modal events
        $('#increase_edit_quantity').on('click', function() {
            const $editQuantity = $('#edit_quantity');
            $editQuantity.val((parseFloat($editQuantity.val()) + 0.1).toFixed(1));
        });

        $('#decrease_edit_quantity').on('click', function() {
            const $editQuantity = $('#edit_quantity');
            $editQuantity.val(Math.max(0.1, (parseFloat($editQuantity.val()) - 0.1)).toFixed(1));
        });

        // Save edit button
        $('#save_edit_btn').on('click', saveEditItem);

        // Delete confirmation button
        $('#confirm_delete_btn').on('click', confirmDeleteItem);
    }

    function initWorkItemsListeners() {
        // Edit item buttons
        $('.edit-item-btn').off('click').on('click', function() {
            const $workItem = $(this).closest('.work-item');
            const itemId = $workItem.data('id');
            openEditModal(itemId, $workItem);
        });

        // Delete item buttons
        $('.delete-item-btn').off('click').on('click', function() {
            const $workItem = $(this).closest('.work-item');
            const itemId = $workItem.data('id');
            openDeleteModal(itemId, $workItem);
        });
    }

    function handleSearch() {
        const query = $searchQuery.val().trim();
        const category = $category.val();
        
        if (query.length < 3) {
            $searchResults.empty();
            return;
        }

        // Show loading indicator
        $searchResults.html('<div class="text-center p-3"><i class="fas fa-spinner fa-spin"></i> Searching...</div>');

        // Get task type
        const type = $zadatakType.val();

        // Make AJAX request
        $.ajax({
            url: basePath + ROUTES.SEARCH,
            method: 'POST',
            data: {
                query: query,
                category: category,
                type: type
            },
            success: function(response) {
                displaySearchResults(response.results);
            },
            error: function() {
                $searchResults.html('<div class="alert alert-danger">Search failed. Please try again.</div>');
            }
        });
    }

    function displaySearchResults(results) {
        if (results.length === 0) {
            $searchResults.html('<div class="text-center p-3">No results found.</div>');
            return;
        }

        let html = '';
        results.forEach(function(item) {
            html += `
                <div class="search-result-item" data-id="${item.id}" data-item='${JSON.stringify(item)}'>
                    <h4>${item.description}</h4>
                    <div class="item-meta">
                        <span class="badge badge-${item.category}">${item.category}</span>
                        <span>${formatCurrency(item.price)} €</span>
                        <span>${item.unit}</span>
                    </div>
                </div>
            `;
        });

        $searchResults.html(html);

        // Add click event to search results
        $('.search-result-item').on('click', function() {
            const item = $(this).data('item');
            selectItem(item);
        });
    }

    function selectItem(item) {
        selectedItem = item;
        
        // Update selected item display
        $('#selected_description').text(item.description);
        $('#selected_category').text(item.category);
        $('#selected_category').removeClass('badge-labor badge-material').addClass(`badge-${item.category}`);
        $('#selected_price').text(`${formatCurrency(item.price)} € per ${item.unit}`);
        $('#unit_label').text(item.unit);
        
        // Reset quantity
        $quantity.val(1);
        
        // Show selected item container
        $selectedItemContainer.removeClass('d-none');
        
        // Hide search results
        $searchResults.empty();
        
        // Clear search input
        $searchQuery.val('');
    }

    function clearSearch() {
        $searchQuery.val('');
        $searchResults.empty();
        $selectedItemContainer.addClass('d-none');
        selectedItem = null;
    }

    function updateQuantity(value) {
        value = Math.max(0.1, value);
        $quantity.val(value.toFixed(1));
    }

    function addWorkItem() {
        if (!selectedItem) return;

        const zadatakId = $zadatakId.val();
        const type = $zadatakType.val();
        const quantity = parseFloat($quantity.val());

        // Validate quantity
        if (isNaN(quantity) || quantity <= 0) {
            alert('Please enter a valid quantity.');
            return;
        }

        // Make AJAX request
        $.ajax({
            url: basePath + ROUTES.ADD_ITEM,
            method: 'POST',
            data: {
                zadatak_id: zadatakId,
                item_id: selectedItem.id,
                type: type,
                quantity: quantity
            },
            success: function(response) {
                if (response.success) {
                    // Add the item to the list
                    addItemToList(response.item);
                    
                    // Update summary
                    updateSummary(response.summary);
                    
                    // Clear selection
                    clearSearch();
                }
            },
            error: function() {
                alert('Failed to add item. Please try again.');
            }
        });
    }

    function addItemToList(item) {
        // Check if there are no items yet
        if ($workItemsContainer.find('.no-items').length) {
            $workItemsContainer.html('<div class="work-items-list"></div>');
        }

        const $workItemsList = $workItemsContainer.find('.work-items-list');
        
        const html = `
            <div class="work-item" data-id="${item.id}">
                <div class="item-content">
                    <div class="item-header">
                        <h4>${item.description}</h4>
                        <div class="item-actions">
                            <button type="button" class="btn btn-sm btn-outline-primary edit-item-btn">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-danger delete-item-btn">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                    <div class="item-details">
                        <span class="badge badge-${item.category}">
                            ${capitalizeFirstLetter(item.category)}
                        </span>
                        <span>${formatCurrency(item.quantity)} ${item.unit}</span>
                        <span>${formatCurrency(item.price)} €</span>
                        <span class="item-total">${formatCurrency(item.total)} €</span>
                    </div>
                </div>
            </div>
        `;
        
        $workItemsList.prepend(html);
        
        // Reinitialize event listeners
        initWorkItemsListeners();
    }

    function openEditModal(itemId, $workItem) {
        const description = $workItem.find('h4').text();
        const unit = $workItem.find('.item-details span:nth-child(2)').text().split(' ').pop();
        const quantity = parseFloat($workItem.find('.item-details span:nth-child(2)').text());

        $('#edit_item_id').val(itemId);
        $('#edit_item_description').text(description);
        $('#edit_quantity').val(quantity.toFixed(1));
        $('#edit_unit_label').text(unit);

        $('#editItemModal').modal('show');
    }

    function saveEditItem() {
        const itemId = $('#edit_item_id').val();
        const quantity = parseFloat($('#edit_quantity').val());
        const zadatakId = $zadatakId.val();

        // Validate quantity
        if (isNaN(quantity) || quantity <= 0) {
            alert('Please enter a valid quantity.');
            return;
        }

        // Make AJAX request
        $.ajax({
            url: basePath + ROUTES.UPDATE_ITEM,
            method: 'POST',
            data: {
                id: itemId,
                zadatak_id: zadatakId,
                quantity: quantity
            },
            success: function(response) {
                if (response.success) {
                    // Update the item in the list
                    updateItemInList(response.item);
                    
                    // Update summary
                    updateSummary(response.summary);
                    
                    // Close modal
                    $('#editItemModal').modal('hide');
                }
            },
            error: function() {
                alert('Failed to update item. Please try again.');
            }
        });
    }

    function updateItemInList(item) {
        const $workItem = $(`.work-item[data-id="${item.id}"]`);
        
        // Update quantity and total
        const $itemDetails = $workItem.find('.item-details');
        $itemDetails.find('span:nth-child(2)').text(`${formatCurrency(item.quantity)} ${item.unit}`);
        $itemDetails.find('.item-total').text(`${formatCurrency(item.total)} €`);
    }

    function openDeleteModal(itemId, $workItem) {
        const description = $workItem.find('h4').text();

        $('#delete_item_id').val(itemId);
        $('#delete_item_description').text(description);

        $('#deleteItemModal').modal('show');
    }

    function confirmDeleteItem() {
        const itemId = $('#delete_item_id').val();
        const zadatakId = $zadatakId.val();

        // Make AJAX request
        $.ajax({
            url: basePath + ROUTES.DELETE_ITEM,
            method: 'POST',
            data: {
                id: itemId,
                zadatak_id: zadatakId
            },
            success: function(response) {
                if (response.success) {
                    // Remove the item from the list
                    $(`.work-item[data-id="${itemId}"]`).remove();
                    
                    // Check if there are no items left
                    if ($('.work-item').length === 0) {
                        $workItemsContainer.html(`
                            <div class="no-items">
                                <i class="fas fa-clipboard-list"></i>
                                <p>No work items added yet. Use the search above to add work items.</p>
                            </div>
                        `);
                    }
                    
                    // Update summary
                    updateSummary(response.summary);
                    
                    // Close modal
                    $('#deleteItemModal').modal('hide');
                }
            },
            error: function() {
                alert('Failed to delete item. Please try again.');
            }
        });
    }

    function updateSummary(summary) {
        $totalAmount.text(`${formatCurrency(summary.total_amount)} €`);
        $itemsCount.text(summary.items_count);
    }

    // Helper functions
    function formatCurrency(value) {
        return parseFloat(value).toLocaleString('hr-HR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    function capitalizeFirstLetter(string) {
        return string.charAt(0).toUpperCase() + string.slice(1);
    }

    function debounce(func, wait) {
        let timeout;
        return function() {
            const context = this, args = arguments;
            clearTimeout(timeout);
            timeout = setTimeout(function() {
                func.apply(context, args);
            }, wait);
        };
    }
});