<?php

namespace travelorders\Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Controllers\BaseController;
use Models\DatabaseTable;
use travelorders\Models\DatabaseTableTravelOrders;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Validation\Validator;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;
use DateTime;
use DateTimeZone;

class TravelOrders extends BaseController 
{
    private DatabaseTable $korisniciTable;
    private DatabaseTableTravelOrders $travelOrdersTable;
    private Validator $validator;
    private const MAX_EXPORT_ROWS = 5000;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseTableTravelOrders $travelOrdersTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->travelOrdersTable = $travelOrdersTable;
        $this->validator = $validator;
    }

    public function edit(Request $request): Response 
    {
        try {
            $travelOrder = null;
            $editToken = null;

            if ($id = $request->getParam('id')) {
                $travelOrder = $this->travelOrdersTable->findById($id);
                if ($travelOrder) {
                    $editToken = $this->createEditToken($travelOrder->id, 'travel_order');
                }
            }

            return $this->response()->render(
                'travelorders/resources/views/edit.html.php',
                [
                    'travelOrder' => $travelOrder,
                    'edit_token' => $editToken
                ],
                'Edit travel order'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading travel order edit form', [
                'message' => $e->getMessage(),
                'id' => $id ?? null
            ]);
            return $this->response()->renderError('Failed to load travel order edit form');
        }
    }

    public function saveEdit(Request $request): Response 
    {
        try {
            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $travelOrder = $request->getPost('travel_order');
            $travelOrderId = null;
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'travel_order');
                if (isset($tokenResult['error'])) {
                    return $this->renderPutniNalogForm(
                        $travelOrder, 
                        $editToken, 
                        $tokenResult
                    );
                }
                // Token je valjan
                $travelOrderId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
                $travelOrder['id'] = $travelOrderId;
            }
            
            // 3. VALIDACIJA PODATAKA FORME
            $rules = [
                'travel_order[datum]' => ['required', 'date'],
                'travel_order[vrijeme_polaska]' => ['required', 'time_24h'],
                'travel_order[vrijeme_dolaska]' => ['required', 'time_24h'],
                'travel_order[vozac]' => ['required', 'alpha_spaces', ['max', 100]],
                'travel_order[registracija]' => ['required', ['max', 20]],
                'travel_order[vozilo]' => ['required', ['max', 100]],
                'travel_order[broj_sasije]' => ['required', ['max', 50]],
                'travel_order[pravac_kretanja]' => ['required', ['max', 200]],
                'travel_order[pocetno_stanje_brojila]' => ['required', 'numeric'],
                'travel_order[zavrsno_stanje_brojila]' => ['required', 'numeric'],
                'travel_order[prijedeni_km]' => ['required', 'numeric'],
                // Uklonjen 'required' za gorivo polja
                'travel_order[gorivo_cijena]' => ['nullable', 'currency'],
                'travel_order[gorivo_kolicina]' => ['nullable', 'currency'],
                'travel_order[gorivo_uk_cijena]' => ['nullable', 'currency']
            ];

            $validation = $request->validate($rules);
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($travelOrderId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'travel_order');
                }
                
                return $this->renderPutniNalogForm(
                    $travelOrder, 
                    $newEditToken, 
                    $validation->errors()
                );
            }

            // 4. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }
            
            // 5. PROCESIRANJE PODATAKA
            // Process date fields
            $this->processDateFields($travelOrder);
            
            // Convert form values to correct format
            if (!empty($travelOrder['gorivo_cijena']) && trim($travelOrder['gorivo_cijena']) !== '') {
                $travelOrder['gorivo_cijena'] = str_replace(',', '.', $travelOrder['gorivo_cijena']);
            } else {
                $travelOrder['gorivo_cijena'] = null;
            }

            if (!empty($travelOrder['gorivo_kolicina']) && trim($travelOrder['gorivo_kolicina']) !== '') {
                $travelOrder['gorivo_kolicina'] = str_replace(',', '.', $travelOrder['gorivo_kolicina']);
            } else {
                $travelOrder['gorivo_kolicina'] = null;
            }

            if (!empty($travelOrder['gorivo_uk_cijena']) && trim($travelOrder['gorivo_uk_cijena']) !== '') {
                $travelOrder['gorivo_uk_cijena'] = str_replace(',', '.', $travelOrder['gorivo_uk_cijena']);
            } else {
                $travelOrder['gorivo_uk_cijena'] = null;
            }

            // Calculate fields if needed
            if (empty($travelOrder['prijedeni_km']) && !empty($travelOrder['pocetno_stanje_brojila']) && !empty($travelOrder['zavrsno_stanje_brojila'])) {
                $travelOrder['prijedeni_km'] = $travelOrder['zavrsno_stanje_brojila'] - $travelOrder['pocetno_stanje_brojila'];
            }

            // Ažurirana kalkulacija ukupne cijene goriva
            if (!empty($travelOrder['gorivo_cijena']) && !empty($travelOrder['gorivo_kolicina']) && 
                $travelOrder['gorivo_cijena'] !== null && $travelOrder['gorivo_kolicina'] !== null &&
                (empty($travelOrder['gorivo_uk_cijena']) || $travelOrder['gorivo_uk_cijena'] === null)) {
                $travelOrder['gorivo_uk_cijena'] = $travelOrder['gorivo_cijena'] * $travelOrder['gorivo_kolicina'];
            }
            
            // 6. SPREMANJE PODATAKA
            $this->travelOrdersTable->save($travelOrder);
            
            // 7. PREUSMJERAVANJE
            return Response::redirect('travelorders/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving travel order', [
                'message' => $e->getMessage(),
                'travel_order' => $travelOrder ?? null
            ]);
            return $this->response()->renderError('Failed to save travel order');
        }
    }

    public function list(Request $request): Response 
    {
        try {
            $travelOrders = $this->travelOrdersTable->findAll();
            $this->sessionManager->set('valid_travelorder_ids', array_map(fn($nalog) => $nalog->id, $travelOrders));

            return $this->response()->render(
                'travelorders/resources/views/list.html.php',
                [
                    'user' => $this->authentication->getUser(),
                    'travelOrders' => $travelOrders
                ],
                'All travel orders'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading travel orders list', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load travel orders list');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $travelOrderId = $request->getPost('id');
            
            // Sigurnosna provjera da li je ID na listi valjanih ID-ova
            $validIds = $this->sessionManager->get('valid_travelorder_ids', []);
            if (!in_array($travelOrderId, $validIds ?? [])) {
                $this->logger->security('Unauthorized travel order deletion attempt', [
                    'id' => $travelOrderId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            // Brisanje putnog naloga
            $this->travelOrdersTable->delete($travelOrderId);
            $this->removeFromSession($travelOrderId);

            return Response::redirect('travelorders/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting travel order', [
                'message' => $e->getMessage(),
                'id' => $travelOrderId ?? null
            ]);
            return $this->response()->renderError('Failed to delete travel order');
        }
    }

    /**
     * Export travel orders to Excel/CSV
     */
    public function export(Request $request): Response 
    {
        try {
            // Check if filtered data is provided
            $filteredData = $request->getPost('filtered_data');
            
            if ($filteredData && is_array($filteredData)) {
                // Use filtered data from frontend
                $travelOrders = $this->processFilteredData($filteredData);
                $filename = "filtered travel orders";
            } else {
                // Export all data
                $travelOrders = $this->travelOrdersTable->findAll();
                $filename = "svi travel orders";
            }
            
            $columns = [
                'id', 'datum', 'vrijeme_polaska', 'vrijeme_dolaska', 
                'vozac', 'vozilo', 'registracija', 'broj_sasije',
                'pravac_kretanja', 'pocetno_stanje_brojila', 'zavrsno_stanje_brojila',
                'prijedeni_km', 'gorivo_cijena', 'gorivo_kolicina', 'gorivo_uk_cijena'
            ];
            
            $exportData = $this->exportTravelOrders($travelOrders, $columns, $filename);
            
            return Response::download(
                $exportData['content'],
                $exportData['filename'],
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id', []),
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Failed to export travel orders');
        }
    }

    /**
     * Process filtered data from frontend into objects
     */
    private function processFilteredData(array $filteredData): array 
    {
        $processedData = [];
        
        foreach ($filteredData as $index => $rowData) {
            $obj = new \stdClass();
            
            // Map the filtered data to object properties
            $obj->id = $index + 1; // Sequential ID for filtered data
            $obj->datum = $rowData['datum'] ?? '';
            $obj->vrijeme_polaska = $rowData['vrijeme_polaska'] ?? '';
            $obj->vrijeme_dolaska = $rowData['vrijeme_dolaska'] ?? '';
            $obj->vozac = $rowData['vozac'] ?? '';
            $obj->vozilo = $rowData['vozilo'] ?? '';
            $obj->registracija = $rowData['registracija'] ?? '';
            $obj->broj_sasije = ''; // Not available in filtered data
            $obj->pravac_kretanja = $rowData['pravac_kretanja'] ?? '';
            $obj->pocetno_stanje_brojila = ''; // Not available in filtered data
            $obj->zavrsno_stanje_brojila = ''; // Not available in filtered data
            $obj->prijedeni_km = $rowData['prijedeni_km'] ?? '';
            $obj->gorivo_cijena = ''; // Not available in filtered data
            $obj->gorivo_kolicina = ''; // Not available in filtered data
            $obj->gorivo_uk_cijena = $rowData['gorivo_uk_cijena'] ?? '';
            
            $processedData[] = $obj;
        }
        
        return $processedData;
    }

    /**
     * Export putni nalozi to Excel/CSV format
     */
    private function exportTravelOrders(array $travelOrders, array $columns, string $filename): array 
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || $currentUser->user_permissions < Korisnik::KORISNIK_tasks_tehnicar) {
                throw new \RuntimeException('Unauthorized to export travel orders');
            }
    
            if (count($travelOrders) > self::MAX_EXPORT_ROWS) {
                throw new \RuntimeException(
                    sprintf('Export limit exceeded: %d rows', self::MAX_EXPORT_ROWS)
                );
            }
    
            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "%s - %s tjedan %s",
                $filename,
                date('d-m-Y'),
                date("W", strtotime(date('d-m-Y')))
            );
    
            return excelexport('travel_orders', $columns, $savefileas, $travelOrders);
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'rows_count' => count($travelOrders)
            ]);
            throw $e;
        }
    }

    /**
     * Remove putni nalog ID from session
     */
    private function removeFromSession(int $travelOrderId): void 
    {
        $validIds = $this->sessionManager->get('valid_travelorder_ids', []);
        $validIds = array_filter($validIds, fn($id) => $id != $travelOrderId);
        $this->sessionManager->set('valid_travelorder_ids', $validIds);
    }
    
    /**
     * Process date fields to convert them to English format
     * 
     * @param array &$travelOrder Reference to the putni nalog data array
     * @return void
     */
    private function processDateFields(array &$travelOrder): void 
    {
        require_once APP_ROOT . '/moduli/travelorders/Helpers/convertDateToEnglishFormat.php';
        
        if (isset($travelOrder['datum'])) {
            $travelOrder['datum'] = convertDateToEnglishFormat($travelOrder['datum']);
        }
        
        // For time fields, we need to ensure they're in the correct format
        // but we don't convert them with the date converter function
        if (isset($travelOrder['vrijeme_polaska']) && !empty($travelOrder['vrijeme_polaska'])) {
            // Ensure time is in HH:MM format
            $time = strtotime($travelOrder['vrijeme_polaska']);
            if ($time !== false) {
                $travelOrder['vrijeme_polaska'] = date('H:i', $time);
            }
        }
        
        if (isset($travelOrder['vrijeme_dolaska']) && !empty($travelOrder['vrijeme_dolaska'])) {
            // Ensure time is in HH:MM format
            $time = strtotime($travelOrder['vrijeme_dolaska']);
            if ($time !== false) {
                $travelOrder['vrijeme_dolaska'] = date('H:i', $time);
            }
        }
    }
    
    /**
     * Helper method to extract ID from edit token
     * 
     * @param string $token The edit token
     * @return int|null The ID from the token or null if invalid
     */
    private function getIdFromEditToken(string $token): ?int
    {
        try {
            $parts = explode(':', $token);
            if (count($parts) >= 2) {
                return (int)$parts[0];
            }
        } catch (\Exception $e) {
            // Token parsing failed
        }
        return null;
    }
    
    /**
     * Helper method to render the putni nalog form with errors
     */
    private function renderPutniNalogForm(array $travelOrder, ?string $editToken, array $errors): Response
    {
        return $this->response()->render(
            'travelorders/resources/views/edit.html.php',
            [
                'travelOrder' => (object)$travelOrder,
                'edit_token' => $editToken,
                'errors' => $errors,
                'user' => $this->authentication->getUser()
            ],
            'Edit travel order'
        );
    }
}