<?php

namespace travelorders\Models;

use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatabaseTableTravelOrders extends DatabaseTable
{
    /**
     * @var \baseKRIZAN\Database\DatabaseConnection
     */
    protected $dbConnection;
    
    /**
     * @var Logger|null
     */
    private ?Logger $logger;
    
    /**
     * Constructor with logger support
     * 
     * @param DatabaseConnection $dbConnection Database connection
     * @param string $table Table name
     * @param string $primaryKey Primary key
     * @param string $className Class name
     * @param array $constructorArgs Constructor arguments
     * @param Logger|null $logger Logger instance
     */
    public function __construct(
        DatabaseConnection $dbConnection, 
        string $table, 
        string $primaryKey, 
        string $className = 'stdClass', 
        array $constructorArgs = [],
        ?Logger $logger = null
    ) {
        parent::__construct($dbConnection, $table, $primaryKey, $className, $constructorArgs);
        $this->dbConnection = $dbConnection;
        $this->logger = $logger;
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        if ($this->logger) {
            $this->logger->modules('DatabaseTableTravelOrders initialized', [
                'table' => $table
            ]);
        }
    }
    
    /**
     * Ensure all required tables exist
     * 
     * @return void
     */
    private function ensureTablesExist(): void
    {
        $this->ensureTravelOrdersTableExists();
    }
    
    /**
     * Ensure putni_nalozi table exists
     * 
     * @return void
     */
    private function ensureTravelOrdersTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = 'putni_nalozi' AND table_schema = DATABASE()"
            );
            
            if ($tableExists === null) {
                // Create table
                $sql = "CREATE TABLE `putni_nalozi` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `datum` date DEFAULT NULL,
                    `vrijeme_polaska` time DEFAULT NULL,
                    `vrijeme_dolaska` time DEFAULT NULL,
                    `vozac` varchar(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
                    `registracija` varchar(20) COLLATE utf8mb4_general_ci DEFAULT NULL,
                    `vozilo` varchar(100) COLLATE utf8mb4_general_ci DEFAULT NULL,
                    `broj_sasije` varchar(50) COLLATE utf8mb4_general_ci DEFAULT NULL,
                    `pravac_kretanja` varchar(200) COLLATE utf8mb4_general_ci DEFAULT NULL,
                    `pocetno_stanje_brojila` int(11) DEFAULT NULL,
                    `zavrsno_stanje_brojila` int(11) DEFAULT NULL,
                    `prijedeni_km` int(11) DEFAULT NULL,
                    `gorivo_cijena` decimal(10,2) DEFAULT NULL,
                    `gorivo_kolicina` decimal(10,2) DEFAULT NULL,
                    `gorivo_uk_cijena` decimal(10,2) DEFAULT NULL,
                    `napomena` text COLLATE utf8mb4_general_ci DEFAULT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
                
                $this->dbConnection->execute($sql);
                
                if ($this->logger) {
                    $this->logger->modules('Created putni_nalozi table');
                }
            }
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in DatabaseTableTravelOrders', [
                    'error' => $e->getMessage(),
                    'function' => 'ensureTravelOrdersTableExists'
                ]);
            }
            throw $e;
        }
    }
    
    // Helper methods for common calculations
    public function formatDate($date): string
    {
        return date('d.m.Y', strtotime($date));
    }
    
    public function formatTime($time): string
    {
        return date('H:i', strtotime($time));
    }
    
    public function formatNumber($number, $decimals = 2): string
    {
        return number_format((float)$number, $decimals, ',', '.');
    }
    
    // Specific methods for putni_nalozi table
    public function findByVozac(string $vozac): array
    {
        $query = "SELECT * FROM {$this->table} WHERE vozac = :vozac ORDER BY datum DESC";
        return $this->dbConnection->queryAndFetchAllObjects($query, ['vozac' => $vozac]);
    }

    public function findByVozilo(string $vozilo): array
    {
        $query = "SELECT * FROM {$this->table} WHERE vozilo = :vozilo ORDER BY datum DESC";
        return $this->dbConnection->queryAndFetchAllObjects($query, ['vozilo' => $vozilo]);
    }

    public function findByRegistracija(string $registracija): array
    {
        $query = "SELECT * FROM {$this->table} WHERE registracija = :registracija ORDER BY datum DESC";
        return $this->dbConnection->queryAndFetchAllObjects($query, ['registracija' => $registracija]);
    }
    
    public function findByDateRange(string $startDate, string $endDate): array
    {
        $query = "SELECT * FROM {$this->table} WHERE datum BETWEEN :startDate AND :endDate ORDER BY datum ASC";
        return $this->dbConnection->queryAndFetchAllObjects($query, ['startDate' => $startDate, 'endDate' => $endDate]);
    }
    
    public function getTotalKilometers(): int
    {
        $query = "SELECT SUM(prijedeni_km) as total FROM {$this->table}";
        $total = $this->dbConnection->querySingleValue($query);
        return (int)($total ?? 0);
    }

    public function getTotalFuelCost(): float
    {
        $query = "SELECT SUM(gorivo_uk_cijena) as total FROM {$this->table}";
        $total = $this->dbConnection->querySingleValue($query);
        return (float)($total ?? 0);
    }
    
    public function getVehicleStatistics(): array
    {
        $query = "SELECT vozilo, COUNT(*) as count, SUM(prijedeni_km) as total_km, 
                SUM(gorivo_uk_cijena) as total_cost 
                FROM {$this->table} 
                GROUP BY vozilo 
                ORDER BY count DESC";
        return $this->dbConnection->queryAndFetchAllAssoc($query);
    }

    public function getDriverStatistics(): array
    {
        $query = "SELECT vozac, COUNT(*) as count, SUM(prijedeni_km) as total_km, 
                SUM(gorivo_uk_cijena) as total_cost 
                FROM {$this->table} 
                GROUP BY vozac 
                ORDER BY count DESC";
        return $this->dbConnection->queryAndFetchAllAssoc($query);
    }
    
    public function getMonthlyStatistics(int $year = null): array
    {
        if ($year === null) {
            $year = date('Y');
        }
        
        $query = "SELECT MONTH(datum) as month, COUNT(*) as count, 
                SUM(prijedeni_km) as total_km, SUM(gorivo_uk_cijena) as total_cost 
                FROM {$this->table} 
                WHERE YEAR(datum) = :year 
                GROUP BY MONTH(datum) 
                ORDER BY month ASC";
        
        return $this->dbConnection->queryAndFetchAllAssoc($query, ['year' => $year]);
    }
    
    public function getRecentOrders(int $limit = 5): array
    {
        $query = "SELECT * FROM {$this->table} ORDER BY datum DESC, id DESC LIMIT " . (int)$limit;
        return $this->dbConnection->queryAndFetchAllAssoc($query);
    }
    
    /**
     * Calculate fuel efficiency (km/l) for a vehicle over a period
     * 
     * @param string $vozilo Vehicle name
     * @param string|null $startDate Start date (Y-m-d format) or null for all time
     * @param string|null $endDate End date (Y-m-d format) or null for all time
     * @return float Average fuel efficiency in km/l
     */
    public function calculateFuelEfficiency(string $vozilo, ?string $startDate = null, ?string $endDate = null): float
    {
        try {
            $params = ['vozilo' => $vozilo];
            $dateCondition = '';
            
            if ($startDate && $endDate) {
                $dateCondition = " AND datum BETWEEN :startDate AND :endDate";
                $params['startDate'] = $startDate;
                $params['endDate'] = $endDate;
            }
            
            $query = "SELECT SUM(prijedeni_km) as total_km, SUM(gorivo_kolicina) as total_fuel 
                    FROM {$this->table} 
                    WHERE vozilo = :vozilo" . $dateCondition;
                    
            $data = $this->dbConnection->queryAndFetchAssoc($query, $params);
            
            if (!$data || (float)$data['total_fuel'] === 0.0) {
                return 0.0;
            }
            
            return (float)$data['total_km'] / (float)$data['total_fuel'];
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in calculateFuelEfficiency', [
                    'error' => $e->getMessage(),
                    'vehicle' => $vozilo
                ]);
            }
            return 0.0;
        }
    }
    
    /**
     * Get average kilometers driven per day for specified period
     * 
     * @param string|null $startDate Start date (Y-m-d format) or null for all time
     * @param string|null $endDate End date (Y-m-d format) or null for current date
     * @return float Average kilometers per day
     */
    public function getAverageKmPerDay(?string $startDate = null, ?string $endDate = null): float
    {
        try {
            $params = [];
            $whereClause = "";
            
            if ($startDate) {
                $whereClause = " WHERE datum >= :startDate";
                $params['startDate'] = $startDate;
                
                if ($endDate) {
                    $whereClause .= " AND datum <= :endDate";
                    $params['endDate'] = $endDate;
                }
            }
            
            // First, get total kilometers
            $kmQuery = "SELECT SUM(prijedeni_km) as total_km FROM {$this->table}" . $whereClause;
            $totalKm = (float)($this->dbConnection->querySingleValue($kmQuery, $params) ?? 0);
            
            if ($totalKm === 0.0) {
                return 0.0;
            }
            
            // Now get the distinct days
            $daysQuery = "SELECT COUNT(DISTINCT datum) as days FROM {$this->table}" . $whereClause;
            $days = (int)($this->dbConnection->querySingleValue($daysQuery, $params) ?? 0);
            
            if ($days === 0) {
                return 0.0;
            }
            
            return $totalKm / $days;
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getAverageKmPerDay', [
                    'error' => $e->getMessage()
                ]);
            }
            return 0.0;
        }
    }
    
    /**
     * Get daily records for a specified month
     * 
     * @param int $month Month (1-12)
     * @param int|null $year Year or null for current year
     * @return array Daily records for the month
     */
    public function getMonthlyDailyRecords(int $month, ?int $year = null): array
    {
        if ($year === null) {
            $year = date('Y');
        }
        
        try {
            $startDate = sprintf("%04d-%02d-01", $year, $month);
            $endDate = date('Y-m-t', strtotime($startDate));
            
            $query = "SELECT datum, 
                    SUM(prijedeni_km) as km, 
                    SUM(gorivo_kolicina) as fuel_amount,
                    SUM(gorivo_uk_cijena) as fuel_cost,
                    COUNT(*) as trips_count
                    FROM {$this->table}
                    WHERE datum BETWEEN :startDate AND :endDate
                    GROUP BY datum
                    ORDER BY datum ASC";
                    
            return $this->dbConnection->queryAndFetchAllAssoc($query, [
                'startDate' => $startDate,
                'endDate' => $endDate
            ]);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getMonthlyDailyRecords', [
                    'error' => $e->getMessage(),
                    'month' => $month,
                    'year' => $year
                ]);
            }
            return [];
        }
    }
    
    /**
     * Get most frequent routes (pravac_kretanja)
     * 
     * @param int $limit Maximum number of routes to return
     * @return array Most frequent routes
     */
    public function getMostFrequentRoutes(int $limit = 10): array
    {
        try {
            $query = "SELECT pravac_kretanja, COUNT(*) as count, 
                    SUM(prijedeni_km) as total_km,
                    AVG(prijedeni_km) as avg_km
                    FROM {$this->table}
                    WHERE pravac_kretanja IS NOT NULL AND pravac_kretanja != ''
                    GROUP BY pravac_kretanja
                    ORDER BY count DESC
                    LIMIT " . (int)$limit;
                    
            return $this->dbConnection->queryAndFetchAllAssoc($query);
        } catch (\PDOException $e) {
            if ($this->logger) {
                $this->logger->error('Database error in getMostFrequentRoutes', [
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
}