/**
 * BORNA Overview Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardcoded API routes
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    attacks: `${publicUrl}/borna/security/api/attacks?period=24h`
  };

  // Initialize the attacks chart
  initializeAttacksChart();
  
  // Refresh button event listeners
  document.getElementById('refreshAttacksChart')?.addEventListener('click', function() {
    refreshAttacksChart();
  });
  
  /**
   * Initialize the attacks chart
   */
  function initializeAttacksChart() {
    const attacksCtx = document.getElementById('attacksChart')?.getContext('2d');
    
    if (!attacksCtx) return;
    
    // Check if chart already exists and destroy it properly
    if (window.attacksChart && typeof window.attacksChart.destroy === 'function') {
      window.attacksChart.destroy();
    }
    
    // Show loading state
    const chartContainer = document.getElementById('attacksChartContainer');
    if (chartContainer) {
      chartContainer.innerHTML = '<div class="loading-spinner"></div><p class="text-center">Loading chart data...</p>';
    }
    
    // Fetch real data from the API
    fetch(routes.attacks)
      .then(response => response.json())
      .then(data => {
        if (chartContainer) {
          // Restore the canvas
          chartContainer.innerHTML = '<canvas id="attacksChart" width="400" height="200"></canvas>';
        }
        
        // Get the new canvas context
        const attacksCtx = document.getElementById('attacksChart')?.getContext('2d');
        if (!attacksCtx) return;
        
        const chartData = prepareChartData(data);
        
        // Create the chart
        window.attacksChart = new Chart(attacksCtx, {
          type: 'bar',
          data: {
            labels: chartData.labels,
            datasets: [{
              label: 'Attack Attempts',
              data: chartData.values,
              backgroundColor: 'rgba(231, 76, 60, 0.5)',
              borderColor: 'rgba(231, 76, 60, 1)',
              borderWidth: 1
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
              legend: {
                labels: {
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
                }
              },
              tooltip: {
                mode: 'index',
                intersect: false
              }
            },
            scales: {
              y: {
                beginAtZero: true,
                title: {
                  display: true,
                  text: 'Attack Count',
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                },
                grid: {
                  color: 'rgba(var(--border-rgb), 0.2)'
                },
                ticks: {
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                }
              },
              x: {
                title: {
                  display: true,
                  text: 'Hour',
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                },
                grid: {
                  color: 'rgba(var(--border-rgb), 0.2)'
                },
                ticks: {
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                }
              }
            }
          }
        });
        
        // Update "Last updated" text if it exists
        const lastUpdatedElement = document.getElementById('lastUpdated');
        if (lastUpdatedElement) {
          const now = new Date();
          lastUpdatedElement.textContent = now.toLocaleTimeString();
        }
      })
      .catch(error => {
        console.error('Error initializing attacks chart:', error);
        
        if (chartContainer) {
          chartContainer.innerHTML = `
            <div class="borna-alert danger">
              <i class="fas fa-exclamation-triangle"></i>
              <div>Failed to load chart data. Please try again later.</div>
            </div>
            <canvas id="attacksChart" width="400" height="200"></canvas>
          `;
        }
        
        // Create empty chart as fallback
        const attacksCtx = document.getElementById('attacksChart')?.getContext('2d');
        if (!attacksCtx) return;
        
        const emptyData = {
          labels: generateHourLabels(24),
          values: new Array(24).fill(0)
        };
        
        window.attacksChart = new Chart(attacksCtx, {
          type: 'bar',
          data: {
            labels: emptyData.labels,
            datasets: [{
              label: 'Attack Attempts',
              data: emptyData.values,
              backgroundColor: 'rgba(231, 76, 60, 0.5)',
              borderColor: 'rgba(231, 76, 60, 1)',
              borderWidth: 1
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
              legend: {
                labels: {
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
                }
              },
              tooltip: {
                mode: 'index',
                intersect: false
              }
            },
            scales: {
              y: {
                beginAtZero: true,
                title: {
                  display: true,
                  text: 'Attack Count',
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                },
                grid: {
                  color: 'rgba(var(--border-rgb), 0.2)'
                },
                ticks: {
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                }
              },
              x: {
                title: {
                  display: true,
                  text: 'Hour',
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                },
                grid: {
                  color: 'rgba(var(--border-rgb), 0.2)'
                },
                ticks: {
                  color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
                }
              }
            }
          }
        });
      });
  }
  
  /**
   * Prepare data for chart from API response
   */
  function prepareChartData(data) {
    // Check if the API returned valid data
    if (data.success && Array.isArray(data.hourly_data)) {
      return {
        labels: generateHourLabels(data.hourly_data.length),
        values: data.hourly_data
      };
    }
    
    // Return empty data if the API didn't return valid data
    return {
      labels: generateHourLabels(24),
      values: new Array(24).fill(0)
    };
  }
  
  /**
   * Generate hour labels for the last N hours
   */
  function generateHourLabels(numHours) {
    const currentHour = new Date().getHours();
    const labels = [];
    
    for (let i = numHours - 1; i >= 0; i--) {
      const hour = (currentHour - i + 24) % 24;
      labels.push(`${hour}:00`);
    }
    
    return labels;
  }
  
  /**
   * Refresh the attacks chart with new data
   */
  function refreshAttacksChart() {
    // Show loading state on button
    const button = document.getElementById('refreshAttacksChart');
    if (button) {
      const originalText = button.innerHTML;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
      button.disabled = true;
      
      // Make API request to get updated attack data
      fetch(routes.attacks)
        .then(response => response.json())
        .then(data => {
          if (data.success && data.hourly_data) {
            // Update chart with new data
            if (window.attacksChart) {
              window.attacksChart.data.datasets[0].data = data.hourly_data;
              window.attacksChart.update();
            }
            
            // Update "Last updated" text if it exists
            const lastUpdatedElement = document.getElementById('lastUpdated');
            if (lastUpdatedElement) {
              const now = new Date();
              lastUpdatedElement.textContent = now.toLocaleTimeString();
            }
          } else {
            console.error('Failed to get attack data', data);
            
            // Show error message
            const errorElement = document.createElement('div');
            errorElement.className = 'borna-alert danger';
            errorElement.innerHTML = `
              <i class="fas fa-exclamation-triangle"></i>
              <div>Failed to update chart data. Please try again later.</div>
            `;
            
            const container = document.getElementById('attacksChartContainer');
            if (container) {
              container.insertBefore(errorElement, container.firstChild);
              
              // Auto-hide the error after 5 seconds
              setTimeout(() => {
                errorElement.style.opacity = '0';
                setTimeout(() => {
                  errorElement.remove();
                }, 300);
              }, 5000);
            }
          }
          
          // Reset button state
          button.innerHTML = originalText;
          button.disabled = false;
        })
        .catch(error => {
          console.error('Error refreshing attacks chart:', error);
          
          // Show error message
          const errorElement = document.createElement('div');
          errorElement.className = 'borna-alert danger';
          errorElement.innerHTML = `
            <i class="fas fa-exclamation-triangle"></i>
            <div>Failed to update chart data. Error: ${error.message}</div>
          `;
          
          const container = document.getElementById('attacksChartContainer');
          if (container) {
            container.insertBefore(errorElement, container.firstChild);
            
            // Auto-hide the error after 5 seconds
            setTimeout(() => {
              errorElement.style.opacity = '0';
              setTimeout(() => {
                errorElement.remove();
              }, 300);
            }, 5000);
          }
          
          // Reset button state
          button.innerHTML = originalText;
          button.disabled = false;
        });
    }
  }
});
//# sourceMappingURL=borna_overview.91511da5.js.map