/**
 * BORNA Events Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardcoded API routes
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    searchEvents: `${publicUrl}/borna/security/api/search-events`,
    eventDetails: `${publicUrl}/borna/security/api/event`
  };

  // Modal functionality
  const modal = document.getElementById('eventDetailsModal');
  const detailsBtns = document.querySelectorAll('.event-details-btn');
  const closeBtn = document.querySelector('.borna-modal-close');
  
  // Search functionality
  const searchInput = document.getElementById('eventsSearch');
  if (searchInput) {
    searchInput.addEventListener('keyup', function() {
      const searchText = this.value.toLowerCase();
      fetchFilteredEvents(searchText);
    });
  }
  
  /**
   * Fetch and filter events via API
   */
  function fetchFilteredEvents(searchText) {
    const eventsTable = document.querySelector('.events-table');
    const noEventsMessage = document.getElementById('noEventsMessage');
    
    if (!eventsTable) return;
    
    const tbody = eventsTable.querySelector('tbody');
    if (tbody) {
      // Show loading state
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="text-center">
            <div class="loading-spinner"></div>
            <p>Searching events...</p>
          </td>
        </tr>
      `;
    }
    
    // Fetch events from API
    fetch(`${routes.searchEvents}?q=${encodeURIComponent(searchText)}`)
      .then(response => {
        if (!response.ok) {
          throw new Error('Network response was not ok');
        }
        return response.json();
      })
      .then(data => {
        // Clear loading state
        if (tbody) {
          tbody.innerHTML = '';
        }
        
        if (data.success && data.events && data.events.length > 0) {
          // Populate table with filtered events
          data.events.forEach(event => {
            const row = document.createElement('tr');
            row.setAttribute('data-event-id', event.id);
            row.innerHTML = `
              <td class="event-time">${escapeHtml(event.timestamp)}</td>
              <td>
                <span class="event-badge ${escapeHtml(event.event_type.toLowerCase().replace(' ', '-'))}">
                  ${escapeHtml(event.event_type)}
                </span>
              </td>
              <td class="event-ip">${escapeHtml(event.ip || 'N/A')}</td>
              <td class="event-country">${escapeHtml(event.country || 'N/A')}</td>
              <td>
                <button class="btn btn-sm btn-primary event-details-btn" data-event-id="${escapeHtml(event.id)}">
                  <i class="fas fa-info-circle"></i> Details
                </button>
              </td>
            `;
            tbody.appendChild(row);
          });
          
          // Hide no events message
          if (noEventsMessage) {
            noEventsMessage.style.display = 'none';
          }
          
          // Reattach event listeners for new detail buttons
          attachEventDetailsListeners();
        } else {
          // No events found
          if (noEventsMessage) {
            noEventsMessage.style.display = 'block';
            noEventsMessage.textContent = searchText 
              ? 'No matching events found.' 
              : 'No events exist.';
          }
        }
      })
      .catch(error => {
        console.error('Error searching events:', error);
        
        // Clear loading state and show error
        if (tbody) {
          tbody.innerHTML = `
            <tr>
              <td colspan="5" class="text-center text-danger">
                <i class="fas fa-exclamation-triangle"></i>
                Failed to search events. ${error.message}
              </td>
            </tr>
          `;
        }
      });
  }
  
  /**
   * Escape HTML to prevent XSS
   * @param {string} html - The string to escape
   * @returns {string} - The escaped string
   */
  function escapeHtml(html) {
    if (!html) return '';
    const div = document.createElement('div');
    div.textContent = html;
    return div.innerHTML;
  }
  
  /**
   * Attach event listeners to event details buttons
   */
  function attachEventDetailsListeners() {
    const detailsBtns = document.querySelectorAll('.event-details-btn');
    detailsBtns.forEach(btn => {
      btn.addEventListener('click', function() {
        const eventId = this.getAttribute('data-event-id');
        openEventDetailsModal(eventId);
      });
    });
  }
  
  // Initial event details button listeners
  attachEventDetailsListeners();
  
  // Reset filters button
  const resetFiltersBtn = document.getElementById('resetFilters');
  if (resetFiltersBtn) {
    resetFiltersBtn.addEventListener('click', function() {
      resetFilters();
    });
  }
  
  const resetFiltersBtnEmpty = document.getElementById('resetFiltersBtn');
  if (resetFiltersBtnEmpty) {
    resetFiltersBtnEmpty.addEventListener('click', function() {
      resetFilters();
    });
  }
  
  // Close modal
  if (closeBtn) {
    closeBtn.addEventListener('click', function() {
      closeModal();
    });
  }
  
  // Close modal when clicking outside
  window.addEventListener('click', function(event) {
    if (event.target === modal) {
      closeModal();
    }
  });
  
  // Initial load state check
  checkEmptyEvents();
  
  /**
   * Check if events table is empty and show message if needed
   */
  function checkEmptyEvents() {
    const eventsTable = document.querySelector('.events-table');
    const noEventsMessage = document.getElementById('noEventsMessage');
    
    if (eventsTable && noEventsMessage) {
      const rows = eventsTable.querySelectorAll('tbody tr');
      
      if (rows.length === 0) {
        // No events, show message
        noEventsMessage.style.display = 'block';
        eventsTable.style.display = 'none';
      } else {
        // Events found, hide message
        noEventsMessage.style.display = 'none';
        eventsTable.style.display = 'table';
      }
    }
  }
  
  /**
   * Reset filter form to default values
   */
  function resetFilters() {
    const form = document.getElementById('eventsFilterForm');
    if (form) {
      // Reset date inputs to last 24 hours
      const to = new Date();
      const from = new Date(to);
      from.setDate(from.getDate() - 1);
      
      document.getElementById('to').value = formatDateTime(to);
      document.getElementById('from').value = formatDateTime(from);
      
      // Reset event type select
      document.getElementById('eventType').value = '';
      
      // Submit form
      form.submit();
    }
  }
  
  /**
   * Format date for datetime-local input
   */
  function formatDateTime(date) {
    return date.getFullYear() + '-' + 
           pad(date.getMonth() + 1) + '-' + 
           pad(date.getDate()) + 'T' + 
           pad(date.getHours()) + ':' + 
           pad(date.getMinutes());
  }
  
  /**
   * Pad number with leading zero if needed
   */
  function pad(number) {
    return number < 10 ? '0' + number : number;
  }
  
  /**
   * Open event details modal
   */
  function openEventDetailsModal(eventId) {
    if (!modal || !eventId) return;
    
    // Show modal
    modal.classList.add('show');
    
    // Set content to loading state
    const modalContent = modal.querySelector('.event-details-content');
    modalContent.innerHTML = `
      <div class="loading-spinner"></div>
      <p class="text-center">Loading event details...</p>
    `;
    
    // Get event data - First try to get it from the row
    const eventRow = document.querySelector(`tr[data-event-id="${eventId}"]`);
    
    if (eventRow) {
      displayEventFromRow(eventRow, modalContent);
    } else {
      // If not found in the table, fetch it from server
      fetchEventDetails(eventId, modalContent);
    }
  }
  
  /**
   * Fetch event details from server
   */
  function fetchEventDetails(eventId, modalContent) {
    fetch(`${routes.eventDetails}/${eventId}`)
      .then(response => {
        if (!response.ok) {
          throw new Error('Failed to fetch event details');
        }
        return response.json();
      })
      .then(eventData => {
        if (eventData.success && eventData.event) {
          displayEventFromApi(eventData.event, modalContent);
        } else {
          throw new Error('Invalid event data received');
        }
      })
      .catch(error => {
        console.error('Error fetching event details:', error);
        
        // Show error
        modalContent.innerHTML = `
          <div class="borna-alert danger">
            <i class="fas fa-exclamation-circle"></i>
            <div>Error loading event details: ${error.message}</div>
          </div>
        `;
      });
  }
  
  /**
   * Display event details from table row data
   */
  function displayEventFromRow(eventRow, modalContent) {
    const timestamp = eventRow.querySelector('.event-time')?.textContent.trim() || '';
    const eventType = eventRow.querySelector('.event-badge')?.textContent.trim() || '';
    const ip = eventRow.querySelector('.event-ip')?.textContent.trim() || '';
    const country = eventRow.querySelector('.event-country')?.textContent.trim() || '';
    const details = eventRow.cells[3]?.innerHTML || '';
    
    // Create event details HTML
    let html = `
      <div class="event-detail-row">
        <div class="detail-label">Timestamp:</div>
        <div class="detail-value">${timestamp}</div>
      </div>
      <div class="event-detail-row">
        <div class="detail-label">Event Type:</div>
        <div class="detail-value">${eventType}</div>
      </div>
    `;
    
    if (ip) {
      html += `
        <div class="event-detail-row">
          <div class="detail-label">IP Address:</div>
          <div class="detail-value">${ip}</div>
        </div>
      `;
    }
    
    if (country) {
      html += `
        <div class="event-detail-row">
          <div class="detail-label">Country:</div>
          <div class="detail-value">${country}</div>
        </div>
      `;
    }
    
    html += `
      <div class="event-detail-row">
        <div class="detail-label">Details:</div>
        <div class="detail-value">${details}</div>
      </div>
    `;
    
    // Get raw event data from data attribute if available
    const rawEventData = eventRow.getAttribute('data-event-json');
    if (rawEventData) {
      try {
        const jsonData = JSON.parse(rawEventData);
        html += `
          <div class="event-detail-row">
            <div class="detail-label">Raw Event Data:</div>
            <div class="event-json"><pre>${JSON.stringify(jsonData, null, 2)}</pre></div>
          </div>
        `;
      } catch (e) {
        console.error('Error parsing event JSON:', e);
      }
    }
    
    // Add actions
    if (eventRow.querySelector('.block-ip-form')) {
      const ipValue = eventRow.querySelector('input[name="ip"]').value;
      html += `
        <div class="event-detail-row">
          <div class="detail-label">Actions:</div>
          <div class="detail-value">
            <form action="ips" method="post" class="block-ip-form">
              <input type="hidden" name="action" value="block">
              <input type="hidden" name="ip" value="${ipValue}">
              <input type="hidden" name="reason" value="Blocked from event details">
              <button type="submit" class="borna-btn danger">
                <i class="fas fa-ban"></i> Block IP
              </button>
            </form>
          </div>
        </div>
      `;
    }
    
    // Update modal content
    modalContent.innerHTML = html;
  }
  
  /**
   * Display event details from API response
   */
  function displayEventFromApi(event, modalContent) {
    // Create event details HTML
    let html = `
      <div class="event-detail-row">
        <div class="detail-label">Timestamp:</div>
        <div class="detail-value">${event.timestamp}</div>
      </div>
      <div class="event-detail-row">
        <div class="detail-label">Event Type:</div>
        <div class="detail-value">${event.event_type}</div>
      </div>
    `;
    
    if (event.data && event.data.ip) {
      html += `
        <div class="event-detail-row">
          <div class="detail-label">IP Address:</div>
          <div class="detail-value">${event.data.ip}</div>
        </div>
      `;
    }
    
    if (event.data && event.data.country) {
      html += `
        <div class="event-detail-row">
          <div class="detail-label">Country:</div>
          <div class="detail-value">${event.data.country}</div>
        </div>
      `;
    }
    
    if (event.data && event.data.threats) {
      html += `
        <div class="event-detail-row">
          <div class="detail-label">Threats:</div>
          <div class="detail-value">
            <ul class="threats-list">
              ${event.data.threats.map(threat => `
                <li>
                  <strong>${threat.type}:</strong> ${threat.description || 'No description'}
                  <span class="badge score-badge">${threat.score}</span>
                </li>
              `).join('')}
            </ul>
          </div>
        </div>
      `;
    }
    
    html += `
      <div class="event-detail-row">
        <div class="detail-label">Raw Event Data:</div>
        <div class="event-json"><pre>${JSON.stringify(event, null, 2)}</pre></div>
      </div>
    `;
    
    // Add actions
    if (event.data && event.data.ip) {
      html += `
        <div class="event-detail-row">
          <div class="detail-label">Actions:</div>
          <div class="detail-value">
            <form action="ips" method="post" class="block-ip-form">
              <input type="hidden" name="action" value="block">
              <input type="hidden" name="ip" value="${event.data.ip}">
              <input type="hidden" name="reason" value="Blocked from event details">
              <button type="submit" class="borna-btn danger">
                <i class="fas fa-ban"></i> Block IP
              </button>
            </form>
          </div>
        </div>
      `;
    }
    
    // Update modal content
    modalContent.innerHTML = html;
  }
  
  /**
   * Close the modal
   */
  function closeModal() {
    if (modal) {
      modal.classList.remove('show');
    }
  }
});
//# sourceMappingURL=borna_security.46746c85.js.map