/**
 * BORNA Settings Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Hardcoded API routes
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    testConnection: `${publicUrl}/borna/security/api/test-connection`,
    resetSettings: `${publicUrl}/borna/security/api/reset-settings`
  };

  // Security level selector
  const levelOptions = document.querySelectorAll('.level-option');
  levelOptions.forEach(option => {
    option.addEventListener('click', function() {
      const level = this.getAttribute('data-level');
      const radioInput = this.querySelector('input[type="radio"]');
      
      // Update radio input
      radioInput.checked = true;
      
      // Update active class
      levelOptions.forEach(opt => {
        opt.classList.remove('active');
      });
      this.classList.add('active');
      
      // Show warning for high security level
      const warningElement = document.getElementById('securityLevelWarning');
      if (warningElement) {
        if (level === '3') {
          warningElement.style.display = 'block';
        } else {
          warningElement.style.display = 'none';
        }
      }
    });
  });
  
  // CSP level selector
  const cspRadios = document.querySelectorAll('input[name="csp_level"]');
  const cspCustomTextarea = document.getElementById('csp-custom-value');
  
  cspRadios.forEach(radio => {
    radio.addEventListener('change', function() {
      if (this.value === 'custom') {
        cspCustomTextarea.disabled = false;
        cspCustomTextarea.focus();
      } else {
        cspCustomTextarea.disabled = true;
      }
    });
  });
  
  // Auto-hide alerts after 5 seconds
  const alerts = document.querySelectorAll('.borna-alert');
  alerts.forEach(alert => {
    setTimeout(() => {
      alert.style.opacity = '0';
      setTimeout(() => {
        alert.style.display = 'none';
      }, 300);
    }, 5000);
  });
  
  // Form submission confirmation
  const settingsForm = document.querySelector('.settings-form');
  if (settingsForm) {
    settingsForm.addEventListener('submit', function(e) {
      const securityLevel = document.querySelector('input[name="security_level"]:checked')?.value;
      
      // If changing to high security level, confirm
      if (securityLevel === '3') {
        if (!confirm('Warning: High security level may impact application performance. Are you sure you want to proceed?')) {
          e.preventDefault();
          return;
        }
      }
      
      // Validate country codes
      const allowedCountries = document.getElementById('allowed_countries')?.value;
      const blockedCountries = document.getElementById('blocked_countries')?.value;
      
      if (allowedCountries && !validateCountryCodes(allowedCountries)) {
        alert('Invalid format in Allowed Countries. Please use comma-separated 2-letter country codes (e.g., US, GB, DE)');
        e.preventDefault();
        return;
      }
      
      if (blockedCountries && !validateCountryCodes(blockedCountries)) {
        alert('Invalid format in Blocked Countries. Please use comma-separated 2-letter country codes (e.g., RU, CN)');
        e.preventDefault();
        return;
      }
    });
  }
  
  // Validate country codes format
  function validateCountryCodes(input) {
    if (!input.trim()) return true; // Empty is valid
    
    // Check format after normalizing
    const normalized = normalizeCountryCodes(input);
    const regex = /^[A-Z]{2}(,\s*[A-Z]{2})*$/;
    return regex.test(normalized);
  }
  
  // Normalize country codes format
  function normalizeCountryCodes(input) {
    // Split by commas, trim whitespace, convert to uppercase, filter empty values
    const codes = input.split(',')
      .map(code => code.trim().toUpperCase())
      .filter(code => code.length > 0);
    
    // Join back with comma
    return codes.join(',');
  }
  
  // Format country codes on blur
  const countryInputs = document.querySelectorAll('#allowed_countries, #blocked_countries');
  countryInputs.forEach(input => {
    input.addEventListener('blur', function() {
      if (this.value.trim() === '') return;
      
      // Split by commas, trim whitespace, convert to uppercase, filter empty values
      const codes = this.value.split(',')
        .map(code => code.trim().toUpperCase())
        .filter(code => code.length > 0);
      
      // Join back with comma and space
      this.value = codes.join(', ');
    });
  });
  
  // Test connection button for database/redis settings
  const testConnectionBtn = document.getElementById('testConnectionBtn');
  if (testConnectionBtn) {
    testConnectionBtn.addEventListener('click', function() {
      // Get connection details from form
      const storageType = document.querySelector('input[name="storage_type"]:checked')?.value;
      
      if (!storageType) {
        alert('Please select a storage type first.');
        return;
      }
      
      // Show loading state
      const originalText = this.innerHTML;
      this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Testing...';
      this.disabled = true;
      
      // Get connection details based on storage type
      let connectionData = {};
      
      if (storageType === 'redis') {
        connectionData = {
          type: 'redis',
          host: document.getElementById('redis_host')?.value,
          port: document.getElementById('redis_port')?.value,
          password: document.getElementById('redis_password')?.value,
          database: document.getElementById('redis_database')?.value
        };
      } else {
        // Database is selected
        connectionData = {
          type: 'database'
        };
      }
      
      // Make API request to test connection
      fetch(routes.testConnection, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(connectionData)
      })
        .then(response => {
          if (!response.ok) {
            throw new Error('Network response was not ok');
          }
          return response.json();
        })
        .then(data => {
          // Show result
          const resultElement = document.getElementById('connectionTestResult');
          
          if (resultElement) {
            if (data.success) {
              resultElement.innerHTML = `
                <div class="borna-alert success">
                  <i class="fas fa-check-circle"></i>
                  <div>Connection successful! ${data.message || ''}</div>
                </div>
              `;
            } else {
              resultElement.innerHTML = `
                <div class="borna-alert danger">
                  <i class="fas fa-exclamation-circle"></i>
                  <div>Connection failed: ${data.message || 'Unknown error'}</div>
                </div>
              `;
            }
            
            // Auto-hide after 10 seconds
            setTimeout(() => {
              resultElement.innerHTML = '';
            }, 10000);
          }
          
          // Reset button state
          this.innerHTML = originalText;
          this.disabled = false;
        })
        .catch(error => {
          console.error('Error testing connection:', error);
          
          // Show error
          const resultElement = document.getElementById('connectionTestResult');
          
          if (resultElement) {
            resultElement.innerHTML = `
              <div class="borna-alert danger">
                <i class="fas fa-exclamation-circle"></i>
                <div>Error testing connection: ${error.message}</div>
              </div>
            `;
            
            // Auto-hide after 10 seconds
            setTimeout(() => {
              resultElement.innerHTML = '';
            }, 10000);
          }
          
          // Reset button state
          this.innerHTML = originalText;
          this.disabled = false;
        });
    });
  }
  
  // Toggle advanced settings
  const advancedToggle = document.getElementById('advancedToggle');
  const advancedSettings = document.getElementById('advancedSettings');
  
  if (advancedToggle && advancedSettings) {
    advancedToggle.addEventListener('click', function() {
      // Toggle visibility
      if (advancedSettings.style.display === 'none' || !advancedSettings.style.display) {
        advancedSettings.style.display = 'block';
        this.innerHTML = '<i class="fas fa-chevron-up"></i> Hide Advanced Settings';
      } else {
        advancedSettings.style.display = 'none';
        this.innerHTML = '<i class="fas fa-chevron-down"></i> Show Advanced Settings';
      }
    });
  }
  
  // Storage type change handler
  const storageTypeRadios = document.querySelectorAll('input[name="storage_type"]');
  const databaseSettings = document.getElementById('database-settings');
  const redisSettings = document.getElementById('redis-settings');
  
  if (storageTypeRadios.length > 0 && databaseSettings && redisSettings) {
    storageTypeRadios.forEach(radio => {
      radio.addEventListener('change', function() {
        if (this.value === 'database') {
          databaseSettings.style.display = 'block';
          redisSettings.style.display = 'none';
        } else if (this.value === 'redis') {
          databaseSettings.style.display = 'none';
          redisSettings.style.display = 'block';
        }
      });
    });
    
    // Initialize based on selected value
    const selectedStorageType = document.querySelector('input[name="storage_type"]:checked')?.value;
    if (selectedStorageType === 'database') {
      databaseSettings.style.display = 'block';
      redisSettings.style.display = 'none';
    } else if (selectedStorageType === 'redis') {
      databaseSettings.style.display = 'none';
      redisSettings.style.display = 'block';
    }
  }
  
  // Settings save confirmation
  const settingsSaveBtn = document.getElementById('settingsSaveBtn');
  if (settingsSaveBtn) {
    settingsSaveBtn.addEventListener('click', function() {
      return confirm('Are you sure you want to save these security settings? Application behavior may change based on your selections.');
    });
  }
  
  // Reset settings to defaults
  const resetSettingsBtn = document.getElementById('resetSettingsBtn');
  if (resetSettingsBtn) {
    resetSettingsBtn.addEventListener('click', function(e) {
      e.preventDefault();
      
      if (confirm('Are you sure you want to reset all security settings to their default values? This action cannot be undone.')) {
        // Fetch to reset settings
        fetch(routes.resetSettings, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          }
        })
        .then(response => response.json())
        .then(data => {
          if (data.success) {
            // Reload page or show success message
            alert('Settings have been reset to default values.');
            window.location.reload();
          } else {
            alert('Failed to reset settings: ' + (data.message || 'Unknown error'));
          }
        })
        .catch(error => {
          console.error('Error resetting settings:', error);
          alert('An error occurred while resetting settings.');
        });
      }
    });
  }
});
//# sourceMappingURL=borna_settings.2488408f.js.map