/**
 * BORNA Performance Page JavaScript
 * Contains functionality for charts and performance displays
 */
document.addEventListener('DOMContentLoaded', function() {
  // Initialize charts
  initCharts();
  
  // Initialize progress bars
  initDataProgressBars();
  
  // Animate results for visual appeal
  animateTestResults();
  
  // Mobile sidebar toggle
  initMobileSidebar();
  
  // Listen for theme changes
  document.addEventListener('themechange', function() {
    // Reinitialize charts when theme changes to update colors
    initCharts();
  });
});

/**
 * Initialize mobile sidebar
 */
function initMobileSidebar() {
  const sidebar = document.querySelector('.borna-sidebar');
  
  // Avoid creating duplicate toggle buttons
  if (!document.querySelector('.borna-mobile-toggle') && sidebar) {
    const mobileToggle = document.createElement('button');
    mobileToggle.className = 'borna-mobile-toggle';
    mobileToggle.innerHTML = '<i class="fas fa-bars"></i>';
    document.body.appendChild(mobileToggle);
    
    // Add click event
    mobileToggle.addEventListener('click', function() {
      sidebar.classList.toggle('show');
      // Change icon based on state
      const icon = this.querySelector('i');
      if (sidebar.classList.contains('show')) {
        icon.className = 'fas fa-times';
      } else {
        icon.className = 'fas fa-bars';
      }
    });
  }
  
  // Close sidebar when clicking outside on mobile
  document.addEventListener('click', function(event) {
    if (!sidebar) return;
    
    const isClickInsideSidebar = sidebar.contains(event.target);
    const isClickOnMobileToggle = event.target.closest('.borna-mobile-toggle');
    
    if (!isClickInsideSidebar && !isClickOnMobileToggle && sidebar.classList.contains('show')) {
      sidebar.classList.remove('show');
      const icon = document.querySelector('.borna-mobile-toggle i');
      if (icon) {
        icon.className = 'fas fa-bars';
      }
    }
  });
}

/**
 * Initialize performance charts
 */
function initCharts() {
  // Destroy existing charts to prevent duplicates on theme change
  if (window.bornaCharts) {
    Object.values(window.bornaCharts).forEach(chart => {
      if (chart && typeof chart.destroy === 'function') {
        chart.destroy();
      }
    });
  }
  
  // Initialize charts array
  window.bornaCharts = {};
  
  // Get theme colors
  const colors = getChartColors();
  
  // Check for chart canvases
  const speedChartCanvas = document.getElementById('speedChart');
  const memoryChartCanvas = document.getElementById('memoryChart');
  
  // If no charts exist, don't initialize
  if (!speedChartCanvas && !memoryChartCanvas) {
    return;
  }
  
  // Check if Chart.js is loaded
  if (typeof Chart === 'undefined') {
    console.warn('Chart.js library not loaded. Performance charts will not be displayed.');
    return;
  }
  
  // Set default chart options for theme compatibility
  Chart.defaults.color = colors.textColor;
  Chart.defaults.borderColor = colors.borderColor;
  Chart.defaults.font.family = "'Poppins', sans-serif";
  
  // Initialize speed chart
  if (speedChartCanvas) {
    initSpeedChart(speedChartCanvas, colors);
  }
  
  // Initialize memory chart
  if (memoryChartCanvas) {
    initMemoryChart(memoryChartCanvas, colors);
  }
}

/**
 * Initialize the speed analysis chart
 * @param {HTMLElement} canvas The chart canvas element
 * @param {Object} colors Theme colors
 */
function initSpeedChart(canvas, colors) {
  const speedData = document.getElementById('speedChartData');
  if (speedData) {
    try {
      const requests = JSON.parse(speedData.getAttribute('data-requests') || '[]');
      const times = JSON.parse(speedData.getAttribute('data-times') || '[]');
      const speedLabels = requests.map(r => r + ' zahtjeva');
      
      window.bornaCharts.speedChart = new Chart(canvas.getContext('2d'), {
        type: 'bar',
        data: {
          labels: speedLabels,
          datasets: [{
            label: 'Prosječno vrijeme analize (ms)',
            data: times,
            backgroundColor: colors.primary + '80', // 50% opacity
            borderColor: colors.primary,
            borderWidth: 1,
            borderRadius: 4
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: {
            legend: {
              labels: {
                color: colors.textColor,
                font: {
                  family: "'Poppins', sans-serif"
                }
              }
            },
            tooltip: {
              backgroundColor: colors.backgroundColor,
              titleColor: colors.textColor,
              bodyColor: colors.textColor,
              borderColor: colors.borderColor,
              borderWidth: 1,
              padding: 10,
              displayColors: true
            }
          },
          scales: {
            y: {
              beginAtZero: true,
              grid: {
                color: colors.gridColor
              },
              ticks: {
                color: colors.textColor
              },
              title: {
                display: true,
                text: 'Vrijeme (ms)',
                color: colors.textColor
              }
            },
            x: {
              grid: {
                color: colors.gridColor
              },
              ticks: {
                color: colors.textColor
              },
              title: {
                display: true,
                text: 'Broj zahtjeva',
                color: colors.textColor
              }
            }
          }
        }
      });
    } catch (e) {
      console.error('Error initializing speed chart:', e);
    }
  }
}

/**
 * Initialize the memory usage chart
 * @param {HTMLElement} canvas The chart canvas element
 * @param {Object} colors Theme colors
 */
function initMemoryChart(canvas, colors) {
  const memData = document.getElementById('memoryChartData');
  if (memData) {
    try {
      const tests = JSON.parse(memData.getAttribute('data-tests') || '[]');
      const usage = JSON.parse(memData.getAttribute('data-usage') || '[]');
      
      window.bornaCharts.memoryChart = new Chart(canvas.getContext('2d'), {
        type: 'bar',
        data: {
          labels: tests,
          datasets: [{
            label: 'Vršna potrošnja memorije (MB)',
            data: usage,
            backgroundColor: colors.secondary + '80', // 50% opacity
            borderColor: colors.secondary,
            borderWidth: 1,
            borderRadius: 4
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: {
            legend: {
              labels: {
                color: colors.textColor,
                font: {
                  family: "'Poppins', sans-serif"
                }
              }
            },
            tooltip: {
              backgroundColor: colors.backgroundColor,
              titleColor: colors.textColor,
              bodyColor: colors.textColor,
              borderColor: colors.borderColor,
              borderWidth: 1,
              padding: 10,
              displayColors: true
            }
          },
          scales: {
            y: {
              beginAtZero: true,
              grid: {
                color: colors.gridColor
              },
              ticks: {
                color: colors.textColor
              },
              title: {
                display: true,
                text: 'Memorija (MB)',
                color: colors.textColor
              }
            },
            x: {
              grid: {
                color: colors.gridColor
              },
              ticks: {
                color: colors.textColor
              },
              title: {
                display: true,
                text: 'Test slučaj',
                color: colors.textColor
              }
            }
          }
        }
      });
    } catch (e) {
      console.error('Error initializing memory chart:', e);
    }
  }
}

/**
 * Initialize progress bars with data-progress attribute
 */
function initDataProgressBars() {
  const progressBars = document.querySelectorAll('[data-progress]');
  
  progressBars.forEach(bar => {
    const progressValue = parseFloat(bar.getAttribute('data-progress') || 0);
    bar.style.width = progressValue + '%';
  });
}

/**
 * Animate test results for visual appeal
 */
function animateTestResults() {
  // Animate progress bars 
  const progressBars = document.querySelectorAll('.borna-test-progress-bar');
  progressBars.forEach(bar => {
    // Read from data-progress attribute
    const value = parseFloat(bar.getAttribute('data-progress') || 0);
    bar.style.width = '0%';
    
    setTimeout(() => {
      bar.style.transition = 'width 1s ease-in-out';
      bar.style.width = value + '%';
    }, 300);
  });
  
  // Animate numeric values
  const testScores = document.querySelectorAll('.metric-box-value, .display-4');
  testScores.forEach(score => {
    const text = score.textContent.trim();
    // Check if it contains a number or fraction
    if (/\d/.test(text)) {
      let finalValue;
      
      // Handle fractions like "5 / 10"
      if (text.includes('/')) {
        // Save the format to restore it later
        const parts = text.split('/');
        const num = parseInt(parts[0].trim(), 10);
        // Don't animate if parsing failed
        if (isNaN(num)) return;
        
        finalValue = num;
        const denom = parts[1].trim();
        
        score.setAttribute('data-denominator', denom);
        score.textContent = '0';
      } else {
        finalValue = parseInt(text, 10);
        // Don't animate if parsing failed
        if (isNaN(finalValue)) return;
        
        score.textContent = '0';
      }
      
      let currentValue = 0;
      const duration = 1000; // 1 second
      const stepTime = 20; // 20ms
      const steps = duration / stepTime;
      const increment = finalValue / steps;
      
      const counter = setInterval(() => {
        currentValue += increment;
        
        if (currentValue >= finalValue) {
          clearInterval(counter);
          
          // Check if we need to restore a fraction format
          const denominator = score.getAttribute('data-denominator');
          if (denominator) {
            score.textContent = finalValue + ' / ' + denominator;
          } else {
            score.textContent = Math.round(finalValue);
          }
        } else {
          score.textContent = Math.round(currentValue);
          // Add denominator during animation if needed
          const denominator = score.getAttribute('data-denominator');
          if (denominator) {
            score.textContent += ' / ' + denominator;
          }
        }
      }, stepTime);
    }
  });
}

/**
 * Get theme-aware colors for charts
 * @returns {Object} Theme colors
 */
function getChartColors() {
  // If KrizanUI is available, use its theme colors
  if (window.KrizanUI && window.KrizanUI.getThemeColors) {
    const themeColors = window.KrizanUI.getThemeColors();
    return {
      primary: themeColors.accentPrimary,
      secondary: themeColors.accentSecondary,
      success: '#2ecc71',
      info: '#3498db',
      warning: '#f39c12',
      danger: '#e74c3c',
      backgroundColor: document.documentElement.getAttribute('data-theme') === 'light' ? 
        'rgba(255, 255, 255, 0.9)' : 
        'rgba(26, 26, 26, 0.9)',
      textColor: document.documentElement.getAttribute('data-theme') === 'light' ? 
        '#2c3e50' : 
        '#ffffff',
      borderColor: document.documentElement.getAttribute('data-theme') === 'light' ? 
        '#e9ecef' : 
        '#2d2d2d',
      gridColor: document.documentElement.getAttribute('data-theme') === 'light' ? 
        'rgba(0, 0, 0, 0.1)' : 
        'rgba(255, 255, 255, 0.1)'
    };
  }
  
  // Default colors if KrizanUI is not available
  return {
    primary: '#3498db',
    secondary: '#2ecc71',
    success: '#2ecc71',
    info: '#3498db',
    warning: '#f39c12',
    danger: '#e74c3c',
    backgroundColor: 'rgba(255, 255, 255, 0.9)',
    textColor: '#2c3e50',
    borderColor: '#e9ecef',
    gridColor: 'rgba(0, 0, 0, 0.1)'
  };
}
//# sourceMappingURL=borna_performance.7afbd69f.js.map