/**
 * LUKA Performance Page JavaScript
 */
document.addEventListener('DOMContentLoaded', function() {
  // Check if Chart.js is loaded
  if (typeof Chart === 'undefined') {
      console.error('Chart.js is not loaded. Please include the Chart.js library before this script.');
      return; // Exit early if Chart is not available
  }
  
  // Hardkodirane rute API-ja
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  // Define all routes used in this file
  const routes = {
    metrics: `${publicUrl}/luka/digital-twin/api/metrics`,
    componentMetrics: `${publicUrl}/luka/digital-twin/api/component-metrics`
  };
  
  // Initialize the detailed performance chart
  initializeDetailedChart();
  
  // Load component performance data
  loadComponentData();
  
  // Generate performance insights
  generatePerformanceInsights();
  
  // Event listeners
  document.getElementById('refreshDetailedChart')?.addEventListener('click', function() {
    refreshDetailedChart();
  });
  
  document.getElementById('refreshComponents')?.addEventListener('click', function() {
    loadComponentData();
  });
  
  document.getElementById('exportPerformanceData')?.addEventListener('click', function() {
    exportPerformanceData();
  });
  
  // Time range selector change
  document.getElementById('performanceTimeRange')?.addEventListener('change', function() {
    refreshDetailedChart();
  });
  
  // Metric filter change events
  document.querySelectorAll('input[name="metric"]').forEach(checkbox => {
    checkbox.addEventListener('change', function() {
      updateChartVisibility();
    });
  });
  
  // Granularity change events
  document.querySelectorAll('input[name="granularity"]').forEach(radio => {
    radio.addEventListener('change', function() {
      refreshDetailedChart();
    });
  });
  
  /**
   * Initialize the detailed performance chart
   */
  function initializeDetailedChart() {
    const ctx = document.getElementById('detailedPerformanceChart')?.getContext('2d');
    
    if (!ctx) return;
    
    // Check if chart already exists and destroy it properly
    if (window.detailedPerformanceChart && typeof window.detailedPerformanceChart.destroy === 'function') {
      window.detailedPerformanceChart.destroy();
    }
    
    // Create empty chart with loading state
    window.detailedPerformanceChart = new Chart(ctx, {
      type: 'line',
      data: {
        labels: [],
        datasets: [
          {
            label: 'Response Time (ms)',
            data: [],
            borderColor: getComputedStyle(document.documentElement).getPropertyValue('--accent-primary'),
            backgroundColor: 'rgba(var(--accent-primary-rgb), 0.1)',
            borderWidth: 2,
            tension: 0.3,
            fill: true,
            yAxisID: 'y'
          },
          {
            label: 'Request Count',
            data: [],
            borderColor: '#e74c3c',
            backgroundColor: 'rgba(231, 76, 60, 0.1)',
            borderWidth: 2,
            tension: 0.3,
            fill: true,
            yAxisID: 'y1'
          },
          {
            label: 'Error Rate (%)',
            data: [],
            borderColor: '#f39c12',
            backgroundColor: 'rgba(243, 156, 18, 0.1)',
            borderWidth: 2,
            tension: 0.3,
            fill: true,
            yAxisID: 'y2',
            hidden: true
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          mode: 'index',
          intersect: false
        },
        plugins: {
          legend: {
            position: 'top',
            labels: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-primary')
            }
          },
          tooltip: {
            mode: 'index',
            intersect: false
          }
        },
        scales: {
          x: {
            title: {
              display: true,
              text: 'Time',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y: {
            type: 'linear',
            display: true,
            position: 'left',
            title: {
              display: true,
              text: 'Response Time (ms)',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              color: 'rgba(var(--border-rgb), 0.2)'
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y1: {
            type: 'linear',
            display: true,
            position: 'right',
            title: {
              display: true,
              text: 'Request Count',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            grid: {
              drawOnChartArea: false
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          },
          y2: {
            type: 'linear',
            display: false,
            position: 'right',
            title: {
              display: true,
              text: 'Error Rate (%)',
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            },
            min: 0,
            max: 100,
            grid: {
              drawOnChartArea: false
            },
            ticks: {
              color: getComputedStyle(document.documentElement).getPropertyValue('--text-secondary')
            }
          }
        }
      }
    });
    
    // Load initial data
    refreshDetailedChart();
  }
  
  /**
   * Refresh the detailed performance chart
   */
  function refreshDetailedChart() {
    // Show loading state
    const button = document.getElementById('refreshDetailedChart');
    let originalText = 'Refresh';
    
    if (button) {
      originalText = button.innerHTML;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
      button.disabled = true;
    }
    
    // Get time range and granularity
    const timeRange = document.getElementById('performanceTimeRange')?.value || 168; // Default to 7 days
    const granularity = document.querySelector('input[name="granularity"]:checked')?.value || 'hour';
    
    // Calculate from date
    const to = new Date();
    const from = new Date(to);
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Make API request
    fetch(`${routes.metrics}?type=performance&from=${from.toISOString()}&to=${to.toISOString()}`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          updateDetailedChart(data.data, granularity);
        } else {
          console.error('No data returned from API or empty dataset');
          showChartError('detailedPerformanceChart', 'No performance data available');
        }
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
      })
      .catch(error => {
        console.error('Error fetching performance data:', error);
        
        // Show error in chart
        showChartError('detailedPerformanceChart', 'Failed to load performance data');
        
        // Reset button state
        if (button) {
          button.innerHTML = originalText;
          button.disabled = false;
        }
      });
  }
  
  /**
   * Show an error message on the chart
   */
  function showChartError(chartId, message) {
    const chartContainer = document.getElementById(chartId)?.parentElement;
    if (!chartContainer) return;
    
    // Remove existing error message if any
    const existingError = chartContainer.querySelector('.chart-error-message');
    if (existingError) {
      existingError.remove();
    }
    
    // Create error message
    const errorDiv = document.createElement('div');
    errorDiv.className = 'chart-error-message';
    errorDiv.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
    chartContainer.appendChild(errorDiv);
  }
  
  /**
   * Update the detailed chart with new data
   */
  function updateDetailedChart(data, granularity) {
    if (!window.detailedPerformanceChart || typeof window.detailedPerformanceChart.update !== 'function') return;
    
    // Remove any error message
    const chartContainer = document.getElementById('detailedPerformanceChart')?.parentElement;
    if (chartContainer) {
      const existingError = chartContainer.querySelector('.chart-error-message');
      if (existingError) {
        existingError.remove();
      }
    }
    
    // Format labels based on granularity
    const labels = data.map(item => {
      const date = new Date(item.timestamp || item.hour || item.day);
      return granularity === 'hour' 
        ? `${date.getMonth() + 1}/${date.getDate()} ${date.getHours()}:00` 
        : `${date.getMonth() + 1}/${date.getDate()}/${date.getFullYear()}`;
    });
    
    // Extract data series
    const responseTimeData = data.map(item => (item.avg_response_time || item.value || 0) * 1000);
    const requestCountData = data.map(item => item.request_count || item.count || 0);
    const errorRateData = data.map(item => item.error_rate || 0);
    
    // Update chart data
    window.detailedPerformanceChart.data.labels = labels;
    window.detailedPerformanceChart.data.datasets[0].data = responseTimeData;
    window.detailedPerformanceChart.data.datasets[1].data = requestCountData;
    window.detailedPerformanceChart.data.datasets[2].data = errorRateData;
    
    // Update chart
    window.detailedPerformanceChart.update();
    
    // Update chart visibility based on checkbox state
    updateChartVisibility();
  }
  
  /**
   * Update chart visibility based on checkbox selections
   */
  function updateChartVisibility() {
    if (!window.detailedPerformanceChart || typeof window.detailedPerformanceChart.update !== 'function') return;
    
    // Get selected metrics
    const selectedMetrics = Array.from(document.querySelectorAll('input[name="metric"]:checked'))
      .map(checkbox => checkbox.value);
    
    // Update dataset visibility
    window.detailedPerformanceChart.data.datasets[0].hidden = !selectedMetrics.includes('response_time');
    window.detailedPerformanceChart.data.datasets[1].hidden = !selectedMetrics.includes('request_count');
    window.detailedPerformanceChart.data.datasets[2].hidden = !selectedMetrics.includes('error_rate');
    
    // Update y-axis visibility
    window.detailedPerformanceChart.options.scales.y.display = selectedMetrics.includes('response_time');
    window.detailedPerformanceChart.options.scales.y1.display = selectedMetrics.includes('request_count');
    window.detailedPerformanceChart.options.scales.y2.display = selectedMetrics.includes('error_rate');
    
    // Update chart
    window.detailedPerformanceChart.update();
  }
  
  /**
   * Load component performance data
   */
  function loadComponentData() {
    // Show loading state
    const table = document.getElementById('componentsTable');
    if (table) {
      const tbody = table.querySelector('tbody');
      tbody.innerHTML = `
        <tr>
          <td colspan="5" class="loading-state">
            <div class="loading-spinner"></div>
            <p>Loading component data...</p>
          </td>
        </tr>
      `;
    }
    
    // Make API request
    fetch(routes.componentMetrics)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.components && data.components.length > 0) {
          updateComponentTable(data.components, data.avg_request_time);
        } else {
          showEmptyComponentState();
        }
      })
      .catch(error => {
        console.error('Error fetching component data:', error);
        showEmptyComponentState('Failed to load component data. Please try again later.');
      });
  }
  
  /**
   * Update component table with data
   */
  function updateComponentTable(components, avgRequestTime) {
    const table = document.getElementById('componentsTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = '';
    
    // Sort components by average time (descending)
    const sortedComponents = components.sort((a, b) => b.avg_time - a.avg_time);
    
    // Create rows for each component
    sortedComponents.forEach(component => {
      const row = document.createElement('tr');
      
      // Calculate percentage of request time
      const percentage = avgRequestTime > 0 
        ? ((component.avg_time / avgRequestTime) * 100).toFixed(1) 
        : 0;
      
      row.innerHTML = `
        <td>${component.name}</td>
        <td>${(component.avg_time * 1000).toFixed(2)}</td>
        <td>${(component.max_time * 1000).toFixed(2)}</td>
        <td>${component.count}</td>
        <td>
          <div class="progress-container">
            <div class="progress-bar" data-progress="${percentage}"></div>
          </div>
          ${percentage}%
        </td>
      `;
      
      tbody.appendChild(row);
    });

    updateProgressBars();
  }

  /**
   * Update progress bars with CSS variables based on data-progress attributes
   */
  function updateProgressBars() {
    const progressBars = document.querySelectorAll('.progress-bar[data-progress]');
    console.log('Found progress bars:', progressBars.length);
    
    progressBars.forEach(bar => {
      const progressValue = parseFloat(bar.getAttribute('data-progress') || 0);
      bar.style.setProperty('--progress-width', progressValue + '%');
    });
  }
  
  /**
   * Show empty state for component table
   */
  function showEmptyComponentState(message = 'No component performance data available.') {
    const table = document.getElementById('componentsTable');
    if (!table) return;
    
    const tbody = table.querySelector('tbody');
    tbody.innerHTML = `
      <tr>
        <td colspan="5" class="empty-table">
          ${message}
        </td>
      </tr>
    `;
  }
  
  /**
   * Generate performance insights based on data
   */
  function generatePerformanceInsights() {
    const insightsContainer = document.getElementById('performanceInsights');
    if (!insightsContainer) return;
    
    // Show loading state
    insightsContainer.innerHTML = `
      <div class="insight">
        <div class="insight-icon">
          <i class="fas fa-spinner fa-spin"></i>
        </div>
        <div class="insight-content">
          <h4>Loading Insights</h4>
          <p>Analyzing performance data to generate insights...</p>
        </div>
      </div>
    `;
    
    // Load performance data to generate insights
    fetch(`${routes.metrics}?type=performance&from=-7 days&to=now`)
      .then(response => {
        if (!response.ok) {
          throw new Error(`API returned status ${response.status}`);
        }
        return response.json();
      })
      .then(data => {
        if (data.success && data.data && data.data.length > 0) {
          createInsightsFromData(data.data);
        } else {
          // Show no data message
          insightsContainer.innerHTML = `
            <div class="insight neutral">
              <div class="insight-icon">
                <i class="fas fa-info-circle"></i>
              </div>
              <div class="insight-content">
                <h4>No Performance Data</h4>
                <p>There is not enough performance data available to generate insights.</p>
              </div>
            </div>
          `;
        }
      })
      .catch(error => {
        console.error('Error fetching data for insights:', error);
        
        // Show error message
        insightsContainer.innerHTML = `
          <div class="insight warning">
            <div class="insight-icon">
              <i class="fas fa-exclamation-circle"></i>
            </div>
            <div class="insight-content">
              <h4>Error Loading Insights</h4>
              <p>Failed to load performance data for insights. Please try refreshing the page.</p>
            </div>
          </div>
        `;
      });
  }
  
  /**
   * Create insights based on the provided data
   */
  function createInsightsFromData(data) {
    const insightsContainer = document.getElementById('performanceInsights');
    if (!insightsContainer) return;
    
    // Calculate metrics
    const avgResponseTime = calculateAverage(data.map(item => (item.avg_response_time || item.value || 0) * 1000));
    const maxResponseTime = Math.max(...data.map(item => (item.avg_response_time || item.value || 0) * 1000));
    
    // Calculate trend (last 24 points vs previous 24 points)
    const midpoint = Math.floor(data.length / 2);
    const recentData = data.slice(-midpoint);
    const previousData = data.slice(-midpoint * 2, -midpoint);
    
    const recentAvg = calculateAverage(recentData.map(item => (item.avg_response_time || item.value || 0) * 1000));
    const previousAvg = calculateAverage(previousData.map(item => (item.avg_response_time || item.value || 0) * 1000));
    
    const percentChange = previousAvg > 0 
      ? ((recentAvg - previousAvg) / previousAvg) * 100 
      : 0;
    
    // Create insights
    const insights = [];
    
    // Overall performance
    insights.push({
      title: 'Average Performance',
      message: `The average response time is ${avgResponseTime.toFixed(2)} ms with a peak of ${maxResponseTime.toFixed(2)} ms.`,
      type: avgResponseTime < 100 ? 'positive' : (avgResponseTime < 300 ? 'neutral' : 'warning'),
      icon: 'tachometer-alt'
    });
    
    // Performance trend
    insights.push({
      title: 'Performance Trend',
      message: percentChange > 0 
        ? `Response times have increased by ${Math.abs(percentChange).toFixed(1)}% compared to the previous period.`
        : `Response times have decreased by ${Math.abs(percentChange).toFixed(1)}% compared to the previous period.`,
      type: percentChange > 10 ? 'negative' : (percentChange < -10 ? 'positive' : 'neutral'),
      icon: percentChange > 0 ? 'arrow-up' : 'arrow-down'
    });
    
    // Add performance recommendations based on response time
    if (avgResponseTime > 300) {
      insights.push({
        title: 'Performance Recommendation',
        message: 'Consider optimizing database queries and implementing caching to improve response times.',
        type: 'warning',
        icon: 'lightbulb'
      });
    }
    
    // Render insights
    insightsContainer.innerHTML = '';
    insights.forEach(insight => {
      const insightElement = document.createElement('div');
      insightElement.className = `insight ${insight.type}`;
      insightElement.innerHTML = `
        <div class="insight-icon">
          <i class="fas fa-${insight.icon}"></i>
        </div>
        <div class="insight-content">
          <h4>${insight.title}</h4>
          <p>${insight.message}</p>
        </div>
      `;
      insightsContainer.appendChild(insightElement);
    });
  }
  
  /**
   * Export performance data to JSON file
   */
  function exportPerformanceData() {
    // Get time range
    const timeRange = document.getElementById('performanceTimeRange')?.value || 168;
    
    // Calculate date range
    const to = new Date();
    const from = new Date();
    from.setHours(from.getHours() - parseInt(timeRange));
    
    // Create filename with date
    const dateStr = to.toISOString().split('T')[0];
    const filename = `performance_data_${dateStr}.json`;
    
    // Show loading state on button
    const button = document.getElementById('exportPerformanceData');
    if (button) {
      const originalText = button.innerHTML;
      button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Exporting...';
      button.disabled = true;
      
      // Make API request to get data
      fetch(`${routes.metrics}?type=performance&from=${from.toISOString()}&to=${to.toISOString()}&detailed=1`)
        .then(response => {
          if (!response.ok) {
            throw new Error(`API returned status ${response.status}`);
          }
          return response.json();
        })
        .then(data => {
          // Create a Blob and download
          const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
          const url = URL.createObjectURL(blob);
          const a = document.createElement('a');
          a.href = url;
          a.download = filename;
          document.body.appendChild(a);
          a.click();
          document.body.removeChild(a);
          URL.revokeObjectURL(url);
          
          // Reset button state
          button.innerHTML = originalText;
          button.disabled = false;
          
          // Show success toast
          showToast('Performance data exported successfully', 'success');
        })
        .catch(error => {
          console.error('Error exporting data:', error);
          
          // Reset button state
          button.innerHTML = originalText;
          button.disabled = false;
          
          // Show error toast
          showToast('Failed to export performance data', 'error');
        });
    }
  }
  
  /**
   * Show toast notification
   */
  function showToast(message, type = 'info') {
    // Create toast container if it doesn't exist
    let toastContainer = document.querySelector('.luka-toast-container');
    if (!toastContainer) {
      toastContainer = document.createElement('div');
      toastContainer.className = 'luka-toast-container';
      document.body.appendChild(toastContainer);
    }
    
    // Create toast
    const toast = document.createElement('div');
    toast.className = `luka-toast ${type}`;
    
    // Set icon based on type
    let icon = 'info-circle';
    if (type === 'success') icon = 'check-circle';
    if (type === 'error') icon = 'exclamation-circle';
    if (type === 'warning') icon = 'exclamation-triangle';
    
    toast.innerHTML = `
      <div class="toast-icon">
        <i class="fas fa-${icon}"></i>
      </div>
      <div class="toast-content">
        <p>${message}</p>
      </div>
    `;
    
    // Add to container
    toastContainer.appendChild(toast);
    
    // Show toast
    setTimeout(() => {
      toast.classList.add('show');
    }, 10);
    
    // Auto close after 3 seconds
    setTimeout(() => {
      toast.classList.remove('show');
      setTimeout(() => {
        toast.remove();
      }, 300);
    }, 3000);
  }
  
  /* Helper Functions */
  
  /**
   * Calculate average of an array of numbers
   */
  function calculateAverage(arr) {
    if (!arr || arr.length === 0) return 0;
    return arr.reduce((sum, val) => sum + val, 0) / arr.length;
  }
});
//# sourceMappingURL=luka_performance.d70d0954.js.map