/**
 * API Tokens Management JavaScript
 * Handles token generation, listing, and revocation
 * Compatible with tables.js functionality
 */
document.addEventListener('DOMContentLoaded', function() {
    // Hardcoded routes for API endpoints
    const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url
    
    // Define all routes used in this file
    const routes = {
        list: `${publicUrl}/api/tokens/list`,
        generate: `${publicUrl}/api/tokens/generate`,
        revoke: `${publicUrl}/api/tokens/revoke`,
        revokeAll: `${publicUrl}/api/tokens/revoke-all`
    };

    // Get elements
    const tokenTable = document.getElementById('tokensTable');
    const tokensList = document.getElementById('tokensList');
    const tokenSearch = document.getElementById('tokenSearch');
    const generateForm = document.getElementById('generateTokenForm');
    const newTokenDisplay = document.getElementById('newTokenDisplay');
    const copyTokenBtn = document.getElementById('copyTokenBtn');
    const revokeAllBtn = document.getElementById('revokeAllBtn');

    // Load tokens on page load
    loadTokens();
    
    // Initialize table sorting if table exists
    if (tokenTable && window.KrizanUI && typeof window.KrizanUI.initTable === 'function') {
        window.KrizanUI.initTable(tokenTable);
    }
    
    // Handle token generation form submission
    if (generateForm) {
        generateForm.addEventListener('submit', function(e) {
            e.preventDefault();
            generateToken();
        });
    }
    
    // Handle copy button click
    if (copyTokenBtn) {
        copyTokenBtn.addEventListener('click', function() {
            copyTokenToClipboard();
        });
    }
    
    // Handle revoke all button click
    if (revokeAllBtn) {
        revokeAllBtn.addEventListener('click', function() {
            if (confirm('Are you sure you want to revoke all tokens? This cannot be undone.')) {
                revokeAllTokens();
            }
        });
    }

    // Add search functionality if search input exists
    if (tokenSearch) {
        tokenSearch.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = tokenTable.querySelectorAll('tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        });
    }

    // Load user tokens
    function loadTokens() {
        if (!tokensList) return;
        
        tokensList.innerHTML = '<tr><td colspan="6" class="text-center">Loading tokens...</td></tr>';
        
        fetch(routes.list, {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayTokens(data.tokens);
                
                // Initialize responsive labels for mobile view
                if (tokenTable) {
                    addResponsiveLabels();
                }
                
                // Add animation to rows
                const rows = tokenTable?.querySelectorAll('tbody tr');
                if (rows) {
                    animateTableLoad(rows);
                }
            } else {
                showToast('Failed to load tokens: ' + (data.message || 'Unknown error'), 'error');
                tokensList.innerHTML = '<tr><td colspan="6" class="text-center text-danger">Failed to load tokens</td></tr>';
            }
        })
        .catch(error => {
            console.error('Error loading tokens:', error);
            tokensList.innerHTML = '<tr><td colspan="6" class="text-center text-danger">Error loading tokens</td></tr>';
            showToast('Failed to load tokens', 'error');
        });
    }

    // Display tokens in the table
    function displayTokens(tokens) {
        if (!tokensList) return;
        
        if (tokens.length === 0) {
            tokensList.innerHTML = '<tr><td colspan="6" class="text-center">No tokens found</td></tr>';
            return;
        }
        
        let html = '';
        tokens.forEach(token => {
            const created = new Date(token.created_at * 1000).toLocaleString();
            const expires = new Date(token.expires_at * 1000).toLocaleString();
            const lastUsed = token.last_used_at 
                ? new Date(token.last_used_at * 1000).toLocaleString()
                : 'Never';
            
            // Format scopes as badges
            const scopesBadges = token.scopes.map(scope => 
                `<span class="scope-badge ${scope.toLowerCase()}">${escapeHtml(scope)}</span>`
            ).join(' ');
            
            html += `<tr>
                <td data-label="Description">${escapeHtml(token.description || 'No description')}</td>
                <td data-label="Scopes">${scopesBadges}</td>
                <td data-label="Created">${created}</td>
                <td data-label="Last Used">${lastUsed}</td>
                <td data-label="Expires">${expires}</td>
                <td data-label="Actions">
                    <div class="token-actions">
                        <button class="action-btn delete" data-id="${token.id}" aria-label="Revoke Token">
                            <i class="fas fa-trash-alt"></i>
                        </button>
                    </div>
                </td>
            </tr>`;
        });
        
        tokensList.innerHTML = html;
        
        // Add event listeners to delete buttons
        const deleteButtons = tokensList.querySelectorAll('.action-btn.delete');
        deleteButtons.forEach(button => {
            button.addEventListener('click', function() {
                const tokenId = this.getAttribute('data-id');
                if (confirm('Are you sure you want to revoke this token? This cannot be undone.')) {
                    revokeToken(tokenId);
                }
            });
        });
    }

    // Generate new token
    function generateToken() {
        const description = generateForm.querySelector('[name="description"]').value;
        
        // Get selected scopes
        const scopeCheckboxes = generateForm.querySelectorAll('[name="scopes[]"]:checked');
        const scopes = Array.from(scopeCheckboxes).map(cb => cb.value).join(',');
        
        fetch(routes.generate, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: `description=${encodeURIComponent(description)}&scopes=${encodeURIComponent(scopes)}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Display the new token
                document.getElementById('newTokenValue').value = data.token.token;
                newTokenDisplay.classList.remove('d-none');
                
                // Add slide-down animation
                newTokenDisplay.style.animationName = 'slideDown';
                
                // Reset form
                generateForm.reset();
                
                // Show success toast
                showToast('Token generated successfully', 'success');
                
                // Reload tokens list
                loadTokens();
            } else {
                showToast('Failed to generate token: ' + (data.message || 'Unknown error'), 'error');
            }
        })
        .catch(error => {
            console.error('Error generating token:', error);
            showToast('Error generating token. Please try again.', 'error');
        });
    }

    // Revoke a token
    function revokeToken(tokenId) {
        fetch(routes.revoke, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: `token_id=${encodeURIComponent(tokenId)}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('Token revoked successfully', 'success');
                loadTokens(); // Reload the tokens list
            } else {
                showToast('Failed to revoke token: ' + (data.message || 'Unknown error'), 'error');
            }
        })
        .catch(error => {
            console.error('Error revoking token:', error);
            showToast('Error revoking token. Please try again.', 'error');
        });
    }

    // Revoke all tokens
    function revokeAllTokens() {
        fetch(routes.revokeAll, {
            method: 'POST',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast(`Successfully revoked ${data.count} tokens.`, 'success');
                loadTokens(); // Reload the tokens list
            } else {
                showToast('Failed to revoke tokens: ' + (data.message || 'Unknown error'), 'error');
            }
        })
        .catch(error => {
            console.error('Error revoking all tokens:', error);
            showToast('Error revoking tokens. Please try again.', 'error');
        });
    }

    // Copy token to clipboard
    function copyTokenToClipboard() {
        const tokenInput = document.getElementById('newTokenValue');
        tokenInput.select();
        document.execCommand('copy');
        
        showToast('Token copied to clipboard', 'success');
    }

    // Add data-label attributes for responsive display
    function addResponsiveLabels() {
        const headers = tokenTable.querySelectorAll('thead th');
        const rows = tokenTable.querySelectorAll('tbody tr');
        
        rows.forEach(row => {
            const cells = row.querySelectorAll('td');
            cells.forEach((cell, index) => {
                if (headers[index]) {
                    cell.setAttribute('data-label', headers[index].textContent.trim());
                }
            });
        });
    }
    
    // Animate table rows on load
    function animateTableLoad(rows) {
        rows.forEach((row, index) => {
            row.classList.add('loading-row');
            
            setTimeout(() => {
                row.classList.remove('loading-row');
            }, 100 + (index * 50));
        });
    }

    // Escape HTML to prevent XSS
    function escapeHtml(str) {
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    // Show toast notification
    function showToast(message, type = 'info') {
        // Create toast container if it doesn't exist
        let toastContainer = document.getElementById('toast-container');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.id = 'toast-container';
            document.body.appendChild(toastContainer);
            
            // Add CSS if not already present
            if (!document.getElementById('toast-css')) {
                const style = document.createElement('style');
                style.id = 'toast-css';
                style.textContent = `
                    #toast-container {
                        position: fixed;
                        bottom: 20px;
                        right: 20px;
                        z-index: 9999;
                    }
                    .toast {
                        min-width: 250px;
                        margin-top: 10px;
                        padding: 15px;
                        border-radius: 4px;
                        box-shadow: 0 2px 10px rgba(0,0,0,0.2);
                        color: white;
                        display: flex;
                        align-items: center;
                        animation: toast-fadein 0.5s, toast-fadeout 0.5s 2.5s;
                        animation-fill-mode: forwards;
                        opacity: 0;
                    }
                    .toast.success { background-color: #2ecc71; }
                    .toast.error { background-color: #e74c3c; }
                    .toast.info { background-color: #3498db; }
                    .toast i { margin-right: 10px; }
                    @keyframes toast-fadein {
                        from { opacity: 0; transform: translateY(20px); }
                        to { opacity: 1; transform: translateY(0); }
                    }
                    @keyframes toast-fadeout {
                        from { opacity: 1; transform: translateY(0); }
                        to { opacity: 0; transform: translateY(-20px); }
                    }
                    @keyframes slideDown {
                        from { max-height: 0; opacity: 0; }
                        to { max-height: 500px; opacity: 1; }
                    }
                `;
                document.head.appendChild(style);
            }
        }
        
        // Create toast
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        
        // Set icon based on type
        let icon = 'info-circle';
        if (type === 'success') icon = 'check-circle';
        if (type === 'error') icon = 'exclamation-circle';
        
        toast.innerHTML = `<i class="fas fa-${icon}"></i> ${message}`;
        toastContainer.appendChild(toast);
        
        // Remove toast after animation
        setTimeout(() => {
            toast.remove();
        }, 3000);
    }
    
    // Make functions globally available for inline button handlers
    window.revokeToken = revokeToken;
    window.revokeAllTokens = revokeAllTokens;
});
//# sourceMappingURL=apitoken.91f49bc0.js.map