/**
 * Global CSP Adapter
 * Automatski dodaje nonce vrijednosti na kritične DOM metode
 * Ova skripta treba biti učitana prije svih ostalih skripti
 */
(function() {
    // Nonce vrijednost će biti dodana od strane servera kroz ResourceLoader
    const NONCE = window.CSP_NONCE || document.currentScript.nonce || '';
    
    // Spremi u globalnu varijablu za druge skripte
    window.CSP_NONCE = NONCE;
    
    // Sačuvaj originalne metode koje ćemo overrideati
    const originalCreateElement = document.createElement;
    const originalSetAttribute = Element.prototype.setAttribute;
    const originalAppendChild = Node.prototype.appendChild;
    
    // 1. Override document.createElement za automatsko dodavanje nonce atributa
    document.createElement = function(tagName, options) {
        const element = originalCreateElement.call(document, tagName, options);
        
        // Dodaj nonce na script i style elemente
        if (tagName.toLowerCase() === 'script' || tagName.toLowerCase() === 'style') {
            element.nonce = NONCE;
        }
        
        return element;
    };
    
    // 2. Override Element.prototype.setAttribute za hvatanje style i inline event handlera
    Element.prototype.setAttribute = function(name, value) {
        // Ako se postavlja inline style, možda će trebati nonce
        // Iako ovo neće pomoći za direktni style="..." atribut, može pomoći 
        // za neke frameworke koji koriste setAttribute
        if (name === 'style' && this.tagName !== 'STYLE') {
            // Umjesto inline stila, stvorimo style element
            const styleEl = document.createElement('style');
            styleEl.textContent = `#${this.id || 'dynamic-' + Math.random().toString(36).substr(2, 9)} { ${value} }`;
            document.head.appendChild(styleEl);
            
            // Ovdje možemo postaviti klasu ili ID ako je potrebno
            if (!this.id) {
                this.id = styleEl.textContent.match(/#([^\s{]+)/)[1];
            }
            
            return;
        }
        
        // Za event handlere prefixed s "on", trebalo bi koristiti addEventListener umjesto
        if (name.startsWith('on') && typeof value === 'string') {
            const eventName = name.slice(2);
            try {
                const handler = new Function('event', value);
                this.addEventListener(eventName, handler);
                return;
            } catch (e) {
                console.error('Failed to create handler for', name, e);
            }
        }
        
        return originalSetAttribute.call(this, name, value);
    };
    
    // 3. Override appendChild za hvatanje dinamički dodavanih script i style elemenata
    Node.prototype.appendChild = function(child) {
        if (child.tagName === 'SCRIPT' || child.tagName === 'STYLE') {
            child.nonce = NONCE;
        }
        
        return originalAppendChild.call(this, child);
    };
    
    // 4. Override innerHTML i outerHTML
    const originalInnerHTML = Object.getOwnPropertyDescriptor(Element.prototype, 'innerHTML');
    const originalOuterHTML = Object.getOwnPropertyDescriptor(Element.prototype, 'outerHTML');
    
    // Funkcija koja parsira HTML string i osigurava da svi script i style elementi imaju nonce
    function processHTML(html) {
        const template = document.createElement('template');
        template.innerHTML = html;
        
        const scripts = template.content.querySelectorAll('script, style');
        scripts.forEach(script => {
            script.nonce = NONCE;
        });
        
        return template.innerHTML;
    }
    
    // Override innerHTML
    Object.defineProperty(Element.prototype, 'innerHTML', {
        set: function(html) {
            if (typeof html === 'string' && (html.includes('<script') || html.includes('<style'))) {
                originalInnerHTML.set.call(this, processHTML(html));
                
                // Skripte dodane kroz innerHTML se ne izvršavaju automatski, moramo ih re-kreirati
                const scripts = this.querySelectorAll('script');
                scripts.forEach(oldScript => {
                    const newScript = document.createElement('script');
                    
                    // Kopiraj sve atribute
                    Array.from(oldScript.attributes).forEach(attr => {
                        if (attr.name !== 'nonce') { // nonce će već biti postavljen kroz createElement
                            newScript.setAttribute(attr.name, attr.value);
                        }
                    });
                    
                    // Kopiraj sadržaj
                    newScript.textContent = oldScript.textContent;
                    
                    // Zamijeni skriptu
                    oldScript.parentNode.replaceChild(newScript, oldScript);
                });
            } else {
                originalInnerHTML.set.call(this, html);
            }
        },
        get: function() {
            return originalInnerHTML.get.call(this);
        }
    });
    
    // Isto za outerHTML
    Object.defineProperty(Element.prototype, 'outerHTML', {
        set: function(html) {
            if (typeof html === 'string' && (html.includes('<script') || html.includes('<style'))) {
                const div = document.createElement('div');
                div.innerHTML = processHTML(html);
                const fragment = document.createDocumentFragment();
                while (div.firstChild) {
                    fragment.appendChild(div.firstChild);
                }
                this.parentNode.replaceChild(fragment, this);
            } else {
                originalOuterHTML.set.call(this, html);
            }
        },
        get: function() {
            return originalOuterHTML.get.call(this);
        }
    });
    
    // 5. Pomoćna funkcija za dodavanje inline stilova na Element
    HTMLElement.prototype.addStyles = function(stylesObject) {
        if (!this.id) {
            this.id = 'el-' + Math.random().toString(36).substr(2, 9);
        }
        
        const styleEl = document.createElement('style');
        let css = `#${this.id} {\n`;
        
        for (const prop in stylesObject) {
            const value = stylesObject[prop];
            const cssProp = prop.replace(/([A-Z])/g, '-$1').toLowerCase(); // camelCase to kebab-case
            css += `  ${cssProp}: ${value};\n`;
        }
        
        css += '}';
        styleEl.textContent = css;
        document.head.appendChild(styleEl);
        
        return this;
    };
    
    console.log('CSP Adapter loaded. Using nonce:', NONCE.slice(0, 8) + '...');
})();
//# sourceMappingURL=app_global-csp-adapter.59fd9648.js.map