/**
 * Theme Management Script
 * Centralna skripta za upravljanje temom i zajedničkim funkcionalnostima
 */

// IIFE za postavljanje početne teme iz lokalnog skladišta
(function() {
  const storedTheme = localStorage.getItem('krizan-theme-preference');
  if (storedTheme) {
    document.documentElement.setAttribute('data-theme', storedTheme);
  }
})();

// Glavne konstante za temu
const THEME = {
  KEY: 'krizan-theme-preference',
  DARK: 'dark',
  LIGHT: 'light'
};

// Globalni objekt za zajedničke funkcije
window.KrizanUI = window.KrizanUI || {};

/**
 * Dohvaća boje teme ovisno o trenutnoj temi
 * @returns {Object} Objekt s bojama za trenutnu temu
 */
window.KrizanUI.getThemeColors = function() {
  const theme = document.documentElement.getAttribute('data-theme') || THEME.DARK;
  
  if (theme === THEME.LIGHT) {
    return {
      backgroundColor: 'rgba(248, 249, 250, 0.05)',
      nodeColor: 'rgba(52, 152, 219, 0.9)',
      highlightColor: 'rgba(46, 204, 113, 1)',
      lineColor: 'rgba(52, 152, 219, 0.6)',
      accentPrimary: '#3498db',
      accentSecondary: '#2ecc71'
    };
  } else {
    return {
      backgroundColor: 'rgba(0, 0, 0, 0.05)',
      nodeColor: 'rgba(255, 107, 0, 0.9)',
      highlightColor: 'rgba(138, 43, 226, 1)',
      lineColor: 'rgba(255, 107, 0, 0.6)',
      accentPrimary: '#ff6b00',
      accentSecondary: '#8a2be2'
    };
  }
};

/**
 * Primjenjuje temu na modalne prozore
 * @param {HTMLElement} modal Element modalnog prozora
 */
window.KrizanUI.applyModalTheme = function(modal) {
  if (!modal) return;
  
  // Osnovni stilovi za modal
  modal.style.display = modal.style.display || 'none';
  modal.style.alignItems = 'center';
  modal.style.justifyContent = 'center';
  modal.style.transition = 'opacity 0.3s ease';
  
  // Stilovi za sadržaj modala
  const modalContent = modal.querySelector('.modal-content');
  if (modalContent) {
    modalContent.style.backgroundColor = 'var(--bg-secondary)';
    modalContent.style.color = 'var(--text-primary)';
    modalContent.style.borderColor = 'var(--border)';
    modalContent.style.boxShadow = '0 0.5rem 1.5rem var(--shadow)';
  }
  
  // Stilovi za naslov modala
  const modalHeading = modal.querySelector('h2');
  if (modalHeading) {
    modalHeading.style.color = 'var(--accent-primary)';
    modalHeading.style.borderBottom = '1px solid var(--border)';
  }
  
  // Stilovi za polja s greškama
  const errorFields = modal.querySelectorAll('.error-field');
  errorFields.forEach(field => {
    field.style.borderBottom = '1px solid var(--border)';
  });
  
  // Stilovi za nazive polja s greškama
  const errorFieldNames = modal.querySelectorAll('.error-field-name');
  errorFieldNames.forEach(name => {
    name.style.color = 'var(--accent-primary)';
  });
  
  // Stilovi za gumb za zatvaranje
  const closeBtn = modal.querySelector('.close');
  if (closeBtn) {
    closeBtn.style.color = 'var(--text-secondary)';
    
    // Ukloni postojeće listenere ako postoje (za slučaj ponovnog postavljanja)
    const newCloseBtn = closeBtn.cloneNode(true);
    closeBtn.parentNode.replaceChild(newCloseBtn, closeBtn);
    
    // Dodaj nove listenere
    newCloseBtn.addEventListener('mouseover', function() {
      this.style.color = 'var(--accent-primary)';
    });
    
    newCloseBtn.addEventListener('mouseout', function() {
      this.style.color = 'var(--text-secondary)';
    });
  }
};

/**
 * Primjenjuje temu na error modalne prozore
 * @param {HTMLElement} errorModal Element error modalnog prozora
 */
window.KrizanUI.applyErrorModalTheme = function(errorModal) {
  if (!errorModal) return;
  
  window.KrizanUI.applyModalTheme(errorModal);
  
  // Dodatno podešavanje za error modal
  window.showModal = function() {
    errorModal.style.display = 'flex';
    errorModal.style.opacity = '0';
    setTimeout(() => {
      errorModal.style.opacity = '1';
    }, 10);
  };
  
  window.closeModal = function() {
    errorModal.style.opacity = '0';
    setTimeout(() => {
      errorModal.style.display = 'none';
    }, 300);
  };
  
  // Dodaj event listener za zatvaranje klikom izvan sadržaja
  errorModal.addEventListener('click', function(e) {
    if (e.target === errorModal) {
      closeModal();
    }
  });
};

document.addEventListener('DOMContentLoaded', () => {
  // Elementi za gumb teme
  const themeButton = document.getElementById('theme-switch');
  const lightIcon = themeButton ? themeButton.querySelector('.light-icon') : null;
  const darkIcon = themeButton ? themeButton.querySelector('.dark-icon') : null;
  
  // Get stored theme preference or use system preference
  const getStoredTheme = () => {
    const storedTheme = localStorage.getItem(THEME.KEY);
    if (storedTheme) {
      return storedTheme;
    }
    
    // Check system preference as fallback
    return window.matchMedia('(prefers-color-scheme: dark)').matches 
      ? THEME.DARK 
      : THEME.LIGHT;
  };
  
  // Set theme in both DOM and localStorage
  const setTheme = (theme) => {
    document.documentElement.setAttribute('data-theme', theme);
    localStorage.setItem(THEME.KEY, theme);
    
    // Update button appearance
    updateThemeButton(theme);
    
    // Dispatch theme change event for other scripts
    document.dispatchEvent(new CustomEvent('themechange', { detail: { theme } }));
  };
  
  // Update theme button appearance
  const updateThemeButton = (theme) => {
    if (!themeButton) return;
    
    if (theme === THEME.DARK) {
      lightIcon.style.display = 'inline-block';
      darkIcon.style.display = 'none';
      themeButton.setAttribute('aria-label', 'Switch to light mode');
      themeButton.style.backgroundColor = window.KrizanUI.getThemeColors().accentPrimary;
    } else {
      lightIcon.style.display = 'none';
      darkIcon.style.display = 'inline-block';
      themeButton.setAttribute('aria-label', 'Switch to dark mode');
      themeButton.style.backgroundColor = window.KrizanUI.getThemeColors().accentPrimary;
    }
  };
  
  // Toggle between light and dark themes
  const toggleTheme = () => {
    const currentTheme = document.documentElement.getAttribute('data-theme');
    const newTheme = currentTheme === THEME.DARK ? THEME.LIGHT : THEME.DARK;
    
    // Apply theme change with animation
    document.documentElement.classList.add('theme-transition');
    setTheme(newTheme);
    
    // Remove transition class after animation completes
    setTimeout(() => {
      document.documentElement.classList.remove('theme-transition');
    }, 500);
  };
  
  // Initialize theme
  const initTheme = () => {
    // Set initial theme
    const initialTheme = getStoredTheme();
    setTheme(initialTheme);
    
    if (!themeButton) return;
    
    // Add theme toggle event listener
    themeButton.addEventListener('click', toggleTheme);
    
    // Add subtle animation to theme button
    themeButton.addEventListener('mouseenter', () => {
      themeButton.classList.add('pulse');
    });
    
    themeButton.addEventListener('mouseleave', () => {
      themeButton.classList.remove('pulse');
    });
    
    // Add system preference change listener
    window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
      const newTheme = e.matches ? THEME.DARK : THEME.LIGHT;
      setTheme(newTheme);
    });
    
    // Add transition styles dynamically
    const style = document.createElement('style');
    style.textContent = `
      .theme-transition,
      .theme-transition * {
        transition: background-color 0.3s ease, color 0.3s ease, border-color 0.3s ease, box-shadow 0.3s ease !important;
      }
      
      #theme-switch {
        overflow: hidden;
        position: relative;
      }
      
      #theme-switch::after {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        width: 5px;
        height: 5px;
        background: rgba(255, 255, 255, 0.5);
        opacity: 0;
        border-radius: 100%;
        transform: scale(1, 1) translate(-50%, -50%);
        transform-origin: 50% 50%;
      }
      
      #theme-switch:focus::after,
      #theme-switch.pulse::after {
        animation: pulse 1s ease-in-out infinite;
      }
      
      @keyframes pulse {
        0% {
          transform: scale(0, 0);
          opacity: 0.5;
        }
        100% {
          transform: scale(25, 25);
          opacity: 0;
        }
      }
    `;
    document.head.appendChild(style);
  };
  
  // Handle dropdown interactions when theme changes
  const setupDropdownInteractions = () => {
    // Handle mobile dropdowns with tap/click
    if (window.matchMedia('(max-width: 768px)').matches) {
      const dropdownTriggers = document.querySelectorAll('.dropdown-trigger');
      
      dropdownTriggers.forEach(trigger => {
        trigger.addEventListener('click', (e) => {
          e.preventDefault();
          const dropdown = trigger.parentElement;
          
          // Close all other dropdowns
          document.querySelectorAll('.dropdown.active').forEach(item => {
            if (item !== dropdown) {
              item.classList.remove('active');
            }
          });
          
          // Toggle this dropdown
          dropdown.classList.toggle('active');
        });
      });
      
      // Close dropdowns when clicking outside
      document.addEventListener('click', (e) => {
        if (!e.target.closest('.dropdown')) {
          document.querySelectorAll('.dropdown.active').forEach(dropdown => {
            dropdown.classList.remove('active');
          });
        }
      });
    }
  };
  
  // Initialize everything
  initTheme();
  setupDropdownInteractions();
  
  // Listen for viewport changes
  window.addEventListener('resize', setupDropdownInteractions);
});
//# sourceMappingURL=app_theme.2828740a.js.map