// Enhanced Neural Network Visualization
const canvas = document.getElementById("neurons");
const ctx = canvas ? canvas.getContext("2d") : null;

// Ensure canvas covers its container
function resizeCanvas() {
  if (!canvas) return;
  const container = canvas.parentElement;
  canvas.width = container.offsetWidth;
  canvas.height = container.offsetHeight;
}

// Initialize variables
let nodes = [];
let mouse = { x: null, y: null, radius: 120 };
const maxDistance = 150;
let nodeCount;
const fixedSpeed = 0.7;
let themeColors;
let animationFrame;

// Event listener for canvas resizing
if (canvas) {
  resizeCanvas();
  window.addEventListener('resize', resizeCanvas);
}

// Theme change listener
document.addEventListener('themechange', function() {
  themeColors = window.KrizanUI.getThemeColors();
});

// Calculate appropriate node count based on screen size
function determineNodeCount() {
  if (!canvas) return;
  const screenArea = canvas.width * canvas.height;
  nodeCount = Math.max(50, Math.floor(screenArea / 5000));
}

// Create node objects
function createNodes() {
  if (!canvas) return;
  nodes = [];
  
  for (let i = 0; i < nodeCount; i++) {
    const radius = Math.random() * 2 + 1;
    const x = Math.random() * canvas.width;
    const y = Math.random() * canvas.height;
    const speedX = fixedSpeed * (Math.random() - 0.5) * (radius / 2);
    const speedY = fixedSpeed * (Math.random() - 0.5) * (radius / 2);
    
    nodes.push({
      x,
      y,
      origX: x,
      origY: y,
      radius,
      speedX,
      speedY,
      depth: Math.random() * 2,
      color: themeColors.nodeColor,
      isHovered: false,
      uniqueAngle: Math.random() * Math.PI * 2, // For circular motion
      uniqueDistance: Math.random() * 10 + 10,   // For circular motion
      pulse: 0,
      pulseSpeed: Math.random() * 0.03 + 0.01
    });
  }
}

// Draw nodes and connections
function drawNodes() {
  if (!canvas || !ctx) return;
  
  // Clear canvas with background
  ctx.fillStyle = themeColors.backgroundColor;
  ctx.fillRect(0, 0, canvas.width, canvas.height);
  
  // Draw connections first (under nodes)
  connectNodes();
  
  // Draw each node
  nodes.forEach((node) => {
    // Calculate distance to mouse
    let distanceToMouse = Math.sqrt(
      (node.x - mouse.x) ** 2 + (node.y - mouse.y) ** 2
    );
    
    // Determine if node is under mouse influence
    const isUnderMouse = distanceToMouse < mouse.radius;
    
    // Set node color based on hover state
    if (isUnderMouse) {
      node.color = themeColors.highlightColor;
      node.isHovered = true;
      
      // Start a pulse effect when newly hovered
      if (!node.isHovered) {
        node.pulse = 1;
      }
    } else {
      let alpha = Math.max(0.2, 1 - distanceToMouse / maxDistance);
      node.color = themeColors.nodeColor.replace('0.9', alpha);
      node.isHovered = false;
    }
    
    // Apply depth effect
    let depthEffect = node.depth / 2 + 1;
    
    // Apply pulse effect
    let pulseRadius = node.radius;
    if (node.pulse > 0) {
      pulseRadius = node.radius * (1 + node.pulse);
      node.pulse -= 0.02;
    }
    
    // Draw the node
    ctx.fillStyle = node.color;
    ctx.beginPath();
    ctx.arc(node.x, node.y, pulseRadius * depthEffect, 0, Math.PI * 2);
    ctx.fill();
    
    // Draw glow effect for hovered nodes
    if (isUnderMouse) {
      ctx.shadowBlur = 15;
      ctx.shadowColor = themeColors.highlightColor;
      ctx.beginPath();
      ctx.arc(node.x, node.y, pulseRadius * depthEffect, 0, Math.PI * 2);
      ctx.fill();
      ctx.shadowBlur = 0;
    }
  });
}

// Draw connections between nearby nodes
function connectNodes() {
  if (!ctx) return;
  
  for (let i = 0; i < nodes.length; i++) {
    for (let j = i + 1; j < nodes.length; j++) {
      let dx = nodes[i].x - nodes[j].x;
      let dy = nodes[i].y - nodes[j].y;
      let distance = Math.sqrt(dx * dx + dy * dy);
      
      if (distance < maxDistance) {
        let alpha = 1 - distance / maxDistance;
        
        // Check if either node is hovered
        const isEitherHovered = nodes[i].isHovered || nodes[j].isHovered;
        
        // Determine line style
        ctx.strokeStyle = isEitherHovered 
          ? themeColors.highlightColor.replace('1', alpha) 
          : themeColors.lineColor.replace('0.6', alpha * 0.6);
        
        ctx.lineWidth = isEitherHovered ? 0.5 : 0.2;
        
        // Draw the connection
        ctx.beginPath();
        ctx.moveTo(nodes[i].x, nodes[i].y);
        ctx.lineTo(nodes[j].x, nodes[j].y);
        ctx.stroke();
      }
    }
  }
}

// Animate node movement
function animateNodes() {
  if (!canvas) return;
  
  nodes.forEach((node) => {
    // Regular movement
    node.x += node.speedX;
    node.y += node.speedY;
    
    // Bounce off edges
    if (node.x < 0 || node.x > canvas.width) {
      node.speedX *= -1;
      
      // Add slight position adjustment to avoid sticking to edges
      node.x = node.x < 0 ? 1 : canvas.width - 1;
    }
    
    if (node.y < 0 || node.y > canvas.height) {
      node.speedY *= -1;
      
      // Add slight position adjustment to avoid sticking to edges
      node.y = node.y < 0 ? 1 : canvas.height - 1;
    }
    
    // Mouse interaction - repel nodes
    if (mouse.x !== null && mouse.y !== null) {
      const dx = node.x - mouse.x;
      const dy = node.y - mouse.y;
      const distance = Math.sqrt(dx * dx + dy * dy);
      
      if (distance < mouse.radius) {
        const angle = Math.atan2(dy, dx);
        const force = (mouse.radius - distance) / mouse.radius;
        
        // Push node away from mouse
        node.x += Math.cos(angle) * force * 2;
        node.y += Math.sin(angle) * force * 2;
      } else if (node.isHovered) {
        // Gradually return to normal movement when no longer hovered
        node.isHovered = false;
      }
    }
  });
  
  drawNodes();
  animationFrame = requestAnimationFrame(animateNodes);
}

if (canvas) {
  // Mouse movement event handler
  canvas.addEventListener("mousemove", (event) => {
    const rect = canvas.getBoundingClientRect();
    mouse.x = event.clientX - rect.left;
    mouse.y = event.clientY - rect.top;
  });

  // Touch event handlers for mobile
  canvas.addEventListener("touchstart", (event) => {
    event.preventDefault();
    const rect = canvas.getBoundingClientRect();
    const touch = event.touches[0];
    mouse.x = touch.clientX - rect.left;
    mouse.y = touch.clientY - rect.top;
  });

  canvas.addEventListener("touchmove", (event) => {
    event.preventDefault();
    const rect = canvas.getBoundingClientRect();
    const touch = event.touches[0];
    mouse.x = touch.clientX - rect.left;
    mouse.y = touch.clientY - rect.top;
  });

  canvas.addEventListener("touchend", () => {
    mouse.x = null;
    mouse.y = null;
  });

  // Mouse leave event handler
  canvas.addEventListener("mouseleave", () => {
    mouse.x = null;
    mouse.y = null;
  });
}

// Window resize handler
window.addEventListener("resize", () => {
  resizeCanvas();
  determineNodeCount();
  createNodes();
});

// Page visibility change handling
document.addEventListener('visibilitychange', () => {
  if (!canvas) return;
  
  if (document.visibilityState === 'hidden') {
    // Pause animation when tab not visible
    cancelAnimationFrame(animationFrame);
  } else {
    // Resume animation when tab becomes visible again
    animationFrame = requestAnimationFrame(animateNodes);
  }
});

// Initialize and start animation
document.addEventListener('DOMContentLoaded', function() {
  if (!canvas) return;
  
  // Inicijaliziramo boje teme iz centralnog objekta
  themeColors = window.KrizanUI.getThemeColors();
  
  determineNodeCount();
  createNodes();
  animateNodes();
});
//# sourceMappingURL=home.050dbf64.js.map