/**
 * FCMHandler.js
 * Handles Firebase Cloud Messaging (FCM) operations for push notifications
 */
class FCMHandler {
    constructor() {
        // Get app_url from AppConfig
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // User data
        const userDataElement = document.getElementById('user-data');
        this.userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
        this.isLoggedIn = this.userData && !!this.userData.id;

        // Don't immediately determine if FCM is enabled
        this.fcmEnabled = null;  // Set to null initially to indicate "not determined yet"
        
        // Logger
        this.logger = this.createLogger();

        // Check if firebase features are already loaded
        if (window.firebaseFeatures) {
            this.fcmEnabled = window.firebaseFeatures.fcm_enabled || false;
            this.logger.log('Firebase features already available', {
                fcm_enabled: this.fcmEnabled
            });
            this.initializeBasedOnSettings();
        } else {
            // Wait for firebase to initialize before checking if FCM is enabled
            this.logger.log('Waiting for Firebase initialization...');
            
            // Listen for firebase initialization events
            document.addEventListener('firebase-ready', this.handleFirebaseReady.bind(this));
            document.addEventListener('firebaseInitialized', this.handleFirebaseReady.bind(this));
            
            // Set a timeout as a fallback
            setTimeout(() => {
                if (this.fcmEnabled === null) {
                    this.fcmEnabled = window.firebaseFeatures?.fcm_enabled || false;
                    this.logger.log('Checking FCM status after timeout', {
                        fcmEnabled: this.fcmEnabled
                    });
                    this.initializeBasedOnSettings();
                }
            }, 2000); // 2 second timeout
        }
    }

    /**
     * Initialize based on current settings
     */
    initializeBasedOnSettings() {
        // Skip initialization if user is not logged in or FCM is disabled
        if (!this.isLoggedIn || !this.fcmEnabled) {
            this.logger.log('User not logged in or FCM disabled, skipping Firebase Messaging initialization', {
                isLoggedIn: this.isLoggedIn,
                fcmEnabled: this.fcmEnabled
            });
            return;
        }

        // API Routes
        this.routes = {
            saveToken: `${this.publicUrl}/fcm/api/save-token`,
            getAuthToken: `${this.publicUrl}/fcm/api/get-token`
        };

        // Firebase messaging instance
        this.messaging = null;
        this.swRegistration = null;
        this.initialized = false;
        this.initAttempts = 0;

        // Initialize
        this.waitForFirebase();
    }

    /**
     * Create logger function with debug mode check
     * @returns {Object} Logger object with log, warn, error methods
     */
    createLogger() {
        return {
            log: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.log(`[FCMHandler] ${message}`, ...args);
                }
            },
            warn: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.warn(`[FCMHandler] ${message}`, ...args);
                }
            },
            error: (message, ...args) => {
                // Always log errors
                console.error(`[FCMHandler] ${message}`, ...args);
            }
        };
    }

    /**
     * Wait for Firebase to be initialized before proceeding
     */
    waitForFirebase() {
        // Check if Firebase is already initialized
        if (this.isFirebaseInitialized()) {
            this.logger.log('Firebase detected as already initialized');
            this.initializeMessaging();
            return;
        }

        // Listen for events
        document.addEventListener('firebase-ready', this.handleFirebaseReady.bind(this));
        document.addEventListener('firebaseInitialized', this.handleFirebaseReady.bind(this));
        
        // Timeout to check again
        setTimeout(() => {
            this.checkFirebaseAndInitialize();
        }, 1000);
    }

    /**
     * Check if Firebase is initialized and try again if needed
     */
    checkFirebaseAndInitialize() {
        if (this.initialized) return;
        
        this.initAttempts++;
        this.logger.log(`Checking Firebase initialization, attempt ${this.initAttempts}`);
        
        if (this.isFirebaseInitialized()) {
            this.logger.log('Firebase now initialized, proceeding with messaging setup');
            this.initializeMessaging();
            return;
        }
        
        // Give up after too many attempts
        if (this.initAttempts >= 5) {
            this.logger.error('Giving up on waiting for Firebase after 5 attempts');
            return;
        }
        
        // Try again after delay
        setTimeout(() => {
            this.checkFirebaseAndInitialize();
        }, 1000);
    }

    /**
     * Check if Firebase is properly initialized
     */
    isFirebaseInitialized() {
        return (
            typeof firebase !== 'undefined' && 
            firebase.apps && 
            firebase.apps.length > 0 && 
            window.firebaseInitialized === true
        );
    }

    /**
     * Handle Firebase ready event
     * @param {Event} event Firebase ready event
     */
    handleFirebaseReady(event) {
        this.logger.log('Firebase ready event received', event.detail);
        
        // Only proceed if not already initialized or if we're still waiting to determine FCM status
        if (this.fcmEnabled !== null && this.initialized) {
            return;
        }
        
        if (event.detail?.success) {
            // Update FCM enabled status if we haven't determined it yet
            if (this.fcmEnabled === null) {
                this.fcmEnabled = window.firebaseFeatures?.fcm_enabled || false;
                this.logger.log('Firebase is ready, FCM enabled:', this.fcmEnabled);
                
                // Initialize based on settings
                this.initializeBasedOnSettings();
            } else {
                // If we've already determined FCM status and it's enabled, proceed with Firebase initialization
                setTimeout(() => {
                    this.initializeMessaging();
                }, 300);
            }
        }
    }

    /**
     * Initialize Firebase Cloud Messaging after Firebase is ready
     */
    async initializeMessaging() {
        if (this.initialized || !this.isLoggedIn) {
            return;
        }
    
        try {
            this.logger.log('Initializing Firebase Cloud Messaging');
    
            // Double check Firebase is initialized
            if (!this.isFirebaseInitialized()) {
                this.logger.error('Firebase not initialized yet, cannot initialize messaging');
                return;
            }
    
            // Check if messaging is available
            if (typeof firebase.messaging !== 'function') {
                this.logger.error('Firebase messaging not available');
                return;
            }
            
            // Get messaging instance
            this.messaging = firebase.messaging();
            
            // Register service worker
            this.swRegistration = await this.registerServiceWorker();
            if (!this.swRegistration) {
                this.logger.error('Service Worker registration failed');
                return;
            }
    
            // Send configuration to service worker when ready
            if (this.swRegistration.active) {
                this.sendConfigToServiceWorker(this.swRegistration);
            } else if (this.swRegistration.installing) {
                this.swRegistration.installing.addEventListener('statechange', (event) => {
                    if (event.target.state === 'activated') {
                        this.sendConfigToServiceWorker(this.swRegistration);
                    }
                });
            }
    
            // Authenticate with Firebase using custom token
            const authSuccess = await this.authenticateWithFirebase();
            if (!authSuccess) {
                this.logger.error('Failed to authenticate with Firebase');
                return;
            }
    
            // Request permission if needed (but only after auth is successful)
            if (Notification.permission !== 'granted') {
                const permissionGranted = await this.requestNotificationPermission();
                if (!permissionGranted) {
                    this.logger.warn('Notification permission not granted');
                    return;
                }
            }
    
            // Get and save FCM token
            await this.getAndSaveToken();
    
            // Setup foreground message handling
            this.setupForegroundMessaging();
    
            this.initialized = true;
            
            // Broadcast that Firebase messaging is ready
            document.dispatchEvent(new CustomEvent('firebase-messaging-ready', { 
                detail: { success: true } 
            }));
            
            this.logger.log('Firebase Cloud Messaging initialized successfully');
    
        } catch (error) {
            this.logger.error('Error initializing Firebase Cloud Messaging', error);
        }
    }

    /**
     * Authenticate with Firebase using custom token from server
     * @returns {Promise<boolean>} Whether authentication was successful
     */
    async authenticateWithFirebase() {
        try {
            this.logger.log('Authenticating with Firebase using custom token');
            
            const token = await this.getFirebaseAuthToken();
            
            if (!token) {
                throw new Error('Failed to get Firebase auth token');
            }
            
            this.logger.log('Got token, signing in with custom token...');
            
            // Sign out first to ensure clean state
            if (firebase.auth().currentUser) {
                await firebase.auth().signOut();
            }
            
            // Sign in with custom token
            const userCredential = await firebase.auth().signInWithCustomToken(token);
            
            this.logger.log('Firebase authentication successful', {
                uid: userCredential.user.uid,
                claims: userCredential.user.getIdTokenResult ? 
                       (await userCredential.user.getIdTokenResult()).claims : 'unknown'
            });
            
            // Dispatch an event so other components know authentication is complete
            document.dispatchEvent(new CustomEvent('firebase-auth-success', { 
                detail: { 
                    uid: userCredential.user.uid
                } 
            }));
            
            return true;
        } catch (error) {
            this.logger.error('Firebase authentication error:', error);
            return false;
        }
    }
    
    /**
     * Get Firebase authentication token from server with caching
     * @returns {Promise<string|null>} Firebase auth token or null on failure
     */
    async getFirebaseAuthToken() {
        try {
            // Check cache first
            const cachedToken = localStorage.getItem('firebase_auth_token');
            const cachedExpiry = localStorage.getItem('firebase_auth_token_expiry');
            const now = Date.now();
            
            // Use cached token if it exists and is not expired
            // Set expiry to 50 minutes to ensure token is refreshed before it expires (usually 1 hour)
            if (cachedToken && cachedExpiry && now < parseInt(cachedExpiry, 10)) {
                this.logger.log('Using cached Firebase auth token');
                return cachedToken;
            }
            
            this.logger.log('Getting Firebase auth token from API');
            
            const response = await this.fetchWithCsrf(this.routes.getAuthToken, {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.success && data.token) {
                // Calculate expiry (50 minutes from now)
                const expiryTime = now + (50 * 60 * 1000);
                
                // Cache the token with expiry
                localStorage.setItem('firebase_auth_token', data.token);
                localStorage.setItem('firebase_auth_token_expiry', expiryTime.toString());
                
                this.logger.log('Firebase auth token received and cached');
                return data.token;
            } else {
                this.logger.error('Invalid token response', data);
                return null;
            }
        } catch (error) {
            this.logger.error('Error fetching Firebase auth token:', error);
            return null;
        }
    }

    /**
     * Register the Firebase service worker
     * @returns {Promise<ServiceWorkerRegistration|null>} Service worker registration or null on failure
     */
    async registerServiceWorker() {
        if (!('serviceWorker' in navigator)) {
            this.logger.error('Service Worker not supported in this browser');
            return null;
        }

        try {
            const swPath = `${this.publicUrl}/firebase-messaging-sw.js`;
            this.logger.log('Registering service worker at:', swPath);

            // Check if service worker is already registered
            const existingRegistrations = await navigator.serviceWorker.getRegistrations();
            for (const reg of existingRegistrations) {
                if (reg.active && reg.active.scriptURL.includes('firebase-messaging-sw.js')) {
                    this.logger.log('Service worker already registered, using existing');
                    return reg;
                }
            }

            // Register new service worker
            return await navigator.serviceWorker.register(swPath, { 
                scope: `${this.publicUrl}/` 
            });
        } catch (error) {
            this.logger.error('Service Worker registration failed:', error);
            return null;
        }
    }

    /**
     * Send configuration to the service worker
     * @param {ServiceWorkerRegistration} registration Service worker registration
     */
    sendConfigToServiceWorker(registration) {
        if (!registration?.active) return;

        // Send Firebase config
        if (firebase.app().options) {
            registration.active.postMessage({
                type: 'FIREBASE_CONFIG',
                config: firebase.app().options
            });
        }

        // Send notification routes
        registration.active.postMessage({
            type: 'NOTIFICATION_ROUTES',
            routes: {
                'general': '',
                'task': 'tasks/list',
                'message': 'notification/home'
            }
        });

        // Send base path
        registration.active.postMessage({
            type: 'BASE_PATH',
            publicUrl: this.publicUrl
        });

        // Send debug setting
        registration.active.postMessage({
            type: 'DEBUG_SETTING',
            debugEnabled: window.firebaseDebugMode || false
        });

        this.logger.log('Configuration sent to service worker');
    }

    /**
     * Request notification permission
     * @returns {Promise<boolean>} Whether permission was granted
     */
    async requestNotificationPermission() {
        try {
            // Handle browser differences
            if (typeof Notification.requestPermission === 'function') {
                if (Notification.requestPermission.length === 0) {
                    // Modern browsers (promise)
                    return await Notification.requestPermission() === 'granted';
                } else {
                    // Older browsers (callback)
                    return new Promise(resolve => {
                        Notification.requestPermission(permission => {
                            resolve(permission === 'granted');
                        });
                    });
                }
            }
            return false;
        } catch (error) {
            this.logger.error('Error requesting notification permission:', error);
            return false;
        }
    }

    /**
     * Get and save FCM token with caching
     * @returns {Promise<string|null>} FCM token or null on failure
     */
    async getAndSaveToken() {
        try {
            // Skip for iOS (not supported)
            if (/iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream) {
                this.logger.log('Skipping FCM token on iOS device');
                return null;
            }

            // Get Firebase configuration
            const vapidKey = firebase.app().options.vapidKey;
            if (!vapidKey) {
                this.logger.error('VAPID key missing in Firebase configuration');
                return null;
            }

            // Get token
            const currentToken = await this.messaging.getToken({
                vapidKey: vapidKey,
                serviceWorkerRegistration: this.swRegistration
            });

            if (!currentToken) {
                this.logger.error('Failed to get FCM token');
                return null;
            }

            this.logger.log('FCM token obtained:', currentToken.substring(0, 10) + '...');

            // Detect device type
            const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
            const isSmallScreen = window.screen?.width < 768;
            const deviceType = (isMobile || isSmallScreen) ? 'mobile' : 'desktop';

            // Check if token needs to be saved
            const savedToken = localStorage.getItem('firebase_fcm_token');
            const lastSaveTime = parseInt(localStorage.getItem('firebase_fcm_token_time') || '0', 10);
            const now = Date.now();
            const ONE_DAY = 24 * 60 * 60 * 1000;
            
            // Only update if token changed or last save was more than a day ago
            if (savedToken !== currentToken || (now - lastSaveTime) > ONE_DAY) {
                // Check if we're currently in a throttled state for token updates
                const lastTokenUpdateAttempt = parseInt(localStorage.getItem('firebase_fcm_token_update_attempt') || '0', 10);
                const TOKEN_UPDATE_THROTTLE = 5 * 60 * 1000; // 5 minutes throttle
                
                if (now - lastTokenUpdateAttempt < TOKEN_UPDATE_THROTTLE) {
                    this.logger.log('Token update throttled, will try later');
                    return currentToken;
                }
                
                // Mark update attempt
                localStorage.setItem('firebase_fcm_token_update_attempt', now.toString());
                
                // Save token to server
                const saved = await this.saveToken(currentToken, deviceType);
                
                if (saved) {
                    localStorage.setItem('firebase_fcm_token', currentToken);
                    localStorage.setItem('firebase_fcm_token_time', now.toString());
                    this.logger.log('Token saved successfully');
                } else {
                    this.logger.error('Failed to save token to server');
                }
            } else {
                this.logger.log('Token already saved recently, skipping server update');
            }

            return currentToken;
        } catch (error) {
            this.logger.error('Error getting or saving Firebase token:', error);
            return null;
        }
    }

    /**
     * Save token to server
     * @param {string} token FCM token
     * @param {string} deviceType Device type (mobile/desktop)
     * @returns {Promise<boolean>} Success status
     */
    async saveToken(token, deviceType) {
        try {
            if (!token || !this.userData || !this.userData.id) {
                return false;
            }

            const response = await this.fetchWithCsrf(this.routes.saveToken, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify({
                    token: token,
                    userId: this.userData.id,
                    email: this.userData.user_email,
                    deviceType: deviceType
                })
            });

            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }

            const result = await response.json();
            return result.success;
        } catch (error) {
            this.logger.error('Error saving token to server:', error);
            return false;
        }
    }

    /**
     * Setup handling for foreground messages
     */
    setupForegroundMessaging() {
        if (!this.messaging) return;

        this.messaging.onMessage((payload) => {
            this.logger.log('Foreground message received:', payload);

            // Extract notification data
            const notification = payload.notification || {};
            const data = payload.data || {};

            // Create notification object
            const notificationData = {
                id: data.notificationId || Date.now().toString(),
                title: notification.title || 'New notification',
                message: notification.body || '',
                timestamp: Date.now(),
                type: data.type || 'general',
                related_id: data.relatedId || null,
                route: data.route || '',
                is_read: false,
                is_sticky: false
            };

            // Dispatch event for notification handlers
            document.dispatchEvent(new CustomEvent('newNotificationReceived', {
                detail: notificationData
            }));

            // Show browser notification if page is not in focus
            if (document.visibilityState !== 'visible') {
                this.showBrowserNotification(notification.title, notification.body, data);
            }
        });
    }

    /**
     * Show browser notification
     * @param {string} title Notification title
     * @param {string} body Notification body
     * @param {Object} data Additional data
     */
    showBrowserNotification(title, body, data = {}) {
        if (Notification.permission !== 'granted') return;

        try {
            // Create notification
            const notification = new Notification(title || 'New notification', {
                body: body || '',
                icon: window.getNotificationIconByType(data.type || 'general', this.publicUrl),
                tag: data.type || 'general',
                requireInteraction: true
            });

            // Handle click
            notification.onclick = () => {
                window.focus();
                
                // Navigate to route if provided
                if (data.route) {
                    window.location.href = `${this.publicUrl}/${data.route}`;
                } else {
                    window.location.href = `${this.publicUrl}/notification/home`;
                }
                
                notification.close();
            };
        } catch (error) {
            this.logger.error('Error showing browser notification:', error);
        }
    }

    /**
     * Fetch with CSRF token
     * @param {string} url URL to fetch
     * @param {Object} options Fetch options
     * @returns {Promise<Response>} Fetch response
     */
    fetchWithCsrf(url, options = {}) {
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
        
        // Add CSRF token to headers
        const headers = {
            ...options.headers,
            'X-CSRF-TOKEN': csrfToken
        };
        
        // Return fetch with updated options
        return fetch(url, {
            ...options,
            headers,
            credentials: 'same-origin'
        });
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.fcmHandler = new FCMHandler();
});
//# sourceMappingURL=FCMHandler.43ed39e8.js.map