/**
 * FirebaseInitializer.js
 * Initializes Firebase and its services
 */
(function() {
    // Debug mode setting - can be hardcoded or set by server
    window.firebaseDebugMode = false;
    
    // Logger for debugging
    const logger = {
        log: function(message, ...args) {
            if (window.firebaseDebugMode) {
                console.log(`[FirebaseInitializer] ${message}`, ...args);
            }
        },
        warn: function(message, ...args) {
            if (window.firebaseDebugMode) {
                console.warn(`[FirebaseInitializer] ${message}`, ...args);
            }
        },
        error: function(message, ...args) {
            console.error(`[FirebaseInitializer] ${message}`, ...args);
        }
    };
    
    // Check if user is logged in
    const userDataElement = document.getElementById('user-data');
    const userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
    const isLoggedIn = userData && !!userData.id;
    
    // Skip initialization if user is not logged in
    if (!isLoggedIn) {
        logger.log('User not logged in, skipping Firebase initialization');
        return;
    }
    
    // Status
    let isInitializing = false;
    
    // Initialize Firebase immediately
    initializeFirebase();
    
    /**
     * Initialize Firebase
     */
    async function initializeFirebase() {
        if (window.firebaseDebugMode) {
            console.log("========= FIREBASE INITIALIZER DEBUG =========");
            console.log("Starting Firebase initialization");
            console.log("User data element exists:", !!userDataElement);
            console.log("User data:", userDataElement ? JSON.parse(userDataElement.textContent) : null);
            console.log("Is logged in:", isLoggedIn);
            console.log("Debug mode:", window.firebaseDebugMode);
            console.log("============================================");
        }
        
        // Check if we're already initializing
        if (isInitializing) {
            logger.log('Firebase initialization already in progress');
            return;
        }
        
        // Mark that initialization is in progress
        isInitializing = true;
        
        // Check if Firebase is already initialized
        if (firebase.apps && firebase.apps.length > 0) {
            logger.log('Firebase already initialized, skipping initialization');
            dispatchFirebaseReadyEvents();
            isInitializing = false;
            return;
        }
        
        try {
            // Fetch configuration
            const config = await fetchFirebaseConfig();
            
            if (!config) {
                logger.error('Failed to fetch Firebase config, cannot initialize Firebase');
                isInitializing = false;
                return;
            }
            
            // Initialize Firebase
            logger.log('Initializing Firebase with config', config);
            
            firebase.initializeApp(config);
            
            // Wait a bit for Firebase APIs to load
            await new Promise(resolve => setTimeout(resolve, 100));
            
            // Check if initialization was successful
            if (firebase.apps.length === 0) {
                logger.error('Firebase initialization failed');
                isInitializing = false;
                return;
            }
            
            logger.log('Firebase successfully initialized');
            
            // Dispatch events that Firebase is ready
            dispatchFirebaseReadyEvents();
        } catch (error) {
            logger.error('Error during Firebase initialization:', error);
        } finally {
            isInitializing = false;
        }
    }
    
    /**
     * Fetch Firebase configuration from API with caching
     */
    async function fetchFirebaseConfig() {
        try {
            // Check if we have cached config and it's not expired
            const cachedConfig = localStorage.getItem('firebase_config');
            const cachedTimestamp = localStorage.getItem('firebase_config_timestamp');
            const currentTime = Date.now();
            const CACHE_EXPIRY = 3600000; // 1 hour in milliseconds
            
            // Use cached config if it exists and is not expired
            if (cachedConfig && cachedTimestamp && 
                (currentTime - parseInt(cachedTimestamp, 10) < CACHE_EXPIRY)) {
                logger.log('Using cached Firebase configuration');
                const data = JSON.parse(cachedConfig);
                
                // Save config globally
                window.firebaseConfig = data.config;
                window.firebaseFeatures = data.features || {
                    fcm_enabled: false,
                    realtime_db_enabled: false
                };
                
                return data.config;
            }
            
            // If cache is missing or expired, fetch from API
            logger.log('Fetching Firebase configuration from API');
            
            // Get CSRF token if it exists
            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
            
            // Get base path from app_config
            const appConfig = document.getElementById('app-config');
            const basePath = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';
            
            const response = await fetch(`${basePath}/fcm/api/get-config`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': csrfToken
                },
                credentials: 'same-origin'
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.success && data.config) {
                logger.log('Firebase configuration received successfully');
                
                // Save configuration globally
                window.firebaseConfig = data.config;
                
                // Save feature flags
                window.firebaseFeatures = data.features || {
                    fcm_enabled: false,
                    realtime_db_enabled: false
                };
                
                // Cache the configuration
                localStorage.setItem('firebase_config', JSON.stringify(data));
                localStorage.setItem('firebase_config_timestamp', currentTime.toString());
                
                return data.config;
            } else {
                logger.error('Invalid configuration response', data);
                return null;
            }    
        } catch (error) {
            logger.error('Error fetching Firebase configuration:', error);
            return null;
        }
    }
    
    /**
     * Dispatch events that Firebase is ready
     */
    function dispatchFirebaseReadyEvents() {
        // New standardized event
        document.dispatchEvent(new CustomEvent('firebase-ready', { 
            detail: { 
                success: true,
                hasMessaging: typeof firebase.messaging !== 'undefined',
                hasDatabase: typeof firebase.database !== 'undefined'
            } 
        }));
        
        // For compatibility with old code
        window.firebaseInitialized = true;
        document.dispatchEvent(new CustomEvent('firebaseInitialized', { 
            detail: { success: true } 
        }));
        
        logger.log('Firebase ready events dispatched');
    }
})();
//# sourceMappingURL=FirebaseInitializer.fed9c4f6.js.map