/**
 * NotificationBadgeHandler.js
 * Handles the notification badge in the header
 */
class NotificationBadgeHandler {
    constructor() {
        // Get app_url from AppConfig
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // User data
        const userDataElement = document.getElementById('user-data');
        this.userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
        this.isLoggedIn = this.userData && !!this.userData.id;

        // Skip initialization if user is not logged in
        if (!this.isLoggedIn) {
            console.log('User not logged in, skipping notification badge handler initialization');
            return;
        }

        // API Routes
        this.routes = {
            fetchNotifications: `${this.publicUrl}/notification/api/get-notifications`
        };

        // DOM Elements
        this.notificationCount = document.getElementById('notification-count');
        this.notificationBell = document.getElementById('notification-bell');

        // State
        this.state = {
            unreadCount: 0,
            lastUpdate: 0
        };

        // Initialize
        this.init();
    }

    /**
     * Initialize the notification handler
     */
    init() {
        // Check if notification badge exists
        if (!this.notificationCount) {
            console.warn('Notification count element not found. Make sure you have the correct ID.');
            return;
        }

        // Set up event listeners
        this.setupEventListeners();

        // Initial fetch
        this.updateCount();

        // Set up periodic updates
        setInterval(() => this.updateCount(), 60000); // Every minute
    }

    /**
     * Set up event listeners for notifications
     */
    setupEventListeners() {
        // Listen for when a notification is marked as read
        document.addEventListener('notificationMarkedAsRead', () => {
            this.updateCount();
        });

        // Listen for when all notifications are marked as read
        document.addEventListener('allNotificationsMarkedAsRead', () => {
            this.updateCount();
        });

        // Handle click on the notification bell
        if (this.notificationBell) {
            this.notificationBell.addEventListener('click', () => {
                // Navigate to notifications page
                window.location.href = `${this.publicUrl}/notification/home`;
            });
        }
    }

    /**
     * Update notification count from API with caching
     */
    updateCount() {
        // Throttle updates to prevent too many requests
        const now = Date.now();
        if (now - this.state.lastUpdate < 5000) { // 5 seconds throttle
            return;
        }
        this.state.lastUpdate = now;

        // Check cache first
        const cachedCount = localStorage.getItem('notification_unread_count');
        const cachedTimestamp = localStorage.getItem('notification_unread_count_timestamp');
        const CACHE_EXPIRY = 30000; // 30 seconds expiry for notification count
        
        if (cachedCount && cachedTimestamp && (now - parseInt(cachedTimestamp, 10) < CACHE_EXPIRY)) {
            // Use cached count
            this.updateBadge(parseInt(cachedCount, 10));
            return;
        }

        // Fetch unread notifications count
        this.fetchWithCsrf(`${this.routes.fetchNotifications}?limit=100&filter=unread`, {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.notifications) {
                // Count unread notifications
                const unreadCount = data.notifications.length;
                
                // Cache the count
                localStorage.setItem('notification_unread_count', unreadCount.toString());
                localStorage.setItem('notification_unread_count_timestamp', now.toString());
                
                this.updateBadge(unreadCount);
            }
        })
        .catch(error => {
            console.error('Error fetching notification count:', error);
        });
    }

    /**
     * Update the notification badge in the UI
     * @param {number} count Unread notification count
     */
    updateBadge(count) {
        this.state.unreadCount = count;

        // Update UI
        if (this.notificationCount) {
            this.notificationCount.textContent = count.toString();
            this.notificationCount.style.display = count > 0 ? 'flex' : 'none';
        }

        // Update page title if there are unread notifications
        const originalTitle = document.title.replace(/^\(\d+\) /, '');
        document.title = count > 0 ? `(${count}) ${originalTitle}` : originalTitle;
    }

    /**
     * Fetch with CSRF token
     * @param {string} url URL to fetch
     * @param {Object} options Fetch options
     * @returns {Promise<Response>} Fetch response
     */
    fetchWithCsrf(url, options = {}) {
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
        
        // Add CSRF token to headers
        const headers = {
            ...options.headers,
            'X-CSRF-TOKEN': csrfToken
        };
        
        // Return fetch with updated options
        return fetch(url, {
            ...options,
            headers,
            credentials: 'same-origin'
        });
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.notificationBadgeHandler = new NotificationBadgeHandler();
});
//# sourceMappingURL=NotificationBadgeHandler.5064936b.js.map