/**
 * PushNotificationHandler.js
 * Handles incoming push notifications and shows notifications to users
 */
class PushNotificationHandler {
    constructor() {
        // Get app_url from AppConfig
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // User data
        const userDataElement = document.getElementById('user-data');
        this.userData = userDataElement ? JSON.parse(userDataElement.textContent) : null;
        this.isLoggedIn = this.userData && !!this.userData.id;

        // Create logger
        this.logger = this.createLogger();

        // Skip initialization if user is not logged in
        if (!this.isLoggedIn) {
            this.logger.log('User not logged in, skipping push notification handler initialization');
            return;
        }

        // Notification settings
        this.notificationQueue = [];
        this.notificationDisplayDelay = 500; // ms between notifications
        this.isShowingNotification = false;

        // Initialize
        this.init();
    }

    /**
     * Create logger function with debug mode check
     * @returns {Object} Logger object with log, warn, error methods
     */
    createLogger() {
        return {
            log: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.log(`[PushNotificationHandler] ${message}`, ...args);
                }
            },
            warn: (message, ...args) => {
                if (window.firebaseDebugMode) {
                    console.warn(`[PushNotificationHandler] ${message}`, ...args);
                }
            },
            error: (message, ...args) => {
                // Always log errors
                console.error(`[PushNotificationHandler] ${message}`, ...args);
            }
        };
    }

    /**
     * Initialize the notification handler
     */
    init() {
        // Listen for new notification events
        document.addEventListener('newNotificationReceived', this.handleNewNotification.bind(this));
        
        // Listen for Firebase messaging ready event
        document.addEventListener('firebase-messaging-ready', () => {
            this.logger.log('Firebase Messaging ready, push notification handler initialized');
        });
    }

    /**
     * Handle a new notification
     * @param {Event} event New notification event
     */
    handleNewNotification(event) {
        const notification = event.detail;
        
        this.logger.log('New notification received:', notification);
        
        // Add to queue
        this.notificationQueue.push(notification);
        
        // Process queue if not already processing
        if (!this.isShowingNotification) {
            this.processNotificationQueue();
        }
    }

    /**
     * Process the notification queue
     */
    async processNotificationQueue() {
        if (this.notificationQueue.length === 0) {
            this.isShowingNotification = false;
            return;
        }
        
        this.isShowingNotification = true;
        
        // Get the next notification
        const notification = this.notificationQueue.shift();
        
        this.logger.log('Processing notification from queue:', notification.id);
        
        // Show notification
        await this.showNotification(notification);
        
        // Wait before showing next notification
        setTimeout(() => {
            this.processNotificationQueue();
        }, this.notificationDisplayDelay);
    }

    /**
     * Show a notification to the user
     * @param {Object} notification Notification data
     * @returns {Promise<void>}
     */
    async showNotification(notification) {
        // Use toastr if available
        if (typeof toastr !== 'undefined') {
            this.logger.log('Showing notification with toastr');
            return new Promise(resolve => {
                toastr.info(notification.message, notification.title, {
                    timeOut: 5000,
                    extendedTimeOut: 2000,
                    onclick: () => {
                        this.navigateToNotification(notification);
                    },
                    onHidden: () => {
                        resolve();
                    }
                });
            });
        }
        
        // Fallback to browser notification if toastr not available
        if (Notification.permission === 'granted') {
            this.logger.log('Showing browser notification');
            return new Promise(resolve => {
                try {
                    const browserNotification = new Notification(notification.title || 'New notification', {
                        body: notification.message || '',
                        icon: window.getNotificationIconByType(notification.type || 'general', this.publicUrl),
                        tag: notification.type || 'general',
                        requireInteraction: false
                    });
                    
                    browserNotification.onclick = () => {
                        this.navigateToNotification(notification);
                        browserNotification.close();
                    };
                    
                    // Auto-close after 5 seconds
                    setTimeout(() => {
                        browserNotification.close();
                        resolve();
                    }, 5000);
                } catch (error) {
                    this.logger.error('Error showing browser notification:', error);
                    resolve();
                }
            });
        }
        
        this.logger.log('No notification method available');
        // No notification method available, resolve immediately
        return Promise.resolve();
    }

    /**
     * Navigate to the notification route
     * @param {Object} notification Notification data
     */
    navigateToNotification(notification) {
        // Navigate to route if provided
        if (notification.route) {
            this.logger.log('Navigating to route:', notification.route);
            window.location.href = `${this.publicUrl}/${notification.route}`;
        } else {
            this.logger.log('Navigating to notification home');
            window.location.href = `${this.publicUrl}/notification/home`;
        }
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.pushNotificationHandler = new PushNotificationHandler();
});
//# sourceMappingURL=PushNotificationHandler.77a8fd45.js.map