/**
 * NotificationUIHandler.js
 * Handles the UI interactions on the notification page
 */
class NotificationUIHandler {
    constructor() {
        // Get app_url from AppConfig for API routes
        const appConfig = document.getElementById('app-config');
        this.publicUrl = appConfig ? JSON.parse(appConfig.textContent).paths.app_url : '';

        // API Routes
        this.routes = {
            fetchNotifications: `${this.publicUrl}/notification/api/get-notifications`,
            markAsRead: `${this.publicUrl}/notification/api/mark-as-read`,
            toggleSticky: `${this.publicUrl}/notification/api/toggle-sticky`
        };

        // State for notification handling
        this.state = {
            currentFilter: 'unread',  // 'unread', 'sticky', 'read'
            currentLimit: 10,
            notifications: [],
            hasMoreNotifications: true
        };

        // DOM elements
        this.elements = {
            notificationsList: document.getElementById('notifications-list'),
            messageStatus: document.getElementById('message_status'),
            loadMore: document.getElementById('load-more'),
            markAllReadBtn: document.getElementById('mark-all-read')
        };

        // Initialize
        this.init();
    }

    /**
     * Initialize notification page
     */
    init() {
        // Check if all required elements exist
        if (!this.elements.notificationsList) {
            console.error('Required DOM elements not found. Make sure you have the correct IDs.');
            return;
        }

        this.setupEventListeners();
        this.loadNotifications();
    }

    /**
     * Set up event listeners for the page
     */
    setupEventListeners() {
        // Filter menu items
        const unreadMenuItem = document.getElementById('unreadMenuItem');
        const stickyMenuItem = document.getElementById('stickyMenuItem');
        const readMenuItem = document.getElementById('readMenuItem');
        const allMenuItem = document.getElementById('allMenuItem'); 

        if (unreadMenuItem) {
            unreadMenuItem.addEventListener('click', () => this.handleFilterClick('unread'));
        }
        if (stickyMenuItem) {
            stickyMenuItem.addEventListener('click', () => this.handleFilterClick('sticky'));
        }
        if (readMenuItem) {
            readMenuItem.addEventListener('click', () => this.handleFilterClick('read'));
        }
        if (allMenuItem) {
            allMenuItem.addEventListener('click', () => this.handleFilterClick('all'));
        }

        // Load more button
        if (this.elements.loadMore) {
            this.elements.loadMore.addEventListener('click', () => {
                this.state.currentLimit += 10;
                this.loadNotifications(false);
            });
        }

        // Mark all as read button
        if (this.elements.markAllReadBtn) {
            this.elements.markAllReadBtn.addEventListener('click', () => {
                this.markAllAsRead();
            });
        }

        // Delegation for notification actions
        document.addEventListener('click', (event) => {
            // Mark as read button click
            if (event.target.closest('.mark-read-btn')) {
                event.preventDefault();
                const button = event.target.closest('.mark-read-btn');
                const notificationId = button.dataset.id;
                this.markNotificationAsRead(notificationId);
            }
            
            // Toggle sticky button click
            if (event.target.closest('.toggle-sticky-btn')) {
                event.preventDefault();
                const button = event.target.closest('.toggle-sticky-btn');
                const notificationId = button.dataset.id;
                this.toggleStickyNotification(notificationId);
            }
        });
    }

    /**
     * Load notifications from API
     * @param {boolean} resetList Whether to reset the list or append to it
     */
    loadNotifications(resetList = true) {
        if (this.elements.messageStatus) {
            this.elements.messageStatus.innerHTML = '';
        }
        
        // Show loading indicator
        const loadingIndicator = document.createElement('div');
        loadingIndicator.className = 'text-center my-3';
        loadingIndicator.innerHTML = '<div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div>';
        if (resetList) {
            this.elements.notificationsList.innerHTML = '';
            this.elements.notificationsList.appendChild(loadingIndicator);
        }
        
        // Prepare request - Map UI filters to API filters
        let apiFilter = 'default';
        switch (this.state.currentFilter) {
            case 'unread':
                apiFilter = 'unread';
                break;
            case 'sticky':
                apiFilter = 'sticky';
                break;
            case 'read':
                apiFilter = 'read';
                break;
            case 'all':
                apiFilter = 'all';
                break;
        }
        
        const params = new URLSearchParams({
            limit: this.state.currentLimit,
            filter: apiFilter
        });
        
        // Fetch notifications
        this.fetchWithCsrf(`${this.routes.fetchNotifications}?${params.toString()}`, {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            // Remove loading indicator
            loadingIndicator.remove();
            
            if (data.success && data.notifications) {
                // Update the state
                this.state.notifications = data.notifications;
                
                // Render notifications
                this.renderNotifications(this.state.notifications, resetList);
                
                // Update "load more" button visibility
                if (data.notifications.length < this.state.currentLimit) {
                    if (this.elements.loadMore) {
                        this.elements.loadMore.style.display = 'none';
                    }
                    this.state.hasMoreNotifications = false;
                } else {
                    if (this.elements.loadMore) {
                        this.elements.loadMore.style.display = 'block';
                    }
                    this.state.hasMoreNotifications = true;
                }
            } else {
                this.elements.notificationsList.innerHTML = '<div class="text-center p-3">No notifications</div>';
                if (this.elements.loadMore) {
                    this.elements.loadMore.style.display = 'none';
                }
            }
        })
        .catch(error => {
            console.error('Error loading notifications:', error);
            loadingIndicator.remove();
            if (this.elements.messageStatus) {
                this.elements.messageStatus.innerHTML = 
                    '<div class="alert alert-warning">Error loading notifications. Please try again.</div>';
            }
        });
    }

    /**
     * Render notifications in the container
     * @param {Array} notifications Array of notification objects
     * @param {boolean} resetList Whether to reset the list or append to it
     */
    renderNotifications(notifications, resetList = true) {
        // If no notifications, show message
        if (!notifications || notifications.length === 0) {
            this.elements.notificationsList.innerHTML = '<div class="text-center p-3">No notifications</div>';
            if (this.elements.loadMore) {
                this.elements.loadMore.style.display = 'none';
            }
            return;
        }
        
        // Clear container if we're resetting the list
        if (resetList) {
            this.elements.notificationsList.innerHTML = '';
        }
        
        // Use notifications as they come from API - already sorted
        notifications.forEach(notification => {
            const notificationElement = this.createNotificationElement(notification);
            this.elements.notificationsList.appendChild(notificationElement);
        });
    }

    /**
     * Create HTML element for a notification
     * @param {Object} notification Notification data
     * @returns {HTMLElement} Notification element
     */
    createNotificationElement(notification) {
        // Format date
        const createdDate = new Date(notification.created_at);
        const formattedDate = this.formatDate(createdDate);
        
        // Create notification container
        const notificationEl = document.createElement('div');
        notificationEl.className = 'tweet';
        notificationEl.dataset.id = notification.id;
        
        // Add classes based on status
        if (!notification.is_read) notificationEl.classList.add('unread');
        if (notification.is_sticky) notificationEl.classList.add('sticky');
        
        // Create image element
        const imgEl = document.createElement('img');
        imgEl.className = 'tweet__author-logo';
        imgEl.src = notification.from_id ? 
            window.getNotificationIconByType(notification.type || 'general', this.publicUrl) : 
            `../default/images/user.png`;
        
        // Create main content container
        const mainEl = document.createElement('div');
        mainEl.className = 'tweet__main';
        
        // Create header
        const headerEl = document.createElement('div');
        headerEl.className = 'tweet__header';
        
        const titleEl = document.createElement('div');
        titleEl.className = 'tweet__author-name';
        titleEl.textContent = notification.title || 'Notification';
        
        const authorEl = document.createElement('div');
        authorEl.className = 'tweet__author-slug';
        authorEl.textContent = notification.sender_email || 'system';
        
        const dateEl = document.createElement('div');
        dateEl.className = 'tweet__publish-time';
        dateEl.textContent = formattedDate;
        
        headerEl.appendChild(titleEl);
        headerEl.appendChild(authorEl);
        headerEl.appendChild(dateEl);
        
        // Create content
        const contentEl = document.createElement('div');
        contentEl.className = 'tweet__content';
        contentEl.innerHTML = notification.message || '';
        
        // Create footer with actions
        const footerEl = document.createElement('div');
        footerEl.className = 'tweet__footer';
        
        // Mark as read button (only for unread notifications)
        if (!notification.is_read) {
            const markReadBtn = document.createElement('span');
            markReadBtn.className = 'mark-read-btn';
            markReadBtn.dataset.id = notification.id;
            markReadBtn.innerHTML = '<i class="fas fa-check"></i> Mark as read';
            footerEl.appendChild(markReadBtn);
        }
        
        // Toggle sticky button
        const toggleStickyBtn = document.createElement('span');
        toggleStickyBtn.className = 'toggle-sticky-btn';
        toggleStickyBtn.dataset.id = notification.id;
        
        if (notification.is_sticky) {
            toggleStickyBtn.innerHTML = '<i class="fas fa-thumbtack text-primary"></i> Remove mark';
        } else {
            toggleStickyBtn.innerHTML = '<i class="fas fa-thumbtack"></i> Mark as important';
        }
        footerEl.appendChild(toggleStickyBtn);
        
        // Assemble notification
        mainEl.appendChild(headerEl);
        mainEl.appendChild(contentEl);
        mainEl.appendChild(footerEl);
        
        notificationEl.appendChild(imgEl);
        notificationEl.appendChild(mainEl);
        
        return notificationEl;
    }

    /**
     * Handle filter menu item click
     * @param {string} filter Filter type
     */
    handleFilterClick(filter) {
        // Only if filter changed
        if (this.state.currentFilter !== filter) {
            // Update state
            this.state.currentFilter = filter;
            
            // Update active class in menu
            document.querySelectorAll('.sidebar-menu__item').forEach(item => {
                item.classList.remove('sidebar-menu__item--active');
            });
            
            const activeMenuItem = document.getElementById(`${filter}MenuItem`);
            if (activeMenuItem) {
                activeMenuItem.classList.add('sidebar-menu__item--active');
            }
            
            // Load notifications with new filter
            this.loadNotifications(true);
        }
    }

    /**
     * Mark a notification as read
     * @param {string} notificationId Notification ID
     */
    markNotificationAsRead(notificationId) {
        const notification = this.state.notifications.find(n => n.id == notificationId);
        if (!notification || notification.is_read) return;
        
        this.fetchWithCsrf(this.routes.markAsRead, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify({ notification_id: notificationId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update notification in state
                notification.is_read = true;
                
                // Update UI
                const notificationEl = this.elements.notificationsList.querySelector(`.tweet[data-id="${notificationId}"]`);
                if (notificationEl) {
                    notificationEl.classList.remove('unread');
                    const markReadBtn = notificationEl.querySelector('.mark-read-btn');
                    if (markReadBtn) markReadBtn.remove();
                }
                
                // If in unread filter, this notification should no longer be visible
                if (this.state.currentFilter === 'unread') {
                    this.loadNotifications(true);
                }
                
                // Show success message
                if (this.elements.messageStatus) {
                    this.elements.messageStatus.innerHTML = 
                        '<div class="alert alert-success">Notification has been marked as read</div>';
                    
                    // Auto-hide message after 3 seconds
                    setTimeout(() => {
                        if (this.elements.messageStatus) {
                            this.elements.messageStatus.innerHTML = '';
                        }
                    }, 3000);
                }
                
                // Trigger event to update badge
                document.dispatchEvent(new CustomEvent('notificationMarkedAsRead', {
                    detail: { id: notificationId }
                }));
            }
        })
        .catch(error => {
            console.error('Error marking notification as read:', error);
            if (this.elements.messageStatus) {
                this.elements.messageStatus.innerHTML = 
                    '<div class="alert alert-danger">Error marking notification. Please try again.</div>';
            }
        });
    }

    /**
     * Mark all notifications as read
     */
    markAllAsRead() {
        this.fetchWithCsrf(this.routes.markAsRead, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify({}) // Empty object to mark all as read
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update notifications in state
                this.state.notifications.forEach(notification => {
                    if (!notification.is_sticky) {
                        notification.is_read = true;
                    }
                });
                
                // Update UI - reload notifications with current filter
                this.loadNotifications(true);
                
                // Show success message
                if (this.elements.messageStatus) {
                    this.elements.messageStatus.innerHTML = 
                        '<div class="alert alert-success">All notifications have been marked as read</div>';
                    
                    // Auto-hide message after 3 seconds
                    setTimeout(() => {
                        if (this.elements.messageStatus) {
                            this.elements.messageStatus.innerHTML = '';
                        }
                    }, 3000);
                }
                
                // Trigger event to update badge
                document.dispatchEvent(new CustomEvent('allNotificationsMarkedAsRead'));
            }
        })
        .catch(error => {
            console.error('Error marking all notifications as read:', error);
            if (this.elements.messageStatus) {
                this.elements.messageStatus.innerHTML = 
                    '<div class="alert alert-danger">Error marking notifications. Please try again.</div>';
            }
        });
    }

    /**
     * Toggle sticky status for a notification
     * @param {string} notificationId Notification ID
     */
    toggleStickyNotification(notificationId) {
        const notification = this.state.notifications.find(n => n.id == notificationId);
        if (!notification) return;
        
        this.fetchWithCsrf(this.routes.toggleSticky, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify({ notification_id: notificationId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update notification in state
                notification.is_sticky = !notification.is_sticky;
                
                // Update UI - reload notifications to ensure proper sorting
                this.loadNotifications(true);
                
                // Show success message
                if (this.elements.messageStatus) {
                    const message = notification.is_sticky ? 
                        'Notification has been marked as important' : 
                        'Notification is no longer marked as important';
                    
                    this.elements.messageStatus.innerHTML = 
                        `<div class="alert alert-success">${message}</div>`;
                    
                    // Auto-hide message after 3 seconds
                    setTimeout(() => {
                        if (this.elements.messageStatus) {
                            this.elements.messageStatus.innerHTML = '';
                        }
                    }, 3000);
                }
            }
        })
        .catch(error => {
            console.error('Error toggling sticky status:', error);
            if (this.elements.messageStatus) {
                this.elements.messageStatus.innerHTML = 
                    '<div class="alert alert-danger">Error updating notification. Please try again.</div>';
            }
        });
    }

    /**
     * Format date for display
     * @param {Date} date Date to format
     * @returns {string} Formatted date string
     */
    formatDate(date) {
        if (!date || !(date instanceof Date) || isNaN(date)) return '';
        
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        
        return `${day}.${month}.${year}, ${hours}:${minutes}`;
    }

    /**
     * Fetch with CSRF token
     * @param {string} url URL to fetch
     * @param {Object} options Fetch options
     * @returns {Promise<Response>} Fetch response
     */
    fetchWithCsrf(url, options = {}) {
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
        
        // Add CSRF token to headers
        const headers = {
            ...options.headers,
            'X-CSRF-TOKEN': csrfToken
        };
        
        // Return fetch with updated options
        return fetch(url, {
            ...options,
            headers,
            credentials: 'same-origin'
        });
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', () => {
    window.notificationUIHandler = new NotificationUIHandler();
});
//# sourceMappingURL=notification_home.0f652b46.js.map