/**
 * User Permissions JavaScript
 * Handles UI enhancements and theme integration
 */
document.addEventListener('DOMContentLoaded', function() {
  
  // Elements
  const form = document.querySelector('form');
  const permissionCheckboxes = document.querySelectorAll('input[name="permissions[]"]');
  const passwordToggleButtons = document.querySelectorAll('.password-toggle');
  
  // Initialize permission selection visual enhancements
  initPermissionCheckboxes();
  
  // Initialize password toggle functionality
  initPasswordToggles();
  
  // Add loading state to form submission
  if (form) {
    form.addEventListener('submit', function() {
      const submitButton = this.querySelector('button[type="submit"]');
      if (submitButton) {
        submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
        submitButton.disabled = true;
      }
    });
  }
  
  /**
   * Initialize permission checkboxes with visual enhancements
   */
  function initPermissionCheckboxes() {
    if (!permissionCheckboxes.length) return;
    
    // Add visual feedback for permission checkboxes
    permissionCheckboxes.forEach(checkbox => {
      const parentLabel = checkbox.closest('.permission-checkbox');
      
      if (parentLabel) {
        // Initial state
        updateCheckboxState(checkbox);
        
        // Listen for changes
        checkbox.addEventListener('change', function() {
          updateCheckboxState(this);
        });
      }
    });
  }
  
  /**
   * Initialize password toggle functionality
   */
  function initPasswordToggles() {
    passwordToggleButtons.forEach(button => {
      button.addEventListener('click', function() {
        const targetId = this.getAttribute('data-target');
        if (targetId) {
          togglePassword(targetId);
        }
      });
    });
  }
  
  /**
   * Update visual state of permission checkbox
   */
  function updateCheckboxState(checkbox) {
    const parentLabel = checkbox.closest('.permission-checkbox');
    
    if (checkbox.checked) {
      parentLabel.style.backgroundColor = 'rgba(var(--accent-primary-rgb), 0.1)';
      parentLabel.style.borderLeft = '3px solid var(--accent-primary)';
    } else {
      parentLabel.style.backgroundColor = '';
      parentLabel.style.borderLeft = '';
    }
  }
  
  /**
   * Toggle password visibility
   */
  function togglePassword(inputId) {
    const input = document.getElementById(inputId);
    if (!input) return;
    
    const button = document.querySelector(`[data-target="${inputId}"]`);
    const icon = button?.querySelector('i');
    
    if (input.type === 'password') {
      input.type = 'text';
      if (icon) {
        icon.classList.remove('fa-eye');
        icon.classList.add('fa-eye-slash');
      }
    } else {
      input.type = 'password';
      if (icon) {
        icon.classList.remove('fa-eye-slash');
        icon.classList.add('fa-eye');
      }
    }
  }
});
//# sourceMappingURL=userpermissions.56e2951a.js.map