/**
 * Firebase Messaging Service Worker
 * Handles background push notifications and notification clicks
 */

// Configuration with default values
// The actual publicUrl will be updated via messages from the main thread
const CONFIG = {
  publicUrl: '/',  // Default value, will be updated via BASE_PATH message
  defaultIcon: '/staticfirebase/notification-icon.png',
  defaultBadge: '/staticfirebase/notification-badge.png',
  defaultTitle: 'Nova obavijest'
};

// Debug setting (set via messages)
let DEBUG_ENABLED = false;

// Logger function that uses DEBUG_ENABLED
const logger = {
  log: function(message, ...args) {
    if (DEBUG_ENABLED) {
      console.log(`[firebase-messaging-sw.js] ${message}`, ...args);
    }
  },
  warn: function(message, ...args) {
    if (DEBUG_ENABLED) {
      console.warn(`[firebase-messaging-sw.js] ${message}`, ...args);
    }
  },
  error: function(message, ...args) {
    // Errors are always displayed
    console.error(`[firebase-messaging-sw.js] ${message}`, ...args);
  }
};

// Default notification routes
let NOTIFICATION_ROUTES = {
  'general': '',
  'task_assigned': 'tasks/list',
  'message': 'notification/home'
};

// Import Firebase scripts from CDN
importScripts('https://cdnjs.cloudflare.com/ajax/libs/firebase/9.6.1/firebase-app-compat.min.js');
importScripts('https://cdnjs.cloudflare.com/ajax/libs/firebase/9.6.1/firebase-messaging-compat.min.js');

/**
 * Initialize Firebase with placeholder config
 * This is required to register event handlers properly
 */
function initializeFirebase() {
  // Basic configuration that will be updated via messages from main thread
  firebase.initializeApp({
    apiKey: "placeholder",
    projectId: "placeholder",
    messagingSenderId: "placeholder",
    appId: "placeholder",
    vapidKey: "placeholder"
  });
  
  // Get messaging instance
  const messaging = firebase.messaging();
  
  // Register background message handler
  registerBackgroundMessageHandler(messaging);
}

/**
 * Register handler for background messages
 * @param {Object} messaging Firebase messaging instance
 */
function registerBackgroundMessageHandler(messaging) {
  messaging.onBackgroundMessage((payload) => {
    logger.log('Background message received:', payload);
    
    const notification = payload.notification || {};
    const data = payload.data || {};
    
    // Create notification options
    const notificationOptions = createNotificationOptions(notification, data);
    
    // Show notification
    return self.registration.showNotification(
      notification.title || CONFIG.defaultTitle, 
      notificationOptions
    );
  });
}

/**
 * Create notification options object
 * @param {Object} notification Notification data
 * @param {Object} data Additional data
 * @returns {Object} Notification options
 */
function createNotificationOptions(notification, data) {
  // Add support for realtime database notifications
  const notificationData = {
    ...data,
    url: getNotificationUrl(data)
  };
  
  return {
    body: notification.body || '',
    icon: `${CONFIG.publicUrl}${CONFIG.defaultIcon}`,
    badge: `${CONFIG.publicUrl}${CONFIG.defaultBadge}`,
    vibrate: [200, 100, 200],
    tag: data.type || 'general',
    data: notificationData,
    actions: [{ action: 'view', title: 'View' }],
    requireInteraction: true
  };
}

/**
 * Determine notification URL based on type and data
 * @param {Object} data Notification data
 * @returns {string} URL
 */
function getNotificationUrl(data) {
  // Get base URL
  const baseUrl = CONFIG.publicUrl;
  
  // Determine route
  let route = '';
  
  // Use provided route if available
  if (data.route) {
    route = '/' + data.route.replace(/^\//, '');
  } 
  // Otherwise use route from mapping
  else if (data.type && NOTIFICATION_ROUTES[data.type]) {
    route = '/' + NOTIFICATION_ROUTES[data.type].replace(/^\//, '');
    
    // Add ID if needed
    if (data.related_id && 
        (data.type === 'task_assigned' || 
         data.type === 'message')) {
      route += `/${data.related_id}`;
    }
  }
  
  // Combine base URL and route
  return `${baseUrl}${route}`;
}

/**
 * Handle push events (for direct push messages)
 * @param {PushEvent} event Push event
 */
function handlePushEvent(event) {
  logger.log('Push event received:', event);
  
  if (!event.data) {
    logger.log('No data in push event');
    return;
  }
  
  try {
    logger.log('Parsing push data...');
    // Try to parse as JSON first
    const data = event.data.json();
    logger.log('Parsed push data:', data);
    
    const notification = data.notification || {};
    const notificationOptions = createNotificationOptions(
      notification, 
      data.data || {}
    );
    
    event.waitUntil(
      self.registration.showNotification(
        notification.title || CONFIG.defaultTitle, 
        notificationOptions
      )
    );
  } catch (error) {
    logger.error('Error processing push data as JSON:', error);
    
    // If parsing as JSON fails, try as plain text
    try {
      const text = event.data.text();
      logger.log('Handling as text:', text);
      
      event.waitUntil(
        self.registration.showNotification(CONFIG.defaultTitle, {
          body: text,
          icon: `${CONFIG.publicUrl}${CONFIG.defaultIcon}`,
          badge: `${CONFIG.publicUrl}${CONFIG.defaultBadge}`,
          vibrate: [200, 100, 200],
          tag: 'general',
          requireInteraction: true
        })
      );
    } catch (textError) {
      logger.error('Error processing push data:', textError);
    }
  }
}

/**
 * Handle notification clicks
 * @param {NotificationEvent} event Notification event
 */
function handleNotificationClick(event) {
  event.notification.close();
  
  // Get notification data
  const data = event.notification.data || {};
  
  // Determine target route - use data.url if available
  const targetRoute = data.url || CONFIG.publicUrl;
  
  logger.log('Notification clicked, navigating to:', targetRoute);
  
  // Focus existing window or open new one
  event.waitUntil(
    self.clients.matchAll({type: 'window'}).then((clientList) => {
      for (const client of clientList) {
        if (client.url.includes(CONFIG.publicUrl) && 'focus' in client) {
          client.focus();
          return client.navigate(targetRoute);
        }
      }
      return self.clients.openWindow(targetRoute);
    })
  );
}

/**
 * Handle messages from the main thread
 * @param {MessageEvent} event Message event
 */
function handleMessage(event) {
  logger.log('Message received:', event.data?.type);
  
  if (event.data?.type === 'FIREBASE_CONFIG') {
    logger.log('Received Firebase config');
    // We don't need to reinitialize Firebase since we've already done that
  }
  
  if (event.data?.type === 'NOTIFICATION_ROUTES') {
    logger.log('Received notification routes');
    NOTIFICATION_ROUTES = {...NOTIFICATION_ROUTES, ...event.data.routes};
  }
  
  if (event.data?.type === 'BASE_PATH') {
    logger.log('Received base path update:', event.data.publicUrl);
    
    // Update publicUrl in config
    CONFIG.publicUrl = event.data.publicUrl || CONFIG.publicUrl;
    
    // Make sure the URL doesn't end with a slash
    if (CONFIG.publicUrl.endsWith('/')) {
      CONFIG.publicUrl = CONFIG.publicUrl.slice(0, -1);
    }
    
    logger.log('Updated publicUrl to:', CONFIG.publicUrl);
  }
  
  // Process DEBUG_SETTING message
  if (event.data?.type === 'DEBUG_SETTING') {
    // Only log if debug was previously enabled or we're enabling it now
    if (DEBUG_ENABLED || event.data.debugEnabled) {
      console.log(`[firebase-messaging-sw.js] Received debug setting: ${event.data.debugEnabled}`);
    }
    DEBUG_ENABLED = event.data.debugEnabled;
  }
}

// Initialize Firebase
initializeFirebase();

// Register event listeners
self.addEventListener('push', handlePushEvent);
self.addEventListener('notificationclick', handleNotificationClick);
self.addEventListener('message', handleMessage);

// Service worker lifecycle events
self.addEventListener('install', (event) => {
  logger.log('Service Worker installing');
  self.skipWaiting(); // Activate immediately
});

self.addEventListener('activate', (event) => {
  logger.log('Service Worker activating');
  event.waitUntil(self.clients.claim()); // Take control of all clients
});