// Enhanced financijeadmincharts.js - KOMPLETNI FAJL SA STATUSOM NAPLATE
document.addEventListener('DOMContentLoaded', function() {
  // Dohvati podatke iz data atributa
  const statsElement = document.getElementById('finance-stats-data');
  
  if (!statsElement) {
    console.error('Finance stats data element not found!');
    return;
  }
  
  try {
    // Dohvati podatke
    const companyData = JSON.parse(statsElement.getAttribute('data-naplata-company') || '[]');
    const statusData = JSON.parse(statsElement.getAttribute('data-naplata-status') || '[]');
    const expenseTypeData = JSON.parse(statsElement.getAttribute('data-rashodi-type') || '[]');
    const combinedMonthlyData = JSON.parse(statsElement.getAttribute('data-combined-monthly') || '[]');
    const profitabilityData = JSON.parse(statsElement.getAttribute('data-profitability') || '[]');
    
    console.log('Chart data loaded:', {
      companies: companyData.length,
      statuses: statusData.length,
      expenses: expenseTypeData.length,
      monthly: combinedMonthlyData.length,
      profitability: profitabilityData.length
    });
    
    // Kreiraj grafove
    createCombinedMonthlyChart(combinedMonthlyData);
    createCompanyChart(companyData);
    createExpenseTypeChart(expenseTypeData);
    createNaplataStatusChart(statusData);
    
  } catch (error) {
    console.error('Error initializing finance charts:', error);
  }
});

/**
 * Create combined monthly chart - naplata vs rashodi
 */
function createCombinedMonthlyChart(monthlyData) {
  const ctx = document.getElementById('combinedMonthlyChart');
  if (!ctx) {
    console.error('Combined monthly chart element not found');
    return;
  }
  
  if (!monthlyData || monthlyData.length === 0) {
    console.warn('No monthly data available for combined chart');
    return;
  }
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  
  // Sort data by year and month
  const sortedData = monthlyData.sort((a, b) => {
    if (a.year !== b.year) return a.year - b.year;
    return a.month - b.month;
  });
  
  const labels = sortedData.map(item => 
    `${monthNames[item.month - 1]} ${item.year}`
  );
  
  const naplataValues = sortedData.map(item => item.naplata || 0);
  const rashodiValues = sortedData.map(item => item.rashodi || 0);
  const profitValues = sortedData.map(item => (item.naplata || 0) - (item.rashodi || 0));
  
  new Chart(ctx.getContext('2d'), {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Naplata (€)',
          data: naplataValues,
          backgroundColor: 'rgba(76, 175, 80, 0.1)',
          borderColor: '#4caf50',
          borderWidth: 3,
          fill: false,
          tension: 0.4,
          pointRadius: 6,
          pointHoverRadius: 8,
          pointBackgroundColor: '#4caf50',
          pointBorderColor: '#ffffff',
          pointBorderWidth: 2,
          yAxisID: 'y'
        },
        {
          label: 'Rashodi (€)',
          data: rashodiValues,
          backgroundColor: 'rgba(244, 67, 54, 0.1)',
          borderColor: '#f44336',
          borderWidth: 3,
          fill: false,
          tension: 0.4,
          pointRadius: 6,
          pointHoverRadius: 8,
          pointBackgroundColor: '#f44336',
          pointBorderColor: '#ffffff',
          pointBorderWidth: 2,
          yAxisID: 'y'
        },
        {
          label: 'Dobit (€)',
          data: profitValues,
          backgroundColor: profitValues.map(val => val >= 0 ? 'rgba(33, 150, 243, 0.1)' : 'rgba(156, 39, 176, 0.1)'),
          borderColor: profitValues.map(val => val >= 0 ? '#2196f3' : '#9c27b0'),
          borderWidth: 2,
          fill: false,
          tension: 0.4,
          pointRadius: 5,
          pointHoverRadius: 7,
          pointBackgroundColor: profitValues.map(val => val >= 0 ? '#2196f3' : '#9c27b0'),
          pointBorderColor: '#ffffff',
          pointBorderWidth: 2,
          yAxisID: 'y1'
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'index',
        intersect: false,
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            usePointStyle: true,
            font: {
              size: 12,
              weight: 'bold'
            }
          }
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.9)',
          titleColor: '#fff',
          bodyColor: '#fff',
          borderColor: '#333',
          borderWidth: 1,
          callbacks: {
            afterLabel: function(context) {
              const dataIndex = context.dataIndex;
              const naplata = naplataValues[dataIndex];
              const rashodi = rashodiValues[dataIndex];
              const profit = naplata - rashodi;
              const margin = naplata > 0 ? ((profit / naplata) * 100).toFixed(1) : '0.0';
              
              return [
                `Dobit: ${profit.toLocaleString('hr-HR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} €`,
                `Marža: ${margin}%`
              ];
            }
          }
        }
      },
      scales: {
        y: {
          type: 'linear',
          display: true,
          position: 'left',
          beginAtZero: true,
          grid: {
            color: 'rgba(255, 255, 255, 0.1)'
          },
          ticks: {
            color: '#666',
            callback: function(value) {
              return value.toLocaleString('hr-HR') + ' €';
            }
          },
          title: {
            display: true,
            text: 'Naplata / Rashodi (€)',
            color: '#666'
          }
        },
        y1: {
          type: 'linear',
          display: true,
          position: 'right',
          grid: {
            drawOnChartArea: false,
          },
          ticks: {
            color: '#666',
            callback: function(value) {
              return value.toLocaleString('hr-HR') + ' €';
            }
          },
          title: {
            display: true,
            text: 'Dobit (€)',
            color: '#666'
          }
        },
        x: {
          grid: {
            display: false
          },
          ticks: {
            color: '#666',
            maxRotation: 45
          }
        }
      }
    }
  });
}

/**
 * Create the company amounts chart (naplata)
 */
function createCompanyChart(companyData) {
  const ctx = document.getElementById('byCompanyChart');
  if (!ctx) {
    console.error('Company chart element not found');
    return;
  }
  
  if (!companyData || companyData.length === 0) {
    console.warn('No company data available');
    return;
  }
  
  const companyLabels = companyData.map(item => {
    const company = item.company || 'Nepoznato';
    return company.length > 25 ? company.substring(0, 22) + '...' : company;
  });
  const companyValues = companyData.map(item => parseFloat(item.amount) || 0);
  
  const companyColors = [
    '#4caf50', '#2196f3', '#ff9800', '#9c27b0', 
    '#f44336', '#00bcd4', '#795548', '#607d8b',
    '#e91e63', '#3f51b5', '#ff5722', '#8bc34a'
  ];
  
  new Chart(ctx.getContext('2d'), {
    type: 'bar',
    data: {
      labels: companyLabels,
      datasets: [{
        label: 'Naplata (€)',
        data: companyValues,
        backgroundColor: companyColors.slice(0, companyLabels.length),
        borderColor: companyColors.slice(0, companyLabels.length),
        borderWidth: 2,
        borderRadius: 8,
        borderSkipped: false
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          display: false
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.9)',
          titleColor: '#fff',
          bodyColor: '#fff',
          borderColor: '#333',
          borderWidth: 1,
          callbacks: {
            title: function(context) {
              // Show full company name in tooltip
              const originalIndex = context[0].dataIndex;
              return companyData[originalIndex].company || 'Nepoznato';
            },
            label: function(context) {
              return `Naplata: ${parseFloat(context.raw).toLocaleString('hr-HR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
              })} €`;
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true,
          grid: {
            color: 'rgba(255, 255, 255, 0.1)'
          },
          ticks: {
            color: '#666',
            callback: function(value) {
              return value.toLocaleString('hr-HR') + ' €';
            }
          }
        },
        x: {
          grid: {
            display: false
          },
          ticks: {
            color: '#666',
            maxRotation: 45,
            font: {
              size: 10
            }
          }
        }
      }
    }
  });
}

/**
 * Create expense type distribution chart
 */
function createExpenseTypeChart(expenseTypeData) {
  const ctx = document.getElementById('byExpenseTypeChart');
  if (!ctx) {
    console.error('Expense type chart element not found');
    return;
  }
  
  if (!expenseTypeData || expenseTypeData.length === 0) {
    console.warn('No expense type data available');
    return;
  }
  
  const typeLabels = expenseTypeData.map(item => {
    const label = item.type || 'Nepoznato';
    return label.length > 20 ? label.substring(0, 17) + '...' : label;
  });
  const typeValues = expenseTypeData.map(item => parseFloat(item.amount) || 0);
  
  const expenseColors = [
    '#f44336', '#e91e63', '#9c27b0', '#673ab7',
    '#3f51b5', '#2196f3', '#03a9f4', '#00bcd4',
    '#009688', '#4caf50', '#8bc34a', '#cddc39',
    '#ffeb3b', '#ffc107', '#ff9800', '#ff5722'
  ];
  
  new Chart(ctx.getContext('2d'), {
    type: 'doughnut',
    data: {
      labels: typeLabels,
      datasets: [{
        data: typeValues,
        backgroundColor: expenseColors.slice(0, typeLabels.length),
        borderColor: '#ffffff',
        borderWidth: 2,
        hoverBorderWidth: 3,
        hoverOffset: 5
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '50%',
      plugins: {
        legend: {
          position: 'right',
          labels: {
            boxWidth: 12,
            font: {
              size: 10
            },
            color: '#666',
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
                  return {
                    text: `${label} (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.9)',
          titleColor: '#fff',
          bodyColor: '#fff',
          borderColor: '#333',
          borderWidth: 1,
          callbacks: {
            title: function(context) {
              // Show full expense type name in tooltip
              const originalIndex = context[0].dataIndex;
              return expenseTypeData[originalIndex].type || 'Nepoznato';
            },
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
              return `${parseFloat(value).toLocaleString('hr-HR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
              })} € (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create naplata status chart - NOVO!
 */
function createNaplataStatusChart(statusData) {
  const ctx = document.getElementById('naplatStatusChart');
  if (!ctx) {
    console.error('Naplata status chart element not found');
    return;
  }
  
  if (!statusData || statusData.length === 0) {
    console.warn('No status data available');
    return;
  }
  
  const statusLabels = statusData.map(item => item.Status || 'Nepoznato');
  const statusValues = statusData.map(item => parseFloat(item.amount) || 0);
  const statusCounts = statusData.map(item => parseInt(item.count) || 0);
  
  // Definiraj boje na temelju statusa
  const statusColors = statusLabels.map(status => {
    switch(status.toLowerCase()) {
      case 'naplaćeno u cijelosti':
        return '#4caf50'; // Zelena
      case 'plaćeno djelomično':
        return '#ff9800'; // Narančasta
      case 'čeka plaćanje':
        return '#2196f3'; // Plava
      case 'kasni s plaćanjem':
        return '#f44336'; // Crvena
      default:
        return '#9e9e9e'; // Siva
    }
  });
  
  new Chart(ctx.getContext('2d'), {
    type: 'pie',
    data: {
      labels: statusLabels,
      datasets: [{
        data: statusValues,
        backgroundColor: statusColors,
        borderColor: '#ffffff',
        borderWidth: 2,
        hoverBorderWidth: 3,
        hoverOffset: 8
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          position: 'bottom',
          labels: {
            boxWidth: 12,
            font: {
              size: 10
            },
            color: '#666',
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const count = statusCounts[i];
                  const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
                  return {
                    text: `${label} (${count} stavki)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.9)',
          titleColor: '#fff',
          bodyColor: '#fff',
          borderColor: '#333',
          borderWidth: 1,
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
              const count = statusCounts[context.dataIndex];
              const status = context.label;
              
              return [
                `${status}:`,
                `Iznos: ${parseFloat(value).toLocaleString('hr-HR', {
                  minimumFractionDigits: 2,
                  maximumFractionDigits: 2
                })} €`,
                `Postotak: ${percentage}%`,
                `Broj stavki: ${count}`
              ];
            }
          }
        }
      }
    }
  });
}

/**
 * Helper function to format period display (backup if needed)
 */
function formatPeriodDisplay(month, year) {
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  const monthIndex = parseInt(month) - 1;
  const monthName = monthNames[monthIndex] || 'N/A';
  return `${monthName} ${year}`;
}

/**
 * Export chart as image (bonus funkcionalnost)
 */
function exportChartAsImage(chartId, filename) {
  const canvas = document.getElementById(chartId);
  if (canvas) {
    const url = canvas.toDataURL('image/png');
    const link = document.createElement('a');
    link.download = filename + '.png';
    link.href = url;
    link.click();
  }
}

/**
 * Resize charts on window resize
 */
window.addEventListener('resize', function() {
  // Chart.js automatically handles resizing, but we can add custom logic here if needed
  setTimeout(() => {
    console.log('Charts resized');
  }, 100);
});
//# sourceMappingURL=financijeadmincharts.aa351c8b.js.map