/**
 * JavaScript za funkcionalnost obrasca za uređivanje financijskih stavki
 * Ažurirano za novi format perioda MM-YYYY
 */

// Inicijalizacija nakon učitavanja DOM-a
$(function () {
    // Inicijalizacija datepickera
    const datePicker = flatpickr("#valuta_placanja_input", {
        dateFormat: "d-m-Y",
        locale: "hr",
        allowInput: true,
        disableMobile: true,
        appendTo: document.body,
        position: "auto"
    });
    
    // Učini ikonu kalendara klikabilnom
    $("#valuta_placanja_input").closest(".input-group").find(".input-group-text").on("click", function() {
        datePicker.open();
    });
    
    // Dodaj event listenere za polja koja utječu na ukupni iznos
    $('#cijena_bez_pdv').on('change', calculateTotal);
    $('#pdv').on('change', calculateTotal);
    
    // Dodaj validaciju za period rada polje
    $('#period_input').on('input', validatePeriodFormat);
    $('#period_input').on('blur', formatPeriodInput);
});

/**
 * Funkcija za izračun ukupnog iznosa (cijena bez PDV + PDV)
 */
function calculateTotal() {
    const cijenaBezPdvEl = document.getElementById('cijena_bez_pdv');
    const pdvEl = document.getElementById('pdv');
    const ukupanIznosEl = document.getElementById('ukupan_iznos');
    
    if (cijenaBezPdvEl && ukupanIznosEl) {
        let cijenaBezPdv = parseFloat(cijenaBezPdvEl.value.replace(',', '.')) || 0;
        let pdv = pdvEl && pdvEl.value ? parseFloat(pdvEl.value.replace(',', '.')) || 0 : 0;
        
        let ukupno = cijenaBezPdv + pdv;
        ukupanIznosEl.value = ukupno.toFixed(2);
    }
}

/**
 * Validacija formata perioda tijekom tipkanja
 */
function validatePeriodFormat() {
    const periodInput = document.getElementById('period_input');
    const value = periodInput.value;
    
    // Ukloni sve što nije broj ili crtica
    const cleanValue = value.replace(/[^\d-]/g, '');
    
    // Ograniči na format MM-YYYY
    let formattedValue = cleanValue;
    
    // Ako ima više od jedne crtice, ukloni višak
    const parts = cleanValue.split('-');
    if (parts.length > 2) {
        formattedValue = parts[0] + '-' + parts[1];
    }
    
    // Ograniči mjesec na maksimalno 2 znamenke
    if (parts[0] && parts[0].length > 2) {
        formattedValue = parts[0].substring(0, 2) + (parts[1] ? '-' + parts[1] : '');
    }
    
    // Ograniči godinu na maksimalno 4 znamenke
    if (parts[1] && parts[1].length > 4) {
        formattedValue = parts[0] + '-' + parts[1].substring(0, 4);
    }
    
    periodInput.value = formattedValue;
    
    // Vizualna povratna informacija
    validatePeriodValue(formattedValue);
}

/**
 * Formatiranje perioda kada korisnik završi s unosom
 */
function formatPeriodInput() {
    const periodInput = document.getElementById('period_input');
    const value = periodInput.value.trim();
    
    if (!value) return;
    
    const parts = value.split('-');
    
    if (parts.length === 2) {
        let mjesec = parseInt(parts[0]) || 0;
        let godina = parseInt(parts[1]) || 0;
        
        // Korekcija mjeseca
        if (mjesec < 1) mjesec = 1;
        if (mjesec > 12) mjesec = 12;
        
        // Korekcija godine
        if (godina < 2020) godina = new Date().getFullYear();
        if (godina > 2050) godina = 2050;
        
        // Formatiranje s vodećim nulama
        const formattedPeriod = String(mjesec).padStart(2, '0') + '-' + godina;
        periodInput.value = formattedPeriod;
        
        validatePeriodValue(formattedPeriod);
    }
}

/**
 * Validacija vrijednosti perioda
 */
function validatePeriodValue(value) {
    const periodInput = document.getElementById('period_input');
    const pattern = /^(\d{1,2})-(\d{4})$/;
    const match = value.match(pattern);
    
    // Ukloni postojeće klase
    periodInput.classList.remove('is-valid', 'is-invalid');
    
    if (match) {
        const mjesec = parseInt(match[1]);
        const godina = parseInt(match[2]);
        
        if (mjesec >= 1 && mjesec <= 12 && godina >= 2020 && godina <= 2050) {
            periodInput.classList.add('is-valid');
            return true;
        }
    }
    
    if (value.length > 0) {
        periodInput.classList.add('is-invalid');
    }
    
    return false;
}

/**
 * Helper funkcija za dobivanje trenutnog mjeseca i godine
 */
function getCurrentPeriod() {
    const now = new Date();
    const mjesec = String(now.getMonth() + 1).padStart(2, '0');
    const godina = now.getFullYear();
    return mjesec + '-' + godina;
}

/**
 * Postaviti trenutni period kao zadanu vrijednost (poziva se po potrebi)
 */
function setCurrentPeriod() {
    const periodInput = document.getElementById('period_input');
    if (periodInput && !periodInput.value) {
        periodInput.value = getCurrentPeriod();
        validatePeriodValue(periodInput.value);
    }
}
//# sourceMappingURL=editnaplata.c528eb20.js.map