/**
 * Naplata Module Configuration - REFACTORED VERSION
 * Client-side processing konfiguracija
 * Čisti konfiguracijski file za naplata modul
 */

// ==========================================
// NAPLATA MODULE CONFIGURATION
// ==========================================

const NAPLATA_CONFIG = {
  // Module basic info
  module: {
    name: 'naplata',
    title: 'Naplata Management',
    version: '1.0.0'
  },

  // Table configuration
  table: {
    selector: '#mojdatatables',
    pageLength: 20,
    lengthMenu: [[20, 50, 100, -1], [20, 50, 100, "All"]],
    scrollX: true,
    scrollCollapse: true,
    stateSave: true,
    cascadingFilters: true,
    filterRefreshMode: 'client', // CLIENT-SIDE processing
    defaultOrder: [[1, 'asc']] // Sort by Tvrtka
  },

  // Localization
  locale: {
    language: 'hr',
    fallback: 'en'
  },

  // Routes configuration
  routes: {
    fetchAjax: '/naplata/api/fetch-naplata',
    editRoute: '/naplata/edit',
    deleteRoute: '/naplata/api/delete'
  },

  // Features configuration
  features: {
    inlineEditing: true,
    columnVisibility: true,
    exportExcel: true,
    resetFilters: true,
    compactMode: true,
    rowStyling: true
  },

  // Debug settings
  debug: {
    enabled: false,
    logLevel: 'info'
  },

  // Performance settings - CLIENT SIDE
  performance: {
    largeDatasetThreshold: 1000,
    serverSideProcessing: false, // CLIENT-SIDE processing
    optimizeForLargeData: false
  },

  // UI customization
  ui: {
    theme: 'auto',
    compactRows: true,
    showRowNumbers: true,
    highlightOnHover: true
  },

  // Export settings
  export: {
    excel: {
      filename: 'naplata-export',
      excludeColumns: [0], // Exclude action column
      includeFiltered: true
    }
  },

  // Custom callbacks
  callbacks: {
    onInitComplete: null,
    onRowClick: null,
    onFilterChange: null,
    beforeExport: null
  }
};

// ==========================================
// INITIALIZATION LOGIC
// ==========================================

/**
 * Dinamički dohvat ruta iz AppConfig
 */
function getNaplataRoutesFromAppConfig() {
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  return {
    fetchAjax: `${publicUrl}${NAPLATA_CONFIG.routes.fetchAjax}`,
    editRoute: `${publicUrl}${NAPLATA_CONFIG.routes.editRoute}`,
    deleteRoute: `${publicUrl}${NAPLATA_CONFIG.routes.deleteRoute}`
  };
}

/**
 * Pripremi potpunu konfiguraciju za DataTables
 */
function prepareNaplataConfiguration() {
  const routes = getNaplataRoutesFromAppConfig();
  
  return {
    // Table settings
    selector: NAPLATA_CONFIG.table.selector,
    locale: NAPLATA_CONFIG.locale.language,
    
    // Routes
    routes: routes,
    
    // Features
    cascadingFilters: NAPLATA_CONFIG.features.cascadingFilters,
    filterRefreshMode: NAPLATA_CONFIG.table.filterRefreshMode,
    
    // Debug
    debug: NAPLATA_CONFIG.debug.enabled,
    
    // Table config
    config: {
      pageLength: NAPLATA_CONFIG.table.pageLength,
      lengthMenu: NAPLATA_CONFIG.table.lengthMenu,
      scrollX: NAPLATA_CONFIG.table.scrollX,
      scrollCollapse: NAPLATA_CONFIG.table.scrollCollapse,
      stateSave: NAPLATA_CONFIG.table.stateSave,
      serverSideThreshold: NAPLATA_CONFIG.performance.largeDatasetThreshold,
      // CLIENT-SIDE specific settings
      clientSideProcessing: true,
      loadAllData: true,
      order: NAPLATA_CONFIG.table.defaultOrder // Default sorting
    }
  };
}

/**
 * Global funkcije za vanjsku upotrebu (legacy compatibility)
 */
window.NaplataTableFunctions = {
  // Reload table
  reloadTable: function() {
    if (window.naplataTable) {
      window.naplataTable.ajax.reload();
    }
  },
  
  // Apply filter to specific column
  applyFilter: function(column, value) {
    if (window.naplataTable) {
      window.naplataTable.column(column).search(value).draw();
    }
  },
  
  // Clear all filters
  clearFilters: function() {
    if (window.naplataTable) {
      window.naplataTable.search('').columns().search('').draw();
    }
  },

  // Get current configuration
  getConfig: function() {
    return NAPLATA_CONFIG;
  },

  // Update configuration
  updateConfig: function(newConfig) {
    Object.assign(NAPLATA_CONFIG, newConfig);
  }
};

// ==========================================
// TABLE INITIALIZATION
// ==========================================

/**
 * Inicijaliziraj naplata table s client-side konfiguration
 */
async function initializeNaplataTable() {
  try {
    // Show loading state
    window.dtIsLoading = true;

    if (NAPLATA_CONFIG.debug.enabled) {
      console.log('=== NAPLATA MODULE INITIALIZATION (CLIENT-SIDE) ===');
      console.log('Configuration:', NAPLATA_CONFIG);
    }

    // Provjeri da li postoji table element
    if (!document.querySelector(NAPLATA_CONFIG.table.selector)) {
      console.warn(`Table element ${NAPLATA_CONFIG.table.selector} not found`);
      return;
    }

    // Dinamički uvezi DataTables factory
    const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
    const factory = await import(`${publicUrl}/srcassets/datatables/js/datatables-factory.js`);
    
    // Pripremi konfiguraciju
    const config = prepareNaplataConfiguration();
    
    if (NAPLATA_CONFIG.debug.enabled) {
      console.log('Final configuration for DataTables (CLIENT-SIDE):', config);
    }

    // Kreiraj DataTable instance
    const table = await factory.createDataTable(
      config.selector,
      config.routes.fetchAjax,
      config
    );

    // Spremi reference globally
    window.naplataTable = table;

    // Apply loading completion styles
    setTimeout(() => {
      const wrapper = document.getElementById('mojdatatables_wrapper');
      const tableEl = document.getElementById('mojdatatables');
      
      if (wrapper) wrapper.classList.add('dt-loaded');
      if (tableEl) tableEl.classList.add('dt-loaded');
      
      // Remove init style
      const styleEl = document.getElementById('dt-init-style');
      if (styleEl) styleEl.remove();
      
      // Reset loading flag
      window.dtIsLoading = false;
      
      if (NAPLATA_CONFIG.debug.enabled) {
        console.log('✓ Naplata table initialized successfully (CLIENT-SIDE)');
      }
    }, 300);

    return table;

  } catch (error) {
    console.error('Error initializing naplata table:', error);
    
    // Reset loading state on error
    window.dtIsLoading = false;
    
    // Show error to user
    if (NAPLATA_CONFIG.debug.enabled) {
      alert('Error loading naplata table. Check console for details.');
    }
  }
}

// ==========================================
// LOADING STATE MANAGEMENT
// ==========================================

/**
 * Setup loading styles
 */
(function setupNaplataLoadingStyles() {
  // Create loading style element
  const styleEl = document.createElement('style');
  styleEl.id = 'dt-init-style';
  styleEl.innerHTML = `
    #mojdatatables, #mojdatatables_wrapper {
      visibility: hidden;
      opacity: 0;
      transition: opacity 0.3s ease;
    }
    .dt-loaded {
      visibility: visible !important;
      opacity: 1 !important;
    }
  `;
  document.head.appendChild(styleEl);
})();

// ==========================================
// MODULE INITIALIZATION
// ==========================================

/**
 * Auto-initialization kada je DOM spreman
 */
(function autoInitializeNaplata() {
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeNaplataTable);
  } else {
    // DOM is already ready
    setTimeout(initializeNaplataTable, 50);
  }
})();

// ==========================================
// EXPORT FOR TESTING/EXTERNAL USE
// ==========================================

// Make configuration available globally for testing
if (NAPLATA_CONFIG.debug.enabled) {
  window.NAPLATA_CONFIG = NAPLATA_CONFIG;
  window.initializeNaplataTable = initializeNaplataTable;
}
//# sourceMappingURL=naplata_listssp.67c61389.js.map