// Enhanced naplataadmin.js - Matching Sales Admin functionality
document.addEventListener('DOMContentLoaded', function() {
  // Initialize all dashboard functionality
  initializeCounterAnimations();
  initializeInteractiveElements();
  initializeSidebar();
  
  // Set up periodic refresh for real-time data
  if (typeof refreshNaplataInterval === 'undefined') {
    let refreshNaplataInterval = setInterval(refreshStats, 300000); // Every 5 minutes
  }
});

/**
 * Initialize counter animations for KPI values
 */
function initializeCounterAnimations() {
  // Animate dashboard cards with currency values
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach(valueElement => {
    animateCounter(valueElement, 1200);
  });
}

/**
 * Animate a counter element from 0 to its final value
 */
function animateCounter(element, duration) {
  const text = element.textContent.trim();
  let finalValue, suffix = '', isPercentage = false, isCurrency = false;
  
  // Handle percentages
  if (text.includes('%')) {
    finalValue = parseFloat(text.replace('%', ''));
    suffix = '%';
    isPercentage = true;
  }
  // Handle currency values (with comma as decimal separator)
  else if (text.includes(',')) {
    // Convert from Croatian format (123.456,78) to standard format
    const cleanText = text.replace(/\./g, '').replace(',', '.');
    finalValue = parseFloat(cleanText);
    isCurrency = true;
  }
  // Handle pure numbers
  else if (/^\d+$/.test(text)) {
    finalValue = parseInt(text);
  }
  // Skip non-numeric values
  else {
    return;
  }
  
  if (isNaN(finalValue)) return;
  
  // Store original formatted value
  const originalValue = text;
  element.setAttribute('data-original', originalValue);
  
  let startTimestamp = null;
  const startValue = 0;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    const currentValue = progress * (finalValue - startValue) + startValue;
    
    if (isPercentage) {
      element.textContent = currentValue.toFixed(1) + suffix;
    } else if (isCurrency) {
      element.textContent = formatCurrency(currentValue);
    } else {
      element.textContent = Math.round(currentValue) + suffix;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      element.textContent = originalValue;
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Initialize interactive elements
 */
function initializeInteractiveElements() {
  // Add hover effects to recent items
  const recentItems = document.querySelectorAll('.recent-item');
  recentItems.forEach(item => {
    item.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
    });
    
    item.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
    });
  });
  
  // Add hover effects to dashboard cards
  const dashboardCards = document.querySelectorAll('.dashboard-card');
  dashboardCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-8px)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
    });
  });
  
  // Add click handlers to dashboard cards for potential drill-down
  dashboardCards.forEach((card, index) => {
    card.addEventListener('click', function() {
      const label = this.querySelector('.card-label').textContent;
      console.log('Clicked on KPI:', label);
      
      // Potential navigation based on card type
      switch(index) {
        case 0: // Ukupan iznos
          console.log('Navigate to total amount details');
          break;
        case 1: // Naplaćeno
          console.log('Navigate to paid amount details');
          break;
        case 2: // Nenaplaćeno
          console.log('Navigate to unpaid amount details');
          break;
        case 3: // Postotak naplate
          console.log('Navigate to collection rate details');
          break;
      }
    });
  });
  
  // Add click handlers to export options
  const exportOptions = document.querySelectorAll('.export-option');
  exportOptions.forEach(option => {
    option.addEventListener('click', function(e) {
      if (!e.target.closest('.export-button')) {
        // Clicked on option but not button - highlight the button
        const button = this.querySelector('.export-button');
        if (button) {
          button.style.transform = 'scale(1.1)';
          setTimeout(() => {
            button.style.transform = '';
          }, 200);
        }
      }
    });
  });
}

/**
 * Initialize sidebar functionality
 */
function initializeSidebar() {
  const mobileToggle = document.getElementById('mobile-toggle');
  const sidebar = document.querySelector('.sidebar');
  
  // Mobile sidebar toggle
  if (mobileToggle) {
    mobileToggle.addEventListener('click', function() {
      sidebar.classList.toggle('open');
      
      // Change icon based on state
      const icon = mobileToggle.querySelector('i');
      if (sidebar.classList.contains('open')) {
        icon.classList.remove('fa-bars');
        icon.classList.add('fa-times');
      } else {
        icon.classList.remove('fa-times');
        icon.classList.add('fa-bars');
      }
    });
  }
  
  // Initialize sidebar items
  const sidebarItems = document.querySelectorAll('.sidebar-item');
  sidebarItems.forEach(item => {
    const link = item.querySelector('.sidebar-link');
    if (link) {
      link.addEventListener('click', function(e) {
        e.preventDefault();
        
        // Toggle active state on the clicked item
        const wasActive = item.classList.contains('active');
        
        // If we're on mobile, don't close the sidebar when toggling items
        const isMobile = window.innerWidth <= 768;
        
        // First, check if we should close other items
        if (!e.ctrlKey) { // Allow keeping multiple items open with Ctrl key
          sidebarItems.forEach(i => {
            if (i !== item) { // Don't touch the clicked item yet
              i.classList.remove('active');
            }
          });
        }
        
        // Toggle the clicked item
        if (wasActive) {
          item.classList.remove('active');
        } else {
          item.classList.add('active');
        }
        
        // Prevent closing sidebar on mobile when clicking items
        e.stopPropagation();
      });
    }
  });
  
  // Handle clicks on submenu links
  const submenuLinks = document.querySelectorAll('.sidebar-submenu li a');
  submenuLinks.forEach(link => {
    link.addEventListener('click', function(e) {
      // Allow normal navigation for these links
      // If we're on mobile, close the sidebar after clicking a submenu link
      if (window.innerWidth <= 768) {
        setTimeout(() => {
          sidebar.classList.remove('open');
          if (mobileToggle) {
            const icon = mobileToggle.querySelector('i');
            icon.classList.remove('fa-times');
            icon.classList.add('fa-bars');
          }
        }, 300);
      }
    });
  });
}

/**
 * Function to refresh stats via AJAX
 */
function refreshStats() {
  fetch('/naplata/admin-stats', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    }
  })
    .then(response => {
      if (!response.ok) {
        throw new Error('Network response was not ok');
      }
      return response.json();
    })
    .then(data => {
      updateDashboardValues(data);
      showUpdateIndicator();
    })
    .catch(error => {
      console.error('Error refreshing stats:', error);
      showErrorIndicator();
    });
}

/**
 * Update dashboard values with new data
 */
function updateDashboardValues(data) {
  // Update main dashboard cards
  updateElementValue('.dashboard-card:nth-child(1) .card-value', data.totalAmount, 'currency');
  updateElementValue('.dashboard-card:nth-child(2) .card-value', data.paidAmount, 'currency');
  updateElementValue('.dashboard-card:nth-child(3) .card-value', data.unpaidAmount, 'currency');
  
  // Calculate and update percentage
  const percentage = data.totalAmount > 0 ? (data.paidAmount / data.totalAmount) * 100 : 0;
  updateElementValue('.dashboard-card:nth-child(4) .card-value', percentage, '%');
}

/**
 * Update a specific element's value with animation
 */
function updateElementValue(selector, newValue, type = '') {
  const element = document.querySelector(selector);
  if (!element || newValue === undefined) return;
  
  let currentValue, targetValue;
  
  if (type === 'currency') {
    // Parse Croatian currency format
    const currentText = element.textContent.replace(/\./g, '').replace(',', '.');
    currentValue = parseFloat(currentText) || 0;
    targetValue = parseFloat(newValue) || 0;
  } else if (type === '%') {
    currentValue = parseFloat(element.textContent.replace('%', '')) || 0;
    targetValue = parseFloat(newValue) || 0;
  } else {
    currentValue = parseFloat(element.textContent.replace(/[^\d.-]/g, '')) || 0;
    targetValue = parseFloat(newValue) || 0;
  }
  
  // Only animate if value has changed
  if (Math.abs(currentValue - targetValue) > 0.01) {
    // Add update flash effect
    element.classList.add('value-updated');
    
    // Animate to new value
    animateValueChange(element, currentValue, targetValue, type, 800);
    
    // Remove flash effect after animation
    setTimeout(() => {
      element.classList.remove('value-updated');
    }, 1000);
  }
}

/**
 * Animate value change from current to new value
 */
function animateValueChange(element, startValue, endValue, type, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    const currentValue = startValue + (endValue - startValue) * progress;
    
    if (type === '%') {
      element.textContent = currentValue.toFixed(1) + '%';
    } else if (type === 'currency') {
      element.textContent = formatCurrency(currentValue);
    } else {
      element.textContent = Math.round(currentValue);
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      if (type === '%') {
        element.textContent = endValue.toFixed(1) + '%';
      } else if (type === 'currency') {
        element.textContent = formatCurrency(endValue);
      } else {
        element.textContent = Math.round(endValue);
      }
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Format currency value (Croatian format)
 */
function formatCurrency(value) {
  return Number(value).toLocaleString('hr-HR', {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2
  });
}

/**
 * Show update indicator
 */
function showUpdateIndicator() {
  // Create or update a small indicator showing last update time
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  const now = new Date();
  indicator.innerHTML = `
    <i class="fas fa-sync-alt"></i>
    <span>Ažurirano: ${now.toLocaleTimeString('hr-HR')}</span>
  `;
  
  // Add pulse animation
  indicator.classList.add('updated');
  setTimeout(() => {
    indicator.classList.remove('updated');
  }, 1000);
}

/**
 * Show error indicator
 */
function showErrorIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator error';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-exclamation-triangle"></i>
    <span>Greška pri ažuriranju</span>
  `;
  
  indicator.classList.add('error');
  setTimeout(() => {
    indicator.classList.remove('error');
  }, 3000);
}

/**
 * Show loading indicator
 */
function showLoadingIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-spinner fa-spin"></i>
    <span>Ažuriranje...</span>
  `;
}

/**
 * Manual refresh function that can be called by user
 */
function manualRefresh() {
  showLoadingIndicator();
  refreshStats();
}
//# sourceMappingURL=naplataadmin.07a047f7.js.map