/**
 * Rashodi Module Configuration - REFACTORED VERSION
 * Client-side processing konfiguracija
 * Čisti konfiguracijski file za rashodi modul
 */

// ==========================================
// RASHODI MODULE CONFIGURATION
// ==========================================

const RASHODI_CONFIG = {
  // Module basic info
  module: {
    name: 'rashodi',
    title: 'Rashodi Management',
    version: '1.0.0'
  },

  // Table configuration - RAZLIČIT SELECTOR!
  table: {
    selector: '#rashodidatatables', // RASHODI koristi drugi selector!
    pageLength: 20,
    lengthMenu: [[20, 50, 100, -1], [20, 50, 100, "All"]],
    scrollX: true,
    scrollCollapse: true,
    stateSave: true,
    cascadingFilters: true,
    filterRefreshMode: 'client', // CLIENT-SIDE processing
    defaultOrder: [[1, 'desc']] // Sort by Period descending
  },

  // Localization
  locale: {
    language: 'hr',
    fallback: 'en'
  },

  // Routes configuration
  routes: {
    fetchAjax: '/naplata/rashodi/api/fetch-rashodi',
    editRoute: '/naplata/rashodi/edit',
    deleteRoute: '/naplata/rashodi/api/delete'
  },

  // Features configuration
  features: {
    inlineEditing: true,
    columnVisibility: true,
    exportExcel: true,
    resetFilters: true,
    compactMode: true,
    rowStyling: true
  },

  // Debug settings
  debug: {
    enabled: false,
    logLevel: 'info'
  },

  // Performance settings - CLIENT SIDE
  performance: {
    largeDatasetThreshold: 1000,
    serverSideProcessing: false, // CLIENT-SIDE processing
    optimizeForLargeData: false
  },

  // UI customization
  ui: {
    theme: 'auto',
    compactRows: true,
    showRowNumbers: true,
    highlightOnHover: true
  },

  // Export settings
  export: {
    excel: {
      filename: 'rashodi-export',
      excludeColumns: [0], // Exclude action column
      includeFiltered: true
    }
  },

  // Custom callbacks
  callbacks: {
    onInitComplete: null,
    onRowClick: null,
    onFilterChange: null,
    beforeExport: null
  }
};

// ==========================================
// INITIALIZATION LOGIC
// ==========================================

/**
 * Dinamički dohvat ruta iz AppConfig
 */
function getRashodiRoutesFromAppConfig() {
  const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
  
  return {
    fetchAjax: `${publicUrl}${RASHODI_CONFIG.routes.fetchAjax}`,
    editRoute: `${publicUrl}${RASHODI_CONFIG.routes.editRoute}`,
    deleteRoute: `${publicUrl}${RASHODI_CONFIG.routes.deleteRoute}`
  };
}

/**
 * Pripremi potpunu konfiguraciju za DataTables
 */
function prepareRashodiConfiguration() {
  const routes = getRashodiRoutesFromAppConfig();
  
  return {
    // Table settings
    selector: RASHODI_CONFIG.table.selector,
    locale: RASHODI_CONFIG.locale.language,
    
    // Routes
    routes: routes,
    
    // Features
    cascadingFilters: RASHODI_CONFIG.features.cascadingFilters,
    filterRefreshMode: RASHODI_CONFIG.table.filterRefreshMode,
    
    // Debug
    debug: RASHODI_CONFIG.debug.enabled,
    
    // Table config
    config: {
      pageLength: RASHODI_CONFIG.table.pageLength,
      lengthMenu: RASHODI_CONFIG.table.lengthMenu,
      scrollX: RASHODI_CONFIG.table.scrollX,
      scrollCollapse: RASHODI_CONFIG.table.scrollCollapse,
      stateSave: RASHODI_CONFIG.table.stateSave,
      serverSideThreshold: RASHODI_CONFIG.performance.largeDatasetThreshold,
      // CLIENT-SIDE specific settings
      clientSideProcessing: true,
      loadAllData: true,
      order: RASHODI_CONFIG.table.defaultOrder // Default sorting by Period DESC
    }
  };
}

/**
 * Global funkcije za vanjsku upotrebu (legacy compatibility)
 */
window.RashodiTableFunctions = {
  // Reload table
  reloadTable: function() {
    if (window.rashodiTable) {
      window.rashodiTable.ajax.reload();
      console.log('Rashodi table reloaded');
    }
  },
  
  // Apply filter to specific column
  applyFilter: function(column, value) {
    if (window.rashodiTable) {
      window.rashodiTable.column(column).search(value).draw();
    }
  },
  
  // Clear all filters
  clearFilters: function() {
    if (window.rashodiTable) {
      window.rashodiTable.search('').columns().search('').draw();
    }
  },

  // Get current configuration
  getConfig: function() {
    return RASHODI_CONFIG;
  },

  // Update configuration
  updateConfig: function(newConfig) {
    Object.assign(RASHODI_CONFIG, newConfig);
  }
};

// ==========================================
// TABLE INITIALIZATION
// ==========================================

/**
 * Inicijaliziraj rashodi table s client-side konfiguration
 */
async function initializeRashodiTable() {
  try {
    // Show loading state
    window.dtIsLoading = true;

    if (RASHODI_CONFIG.debug.enabled) {
      console.log('=== RASHODI MODULE INITIALIZATION (CLIENT-SIDE) ===');
      console.log('Configuration:', RASHODI_CONFIG);
    }

    // Provjeri da li postoji table element
    if (!document.querySelector(RASHODI_CONFIG.table.selector)) {
      console.warn(`Table element ${RASHODI_CONFIG.table.selector} not found`);
      return;
    }

    // Dinamički uvezi DataTables factory
    const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;
    const factory = await import(`${publicUrl}/srcassets/datatables/js/datatables-factory.js`);
    
    // Pripremi konfiguraciju
    const config = prepareRashodiConfiguration();
    
    if (RASHODI_CONFIG.debug.enabled) {
      console.log('Final configuration for DataTables (CLIENT-SIDE):', config);
    }

    // Kreiraj DataTable instance
    const table = await factory.createDataTable(
      config.selector,
      config.routes.fetchAjax,
      config
    );

    // Spremi reference globally
    window.rashodiTable = table;

    // Apply loading completion styles - SPECIFIČNI ZA RASHODI
    setTimeout(() => {
      const wrapper = document.getElementById('rashodidatatables_wrapper');
      const tableEl = document.getElementById('rashodidatatables');
      
      if (wrapper) wrapper.classList.add('dt-rashodi-loaded');
      if (tableEl) tableEl.classList.add('dt-rashodi-loaded');
      
      // Remove init style
      const styleEl = document.getElementById('dt-rashodi-init-style');
      if (styleEl) styleEl.remove();
      
      // Reset loading flag
      window.dtIsLoading = false;
      
      if (RASHODI_CONFIG.debug.enabled) {
        console.log('✓ Rashodi table initialized successfully (CLIENT-SIDE)');
      }
      
      console.log('Rashodi table initialized successfully');
    }, 300);

    return table;

  } catch (error) {
    console.error('Error initializing rashodi table:', error);
    
    // Reset loading state on error
    window.dtIsLoading = false;
    
    // Show error to user
    if (RASHODI_CONFIG.debug.enabled) {
      alert('Error loading rashodi table. Check console for details.');
    }
  }
}

// ==========================================
// LOADING STATE MANAGEMENT
// ==========================================

/**
 * Setup loading styles - SPECIFIČNI ZA RASHODI
 */
(function setupRashodiLoadingStyles() {
  // Create loading style element
  const styleEl = document.createElement('style');
  styleEl.id = 'dt-rashodi-init-style';
  styleEl.innerHTML = `
    #rashodidatatables, #rashodidatatables_wrapper {
      visibility: hidden;
      opacity: 0;
      transition: opacity 0.3s ease;
    }
    .dt-rashodi-loaded {
      visibility: visible !important;
      opacity: 1 !important;
    }
  `;
  document.head.appendChild(styleEl);
})();

// ==========================================
// MODULE INITIALIZATION
// ==========================================

/**
 * Auto-initialization kada je DOM spreman
 */
(function autoInitializeRashodi() {
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeRashodiTable);
  } else {
    // DOM is already ready
    setTimeout(initializeRashodiTable, 50);
  }
})();

// ==========================================
// EXPORT FOR TESTING/EXTERNAL USE
// ==========================================

// Make configuration available globally for testing
if (RASHODI_CONFIG.debug.enabled) {
  window.RASHODI_CONFIG = RASHODI_CONFIG;
  window.initializeRashodiTable = initializeRashodiTable;
}
//# sourceMappingURL=rashodi_listssp.107d3d2a.js.map