// Uzmi public_url iz AppConfig.paths
const publicUrl = JSON.parse(document.getElementById('app-config').textContent).paths.app_url;

// Route definitions directly in JavaScript
const getTasks = `${publicUrl}/pm/api/get-tasks`;  // Adjusted from data-get-tasks-route
const addTask = `${publicUrl}/pm/addtask`;    // Adjusted from data-add-task-route
const updateTask = `${publicUrl}/pm/updatetask`; // Adjusted from data-update-task-route
const deleteTask = `${publicUrl}/pm/api/delete-task`; // Adjusted from data-delete-task-route
const editTask = `${publicUrl}/pm/api/edit-task`;  // Adjusted from data-edit-task-route

let taskToDelete = null;
let csrfToken = null;

// Retrieve theme and apply appropriate UI adjustments
function applyTheme() {
    const theme = document.documentElement.getAttribute('data-theme') || 'dark';
    // Set theme-specific UI elements
    const accent = theme === 'dark' ? '#ff6b00' : '#3498db';
    
    // Update modal title colors for the current theme
    const modalTitles = document.querySelectorAll('.modal h2');
    modalTitles.forEach(title => {
        title.style.color = accent;
    });
}

// Modal functionality
function openModal(task = null) {
    const modal = document.getElementById('taskModal');
    const modalTitle = document.getElementById('modalTitle');
    const submitButton = document.getElementById('submitButton');
    const taskForm = document.getElementById('taskForm');
    const taskIdInput = document.getElementById('taskId');
    
    if (task) {
        modalTitle.textContent = 'Edit Task';
        submitButton.textContent = 'Update Task';
        document.getElementById('taskTitle').value = task.title;
        document.getElementById('taskDescription').value = task.description;
        taskIdInput.value = task.id;
    } else {
        modalTitle.textContent = 'Add New Task';
        submitButton.textContent = 'Add Task';
        taskForm.reset();
        taskIdInput.value = '';
    }
    
    // Show modal with animation
    modal.style.display = 'flex';
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
    
    // Apply theme styling
    applyTheme();
}

function closeModal() {
    const modal = document.getElementById('taskModal');
    modal.classList.remove('show');
    setTimeout(() => {
        modal.style.display = 'none';
    }, 300);
}

function openConfirmModal(id) {
    taskToDelete = id;
    const confirmModal = document.getElementById('confirmModal');
    confirmModal.style.display = 'flex';
    setTimeout(() => {
        confirmModal.classList.add('show');
    }, 10);
    
    // Apply theme styling
    applyTheme();
}

function closeConfirmModal() {
    const confirmModal = document.getElementById('confirmModal');
    confirmModal.classList.remove('show');
    setTimeout(() => {
        confirmModal.style.display = 'none';
    }, 300);
    taskToDelete = null;
}

// Close modal on clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('taskModal');
    const confirmModal = document.getElementById('confirmModal');
    
    if (event.target === modal) {
        closeModal();
    }
    
    if (event.target === confirmModal) {
        closeConfirmModal();
    }
};

function confirmDelete() {
    if (!taskToDelete) {
        console.error('Missing taskId for delete operation');
        return;
    }
    
    const deleteBtn = document.querySelector('.delete-btn');
    const originalText = deleteBtn.textContent;
    
    // Show loading state
    deleteBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Deleting...';
    deleteBtn.disabled = true;
    
    window.fetchWithCsrf(deleteTask, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ id: taskToDelete })
    })    
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            fetchTasks();
            showToast('Task deleted successfully');
        } else {
            showToast('Error: ' + (data.message || 'Could not delete task'), 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Error: ' + error.message, 'error');
    })
    .finally(() => {
        // Reset button state
        deleteBtn.textContent = originalText;
        deleteBtn.disabled = false;
        closeConfirmModal();
    });
}

// Create a toast notification - pojednostavljeno, bez animacija
function showToast(message, type = 'success') {
    // Check if a toast already exists and remove it
    let existingToast = document.querySelector('.toast');
    if (existingToast) {
        document.body.removeChild(existingToast);
    }
    
    // Create new toast
    const toast = document.createElement('div');
    toast.className = 'toast';
    
    // Set icon based on type
    const icon = type === 'success' ? 'check-circle' : 'exclamation-circle';
    const iconClass = type === 'success' ? 'toast-icon-success' : 'toast-icon-error';
    
    toast.innerHTML = `
        <div class="toast-content">
            <i class="fas fa-${icon} toast-icon ${iconClass}"></i>
            <div class="toast-message">${message}</div>
        </div>
    `;
    
    document.body.appendChild(toast);
    
    // Prikaži toast odmah
    toast.classList.add('show');
    
    // Sakrij toast nakon 3 sekunde
    setTimeout(() => {
        if (document.body.contains(toast)) {
            document.body.removeChild(toast);
        }
    }, 3000);
}

// Fetch tasks function
// Fetch tasks function
// Fetch tasks function
function fetchTasks() {
    window.fetchWithCsrf(getTasks)
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(tasks => {
            // Clear existing tasks
            document.querySelectorAll(".task-list").forEach(list => list.innerHTML = "");
            
            if (!Array.isArray(tasks)) {
                console.error('Expected array of tasks but got:', tasks);
                return;
            }
            
            tasks.forEach(task => {
                // Skip if no id or missing required properties
                if (!task.id || !task.title) {
                    console.warn('Task missing required properties:', task);
                    return;
                }
                
                const taskElement = document.createElement("div");
                taskElement.classList.add("task");
                taskElement.setAttribute("draggable", "true");
                taskElement.setAttribute("data-id", task.id);
                
                // Create task content
                const taskContent = document.createElement("div");
                taskContent.innerHTML = `
                    <strong>${task.title}</strong>
                    <p>${task.description || ''}</p>
                `;
                
                // Create edit button - removed onclick handler
                const editButton = document.createElement("i");
                editButton.className = "fas fa-pencil-alt edit-button";
                
                taskElement.appendChild(taskContent);
                taskElement.appendChild(editButton);
                
                // Add to proper column
                const statusColumn = task.pm_status || 'todo';
                const columnList = document.getElementById(`${statusColumn}-list`);
                
                if (columnList) {
                    columnList.appendChild(taskElement);
                } else {
                    console.warn(`Column list for status "${statusColumn}" not found`);
                }
                
                // Add drag event listeners
                taskElement.addEventListener("dragstart", (e) => {
                    e.dataTransfer.setData("text/plain", task.id);
                    taskElement.classList.add('dragging');
                    
                    // Set drag ghost image (optional)
                    const ghostElement = taskElement.cloneNode(true);
                    ghostElement.style.position = 'absolute';
                    ghostElement.style.top = '-1000px';
                    document.body.appendChild(ghostElement);
                    e.dataTransfer.setDragImage(ghostElement, 20, 20);
                    
                    // Remove ghost after drag operation
                    setTimeout(() => {
                        if (document.body.contains(ghostElement)) {
                            document.body.removeChild(ghostElement);
                        }
                    }, 0);
                });
                
                taskElement.addEventListener("dragend", () => {
                    taskElement.classList.remove('dragging');
                });
            });
        })
        .catch(error => {
            console.error('Error fetching tasks:', error);
            showToast('Error loading tasks: ' + error.message, 'error');
        });
}

document.addEventListener("DOMContentLoaded", function () {
    // Get CSRF token
    const csrfInput = document.querySelector('input[name="csrf_token"]');
    if (csrfInput) {
        csrfToken = csrfInput.value;
    } else {
        console.error('CSRF token not found in the page');
        showToast('Error: Security token not found', 'error');
    }
    
    // Apply theme
    applyTheme();
    
    // Initial tasks fetch
    fetchTasks();
    
    // Listen for theme changes
    document.addEventListener('themechange', applyTheme);

    // Dodani event listeneri za gumbe (zamjena za inline onclick)
    // Add task button
    document.getElementById('addTaskButton').addEventListener('click', () => {
        openModal();
    });

    // Close modal button
    document.getElementById('closeModalButton').addEventListener('click', () => {
        closeModal();
    });

    // Delete confirm button
    document.getElementById('deleteConfirmButton').addEventListener('click', () => {
        confirmDelete();
    });

    // Cancel delete button
    document.getElementById('cancelDeleteButton').addEventListener('click', () => {
        closeConfirmModal();
    });

    // Event delegation for edit buttons - NEW CODE
    document.addEventListener('click', function(e) {
        // Check if the clicked element is an edit button
        if (e.target.classList.contains('edit-button')) {
            e.stopPropagation();
            const taskElement = e.target.closest('.task');
            if (taskElement) {
                const taskId = taskElement.getAttribute('data-id');
                const titleElement = taskElement.querySelector('strong');
                const descElement = taskElement.querySelector('p');
                
                openModal({
                    id: taskId,
                    title: titleElement.textContent,
                    description: descElement.textContent
                });
            }
        }
    });

    // Task form submission (Add/Edit)
    document.getElementById("taskForm").addEventListener("submit", function (e) {
        e.preventDefault();
        
        const taskId = document.getElementById("taskId").value;
        const title = document.getElementById("taskTitle").value;
        const description = document.getElementById("taskDescription").value;
        const submitButton = document.getElementById("submitButton");
        
        // Show loading state
        const originalText = submitButton.textContent;
        submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
        submitButton.disabled = true;
    
        const url = taskId ? editTask : addTask;
        const body = {
            title: title,
            description: description
        };
    
        if (taskId) {
            body.id = taskId;
        } else {
            body.status = 'todo';
        }
    
        window.fetchWithCsrf(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(body)
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                fetchTasks();
                document.getElementById("taskForm").reset();
                closeModal();
                
                const action = taskId ? 'updated' : 'added';
                showToast(`Task ${action} successfully`);
            } else {
                showToast('Error: ' + (data.message || 'Could not save task'), 'error');
            }
        })
        .catch(error => {
            console.error('Fetch error:', error);
            showToast('Error: ' + error.message, 'error');
        })
        .finally(() => {
            // Reset button state
            submitButton.textContent = originalText;
            submitButton.disabled = false;
        });
    });

    // Drag & drop functionality
    let columns = document.querySelectorAll(".kanban-column");

    columns.forEach(column => {
        // Highlight column on dragover
        column.addEventListener("dragover", (e) => {
            e.preventDefault();
            column.classList.add('drag-over');
        });
        
        column.addEventListener("dragleave", () => {
            column.classList.remove('drag-over');
        });

        column.addEventListener("drop", (e) => {
            e.preventDefault();
            column.classList.remove('drag-over');
            
            let taskId = e.dataTransfer.getData("text/plain");
            if (!taskId) {
                console.error('No task ID found in drag data');
                return;
            }
            
            const taskElement = document.querySelector(`[data-id='${taskId}']`);
            if (!taskElement) {
                console.error(`Task element with ID ${taskId} not found`);
                return;
            }
            
            // Move task to this column
            const statusId = column.id;
            const taskList = column.querySelector(".task-list");
            
            if (taskList) {
                taskList.appendChild(taskElement);
                
                // Update task status in backend
                window.fetchWithCsrf(updateTask, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ 
                        id: taskId, 
                        status: statusId
                    })
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Failed to update task status');
                    }
                    return response.json();
                })
                .then(data => {
                    if (!data.success) {
                        showToast('Error updating task status', 'error');
                        // Refresh to get current state
                        fetchTasks();
                    }
                })
                .catch(error => {
                    console.error('Error updating task status:', error);
                    showToast('Error updating task status', 'error');
                    // Refresh to get current state
                    fetchTasks();
                });
            }
        });
    });

    // Recycle button drag & drop
    const recycleButton = document.getElementById('recycleButton');
    
    recycleButton.addEventListener('dragover', (e) => {
        e.preventDefault();
        recycleButton.style.transform = 'scale(1.2)';
        recycleButton.style.backgroundColor = '#c82333';
    });

    recycleButton.addEventListener('dragleave', () => {
        recycleButton.style.transform = 'scale(1)';
        recycleButton.style.backgroundColor = '';
    });

    recycleButton.addEventListener('drop', (e) => {
        e.preventDefault();
        recycleButton.style.transform = 'scale(1)';
        recycleButton.style.backgroundColor = '';
        
        let taskId = e.dataTransfer.getData("text/plain");
        if (taskId) {
            openConfirmModal(taskId);
        }
    });
});
//# sourceMappingURL=dashboard.2a0b2e43.js.map