/**
 * Agent Bonus Bodovanje Charts - PROFESSIONAL CHARTS
 */
document.addEventListener('DOMContentLoaded', function() {
  const chartsData = document.getElementById('agentbonusbodovanje-charts-data');
  
  if (!chartsData) {
    console.error('Agent bonus bodovanje charts data element not found!');
    return;
  }
  
  try {
    const monthlyTrend = JSON.parse(chartsData.getAttribute('data-monthly-trend') || '[]');
    const groupDistribution = JSON.parse(chartsData.getAttribute('data-group-distribution') || '[]');
    const topPerformers = JSON.parse(chartsData.getAttribute('data-top-performers') || '[]');
    
    // Create charts
    createMonthlyPointsChart(monthlyTrend);
    createGroupDistributionChart(groupDistribution);
    
  } catch (error) {
    console.error('Error initializing agent bonus bodovanje charts:', error);
  }
});

/**
 * Create monthly points trend chart
 */
function createMonthlyPointsChart(monthlyTrend) {
  const canvas = document.getElementById('monthlyPointsChart');
  if (!canvas) return;
  
  if (!monthlyTrend || monthlyTrend.length === 0) {
    canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-line"></i><p>Nema podataka o mjesečnim trendovima</p></div>';
    return;
  }
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  const labels = monthlyTrend.map(item => `${monthNames[item.month - 1]} ${item.year}`);
  const pointsData = monthlyTrend.map(item => parseInt(item.total_points || 0));
  const bonusData = monthlyTrend.map(item => parseFloat(item.total_bonus || 0));
  
  const ctx = canvas.getContext('2d');
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Ukupni bodovi',
          data: pointsData,
          backgroundColor: 'rgba(255, 215, 0, 0.15)',
          borderColor: '#ffd700',
          borderWidth: 3,
          fill: false,
          tension: 0.4,
          pointBackgroundColor: '#ffd700',
          pointBorderColor: '#ffffff',
          pointBorderWidth: 3,
          pointRadius: 6,
          pointHoverRadius: 8,
          pointHoverBackgroundColor: '#ffed4e',
          pointHoverBorderColor: '#ffffff',
          pointHoverBorderWidth: 3,
          yAxisID: 'y'
        },
        {
          label: 'Ukupni bonus (€)',
          data: bonusData,
          backgroundColor: 'rgba(40, 167, 69, 0.15)',
          borderColor: '#28a745',
          borderWidth: 3,
          fill: false,
          tension: 0.4,
          pointBackgroundColor: '#28a745',
          pointBorderColor: '#ffffff',
          pointBorderWidth: 3,
          pointRadius: 6,
          pointHoverRadius: 8,
          pointHoverBackgroundColor: '#20c997',
          pointHoverBorderColor: '#ffffff',
          pointHoverBorderWidth: 3,
          yAxisID: 'y1'
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        intersect: false,
        mode: 'index'
      },
      scales: {
        y: {
          type: 'linear',
          display: true,
          position: 'left',
          beginAtZero: true,
          grid: {
            color: 'rgba(255, 255, 255, 0.1)',
            drawBorder: false
          },
          ticks: {
            color: '#6c757d',
            font: {
              size: 11
            },
            callback: function(value) {
              return value.toLocaleString();
            }
          },
          title: {
            display: true,
            text: 'Bodovi',
            color: '#ffd700',
            font: {
              size: 12,
              weight: 'bold'
            }
          }
        },
        y1: {
          type: 'linear',
          display: true,
          position: 'right',
          beginAtZero: true,
          grid: {
            drawOnChartArea: false,
          },
          ticks: {
            color: '#6c757d',
            font: {
              size: 11
            },
            callback: function(value) {
              return value.toLocaleString() + ' €';
            }
          },
          title: {
            display: true,
            text: 'Bonus (€)',
            color: '#28a745',
            font: {
              size: 12,
              weight: 'bold'
            }
          }
        },
        x: {
          grid: {
            display: false
          },
          ticks: {
            color: '#6c757d',
            font: {
              size: 11
            },
            maxRotation: 0
          }
        }
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            boxWidth: 15,
            boxHeight: 15,
            font: {
              size: 12,
              weight: '600'
            },
            color: '#495057',
            padding: 15,
            usePointStyle: true,
            pointStyle: 'circle'
          }
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.8)',
          titleColor: '#ffffff',
          bodyColor: '#ffffff',
          borderColor: '#ffd700',
          borderWidth: 1,
          cornerRadius: 8,
          padding: 12,
          callbacks: {
            title: function(context) {
              return `Mjesec: ${context[0].label}`;
            },
            label: function(context) {
              if (context.datasetIndex === 0) {
                return `Bodovi: ${context.raw.toLocaleString()}`;
              } else {
                return `Bonus: ${context.raw.toLocaleString()} €`;
              }
            }
          }
        }
      }
    }
  });
}

/**
 * Create group distribution chart (doughnut chart)
 */
function createGroupDistributionChart(groupDistribution) {
  const canvas = document.getElementById('groupDistributionChart');
  if (!canvas) return;
  
  if (!groupDistribution || groupDistribution.length === 0) {
    canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-pie"></i><p>Nema podataka o distribuciji po grupama</p></div>';
    return;
  }
  
  // Filter out groups with zero points
  const validGroups = groupDistribution.filter(group => group.points > 0);
  
  if (validGroups.length === 0) {
    canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-pie"></i><p>Nema bodova za prikaz</p></div>';
    return;
  }
  
  const labels = validGroups.map(group => group.group);
  const pointsData = validGroups.map(group => parseInt(group.points));
  const agentsData = validGroups.map(group => parseInt(group.agents));
  
  const colors = ['#ff6384', '#36a2eb', '#ffce56'];
  const backgroundColors = colors.slice(0, validGroups.length);
  
  const ctx = canvas.getContext('2d');
  new Chart(ctx, {
    type: 'doughnut',
    data: {
      labels: labels,
      datasets: [{
        label: 'Bodovi po grupama',
        data: pointsData,
        backgroundColor: backgroundColors,
        borderWidth: 3,
        borderColor: '#ffffff',
        hoverBorderWidth: 4,
        hoverBorderColor: '#ffffff'
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '60%',
      plugins: {
        legend: {
          position: 'bottom',
          labels: {
            boxWidth: 15,
            boxHeight: 15,
            font: {
              size: 12,
              weight: '600'
            },
            color: '#495057',
            padding: 15,
            usePointStyle: true,
            pointStyle: 'circle',
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                return data.labels.map((label, i) => {
                  const points = data.datasets[0].data[i];
                  const agents = agentsData[i];
                  const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                  const percentage = ((points / total) * 100).toFixed(1);
                  return {
                    text: `${label}: ${points.toLocaleString()} bodova (${agents} djelatnika)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          backgroundColor: 'rgba(0, 0, 0, 0.8)',
          titleColor: '#ffffff',
          bodyColor: '#ffffff',
          borderColor: '#495057',
          borderWidth: 1,
          cornerRadius: 8,
          padding: 12,
          callbacks: {
            title: function(context) {
              return `Grupa: ${context[0].label}`;
            },
            label: function(context) {
              const points = context.raw;
              const agents = agentsData[context.dataIndex];
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = ((points / total) * 100).toFixed(1);
              return [
                `Bodovi: ${points.toLocaleString()} (${percentage}%)`,
                `Djelatnici: ${agents}`
              ];
            }
          }
        }
      }
    }
  });
}

/**
 * Helper function to format numbers
 */
function formatNumber(num) {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M';
  } else if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'K';
  }
  return num.toString();
}

/**
 * Helper function to generate colors
 */
function generateColors(count) {
  const baseColors = [
    '#ff6384', '#36a2eb', '#ffce56', '#4bc0c0', 
    '#9966ff', '#ff9f40', '#ff6384', '#c9cbcf'
  ];
  
  const colors = [];
  for (let i = 0; i < count; i++) {
    colors.push(baseColors[i % baseColors.length]);
  }
  
  return colors;
}

// Export functions for global access
window.AgentBonusBodovanjeCharts = {
  createMonthlyPointsChart: createMonthlyPointsChart,
  createGroupDistributionChart: createGroupDistributionChart,
  formatNumber: formatNumber
};
//# sourceMappingURL=agentbonusbodovanjecharts.2c410edf.js.map