// agentbonusbodovanjemybonus.js - Complete Personal Agent Dashboard
document.addEventListener('DOMContentLoaded', function() {
    // Initialize charts
    initializeMyBonusCharts();
    
    // Animate statistics values
    animateStatistics();
    
    // Setup period filter auto-submit
    setupPeriodFilter();
    
    // Add interactive effects
    addInteractiveEffects();
    
    // Setup progress bar animation
    animateProgressBar();
    
    // Initialize feedback functionality
    initializeFeedbackSystem();
    
    console.log('Complete My Bonus Dashboard initialized');
});

/**
 * Initialize feedback system for my bonus dashboard
 */
function initializeFeedbackSystem() {
    // View All Feedbacks Button
    const viewAllFeedbacksBtn = document.getElementById('viewAllFeedbacksBtn');
    if (viewAllFeedbacksBtn) {
        viewAllFeedbacksBtn.addEventListener('click', function() {
            openAllFeedbacksModal();
        });
    }
    
    // Close All Feedbacks Button
    const closeAllFeedbacksBtn = document.getElementById('closeAllFeedbacksBtn');
    if (closeAllFeedbacksBtn) {
        closeAllFeedbacksBtn.addEventListener('click', function() {
            closeModal(document.getElementById('allFeedbacksModal'));
        });
    }
    
    // Modal close functionality
    const allFeedbacksModal = document.getElementById('allFeedbacksModal');
    if (allFeedbacksModal) {
        const modalClose = allFeedbacksModal.querySelector('.modal-close');
        if (modalClose) {
            modalClose.addEventListener('click', function() {
                closeModal(allFeedbacksModal);
            });
        }
        
        // Close when clicking outside
        allFeedbacksModal.addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal(this);
            }
        });
    }
}

/**
 * Open all feedbacks modal
 */
function openAllFeedbacksModal() {
    const modal = document.getElementById('allFeedbacksModal');
    if (!modal) return;
    
    showModal(modal);
    
    // Load all feedbacks
    loadAllFeedbacks();
}

/**
 * Load all feedbacks for current user
 */
function loadAllFeedbacks() {
    const container = document.getElementById('feedbacksTimelineContainer');
    if (!container) return;
    
    // Show loading
    container.innerHTML = '<div class="loading-state"><i class="fas fa-spinner fa-spin"></i> Učitavanje feedbackova...</div>';
    
    fetch('/agentbonusbodovanje/my-feedback')
        .then(response => response.json())
        .then(data => {
            if (data.feedbacks && data.feedbacks.length > 0) {
                renderMyFeedbacksTimeline(data.feedbacks, container);
            } else {
                container.innerHTML = '<div class="empty-state"><i class="fas fa-comment-slash"></i><p>Nema feedbackova za prikaz</p></div>';
            }
        })
        .catch(error => {
            console.error('Error loading my feedbacks:', error);
            container.innerHTML = '<div class="error-state"><i class="fas fa-exclamation-triangle"></i><p>Greška pri učitavanju feedbackova</p></div>';
        });
}

/**
 * Render my feedbacks timeline (read-only)
 */
function renderMyFeedbacksTimeline(feedbacks, container) {
    const timeline = document.createElement('div');
    timeline.className = 'feedbacks-timeline';
    
    feedbacks.forEach(feedback => {
        const timelineItem = document.createElement('div');
        timelineItem.className = 'timeline-item';
        
        const date = new Date(feedback.datum);
        const formattedDate = date.toLocaleDateString('hr-HR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        
        let ratingHtml = '';
        if (feedback.ocjena) {
            const rating = parseFloat(feedback.ocjena);
            const ratingClass = rating >= 4 ? 'excellent' : (rating >= 3 ? 'good' : 'poor');
            const stars = Array.from({length: 5}, (_, i) => 
                `<i class="fas fa-star ${i < rating ? 'star-filled' : 'star-empty'}"></i>`
            ).join('');
            
            ratingHtml = `
                <div class="timeline-rating">
                    <span class="timeline-rating-value rating-${ratingClass}">${rating.toFixed(2)}/5.00</span>
                    <div class="timeline-stars">${stars}</div>
                </div>
            `;
        }
        
        timelineItem.innerHTML = `
            <div class="timeline-content">
                <div class="timeline-header">
                    <div class="timeline-date">${formattedDate}</div>
                </div>
                ${feedback.feedback ? `<div class="timeline-feedback">${escapeHtml(feedback.feedback).replace(/\n/g, '<br>')}</div>` : ''}
                ${ratingHtml}
                <div class="timeline-meta">
                    <i class="fas fa-user"></i>
                    <span>${escapeHtml(feedback.created_by)}</span>
                </div>
            </div>
        `;
        
        timeline.appendChild(timelineItem);
    });
    
    container.innerHTML = '';
    container.appendChild(timeline);
}

/**
 * Improve chart legend visibility and interaction
 */
function improveChartLegend() {
    // Find all chart legend items and add better visual feedback
    const legendItems = document.querySelectorAll('.chartjs-legend li');
    
    legendItems.forEach(item => {
        item.classList.add('chart-legend-item');
        
        // Add click feedback
        item.addEventListener('click', function() {
            setTimeout(() => {
                // Update visual state after chart updates
                const isHidden = this.style.textDecoration === 'line-through';
                if (isHidden) {
                    this.classList.add('chart-legend-hidden');
                } else {
                    this.classList.remove('chart-legend-hidden');
                }
            }, 100);
        });
    });
}

/**
 * Initialize charts for personal dashboard
 */
function initializeMyBonusCharts() {
    const chartsData = document.getElementById('my-bonus-charts-data');
    
    if (!chartsData) {
        console.error('My bonus charts data element not found!');
        return;
    }
    
    try {
        const historicalPerformance = JSON.parse(chartsData.getAttribute('data-historical-performance') || '[]');
        const serviceBreakdown = JSON.parse(chartsData.getAttribute('data-service-breakdown') || '[]');
        const rankingSystem = JSON.parse(chartsData.getAttribute('data-ranking-system') || '[]');
        
        // Create charts
        createHistoricalPerformanceChart(historicalPerformance);
        createServiceBreakdownChart(serviceBreakdown);
        
        // Improve legend after charts are created
        setTimeout(improveChartLegend, 500);
        
    } catch (error) {
        console.error('Error initializing my bonus charts:', error);
    }
}

/**
 * Create historical performance chart
 */
function createHistoricalPerformanceChart(historicalData) {
    const canvas = document.getElementById('historicalPerformanceChart');
    if (!canvas) return;
    
    if (!historicalData || historicalData.length === 0) {
        canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-line"></i><p>Nema podataka o povijesnim performansama</p></div>';
        return;
    }
    
    const labels = historicalData.map(item => `${item.month_name} ${item.year}`);
    const pointsData = historicalData.map(item => parseInt(item.total_points || 0));
    const bonusData = historicalData.map(item => parseFloat(item.bonus_amount || 0));
    
    const ctx = canvas.getContext('2d');
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [
                {
                    label: 'Bodovi',
                    data: pointsData,
                    backgroundColor: 'rgba(255, 215, 0, 0.15)',
                    borderColor: '#ffd700',
                    borderWidth: 4,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: '#ffd700',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 3,
                    pointRadius: 8,
                    pointHoverRadius: 10,
                    pointHoverBackgroundColor: '#ffed4e',
                    pointHoverBorderColor: '#ffffff',
                    pointHoverBorderWidth: 4,
                    yAxisID: 'y'
                },
                {
                    label: 'Bonus (€)',
                    data: bonusData,
                    backgroundColor: 'rgba(40, 167, 69, 0.15)',
                    borderColor: '#28a745',
                    borderWidth: 4,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: '#28a745',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 3,
                    pointRadius: 8,
                    pointHoverRadius: 10,
                    pointHoverBackgroundColor: '#20c997',
                    pointHoverBorderColor: '#ffffff',
                    pointHoverBorderWidth: 4,
                    yAxisID: 'y1'
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            interaction: {
                intersect: false,
                mode: 'index'
            },
            scales: {
                y: {
                    type: 'linear',
                    display: true,
                    position: 'left',
                    beginAtZero: true,
                    grid: {
                        color: 'rgba(255, 255, 255, 0.1)',
                        drawBorder: false
                    },
                    ticks: {
                        color: '#6c757d',
                        font: {
                            size: 12
                        },
                        callback: function(value) {
                            return value.toLocaleString();
                        }
                    },
                    title: {
                        display: true,
                        text: 'Bodovi',
                        color: '#ffd700',
                        font: {
                            size: 14,
                            weight: 'bold'
                        }
                    }
                },
                y1: {
                    type: 'linear',
                    display: true,
                    position: 'right',
                    beginAtZero: true,
                    grid: {
                        drawOnChartArea: false,
                    },
                    ticks: {
                        color: '#6c757d',
                        font: {
                            size: 12
                        },
                        callback: function(value) {
                            return value.toLocaleString() + ' €';
                        }
                    },
                    title: {
                        display: true,
                        text: 'Bonus (€)',
                        color: '#28a745',
                        font: {
                            size: 14,
                            weight: 'bold'
                        }
                    }
                },
                x: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        color: '#6c757d',
                        font: {
                            size: 12
                        },
                        maxRotation: 45
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'top',
                    labels: {
                        boxWidth: 20,
                        boxHeight: 20,
                        font: {
                            size: 14,
                            weight: '600'
                        },
                        color: '#495057',
                        padding: 20,
                        usePointStyle: true,
                        pointStyle: 'circle'
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.9)',
                    titleColor: '#ffffff',
                    bodyColor: '#ffffff',
                    borderColor: '#ffd700',
                    borderWidth: 2,
                    cornerRadius: 10,
                    padding: 15,
                    titleFont: {
                        size: 14,
                        weight: 'bold'
                    },
                    bodyFont: {
                        size: 13
                    },
                    callbacks: {
                        title: function(context) {
                            return `Mjesec: ${context[0].label}`;
                        },
                        label: function(context) {
                            if (context.datasetIndex === 0) {
                                return `Bodovi: ${context.raw.toLocaleString()}`;
                            } else {
                                return `Bonus: ${context.raw.toLocaleString()} €`;
                            }
                        }
                    }
                }
            }
        }
    });
}

/**
 * Create service breakdown chart (doughnut)
 */
function createServiceBreakdownChart(serviceData) {
    const canvas = document.getElementById('serviceBreakdownChart');
    if (!canvas) return;
    
    if (!serviceData || serviceData.length === 0) {
        canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-pie"></i><p>Nema podataka o uslugama za prikaz</p></div>';
        return;
    }
    
    // Filter out services with zero points
    const validServices = serviceData.filter(service => service.total_points > 0);
    
    if (validServices.length === 0) {
        canvas.parentElement.innerHTML = '<div class="empty-state"><i class="fas fa-chart-pie"></i><p>Nema bodova za prikaz</p></div>';
        return;
    }
    
    const labels = validServices.map(service => service.tarifa);
    const pointsData = validServices.map(service => parseInt(service.total_points));
    const countData = validServices.map(service => parseInt(service.count));
    
    // Generate colors
    const colors = generateServiceColors(validServices.length);
    
    const ctx = canvas.getContext('2d');
    const chart = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: labels,
            datasets: [{
                label: 'Bodovi po uslugama',
                data: pointsData,
                backgroundColor: colors,
                borderWidth: 3,
                borderColor: '#ffffff',
                hoverBorderWidth: 5,
                hoverBorderColor: '#ffffff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '65%',
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        boxWidth: 18,
                        boxHeight: 18,
                        font: {
                            size: 12,
                            weight: '600'
                        },
                        color: '#495057',
                        padding: 15,
                        usePointStyle: true,
                        pointStyle: 'circle',
                        generateLabels: function(chart) {
                            const data = chart.data;
                            if (data.labels.length && data.datasets.length) {
                                return data.labels.map((label, i) => {
                                    const dataset = data.datasets[0];
                                    const points = dataset.data[i];
                                    const count = countData[i];
                                    const meta = chart.getDatasetMeta(0);
                                    const hidden = meta.data[i] ? meta.data[i].hidden : false;
                                    
                                    return {
                                        text: `${label}: ${points.toLocaleString()} bodova (${count} kom)`,
                                        fillStyle: hidden ? '#cccccc' : dataset.backgroundColor[i],
                                        strokeStyle: hidden ? '#999999' : dataset.backgroundColor[i],
                                        lineWidth: hidden ? 1 : 2,
                                        hidden: hidden,
                                        index: i,
                                        fontColor: hidden ? '#999999' : '#495057'
                                    };
                                });
                            }
                            return [];
                        }
                    },
                    onClick: function(e, legendItem, legend) {
                        const index = legendItem.index;
                        const chart = legend.chart;
                        const meta = chart.getDatasetMeta(0);
                        
                        // Toggle visibility
                        meta.data[index].hidden = !meta.data[index].hidden;
                        
                        // Update chart
                        chart.update();
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.9)',
                    titleColor: '#ffffff',
                    bodyColor: '#ffffff',
                    borderColor: '#495057',
                    borderWidth: 2,
                    cornerRadius: 10,
                    padding: 15,
                    titleFont: {
                        size: 14,
                        weight: 'bold'
                    },
                    bodyFont: {
                        size: 13
                    },
                    callbacks: {
                        title: function(context) {
                            return `Usluga: ${context[0].label}`;
                        },
                        label: function(context) {
                            const points = context.raw;
                            const count = countData[context.dataIndex];
                            const total = context.dataset.data.reduce((acc, val, idx) => {
                                const meta = context.chart.getDatasetMeta(0);
                                return meta.data[idx].hidden ? acc : acc + val;
                            }, 0);
                            const percentage = total > 0 ? ((points / total) * 100).toFixed(1) : 0;
                            return [
                                `Bodovi: ${points.toLocaleString()} (${percentage}%)`,
                                `Količina: ${count} kom`
                            ];
                        }
                    }
                }
            }
        }
    });
    
    // Store chart instance for external access
    window.serviceBreakdownChart = chart;
}

/**
 * Generate colors for service breakdown chart
 */
function generateServiceColors(count) {
    const baseColors = [
        '#ff6384', '#36a2eb', '#ffce56', '#4bc0c0', 
        '#9966ff', '#ff9f40', '#ff6384', '#c9cbcf',
        '#4bc0c0', '#ff9f40', '#ff6384', '#36a2eb'
    ];
    
    const colors = [];
    for (let i = 0; i < count; i++) {
        colors.push(baseColors[i % baseColors.length]);
    }
    
    return colors;
}

/**
 * Animate statistics values on page load
 */
function animateStatistics() {
    // Animate main stat cards
    const statValues = document.querySelectorAll('.stat-value');
    statValues.forEach(valueElement => {
        let originalText = valueElement.textContent;
        
        // Skip rank values (non-numeric)
        if (originalText.match(/^[a-zA-Z]/)) {
            return;
        }
        
        // Extract numeric value
        let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
        if (!isNaN(numericValue)) {
            if (originalText.includes('€')) {
                // EUR value animation
                animateEuroValue(valueElement, 0, numericValue, 1500);
            } else {
                // Simple integer animation for points and counts
                animateValue(valueElement, 0, numericValue, 1200);
            }
        }
    });
    
    // Animate service table values
    const serviceValues = document.querySelectorAll('.service-total');
    serviceValues.forEach(valueElement => {
        let originalText = valueElement.textContent;
        let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
        
        if (!isNaN(numericValue)) {
            animateValue(valueElement, 0, numericValue, 1000);
        }
    });
    
    // Animate feedback stats
    const feedbackStatNumbers = document.querySelectorAll('.feedback-stat-item .stat-number');
    feedbackStatNumbers.forEach(valueElement => {
        let originalText = valueElement.textContent;
        
        // Skip non-numeric values like dates
        if (originalText.match(/\d{2}\.\d{2}\.\d{4}/)) {
            return;
        }
        
        let numericValue = parseFloat(originalText.replace(/[^\d.-]/g, ''));
        if (!isNaN(numericValue)) {
            animateValue(valueElement, 0, numericValue, 1000);
        }
    });
}

/**
 * Animate counter value from start to end
 */
function animateValue(element, start, end, duration) {
    let startTimestamp = null;
    
    const step = (timestamp) => {
        if (!startTimestamp) startTimestamp = timestamp;
        const progress = Math.min((timestamp - startTimestamp) / duration, 1);
        const value = Math.floor(progress * (end - start) + start);
        
        element.textContent = value.toLocaleString();
        
        if (progress < 1) {
            window.requestAnimationFrame(step);
        }
    };
    
    window.requestAnimationFrame(step);
}

/**
 * Animate EUR value from start to end
 */
function animateEuroValue(element, start, end, duration) {
    let startTimestamp = null;
    
    const step = (timestamp) => {
        if (!startTimestamp) startTimestamp = timestamp;
        const progress = Math.min((timestamp - startTimestamp) / duration, 1);
        const value = progress * (end - start) + start;
        
        // Format as currency
        const formattedValue = value.toLocaleString('hr-HR', {
            minimumFractionDigits: 0,
            maximumFractionDigits: 0
        });
        
        element.textContent = formattedValue + ' €';
        
        if (progress < 1) {
            window.requestAnimationFrame(step);
        }
    };
    
    window.requestAnimationFrame(step);
}

/**
 * Setup period filter auto-submit functionality
 */
function setupPeriodFilter() {
    const monthSelect = document.getElementById('month');
    const yearSelect = document.getElementById('year');
    
    if (monthSelect && yearSelect) {
        monthSelect.addEventListener('change', function() {
            this.form.submit();
        });
        
        yearSelect.addEventListener('change', function() {
            this.form.submit();
        });
    }
}

/**
 * Add interactive effects to various elements
 */
function addInteractiveEffects() {
    // Stat cards hover effects - using CSS classes
    const statCards = document.querySelectorAll('.stat-card');
    statCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.classList.add('stat-card-hover');
        });
        
        card.addEventListener('mouseleave', function() {
            this.classList.remove('stat-card-hover');
        });
    });

    // Top agent hover effects - using CSS classes
    const topAgents = document.querySelectorAll('.top-agent');
    topAgents.forEach(agent => {
        agent.addEventListener('mouseenter', function() {
            this.classList.add('top-agent-hover');
        });
        
        agent.addEventListener('mouseleave', function() {
            this.classList.remove('top-agent-hover');
        });
    });

    // Rank item hover effects - using CSS classes
    const rankItems = document.querySelectorAll('.rank-item');
    rankItems.forEach(item => {
        item.addEventListener('mouseenter', function() {
            this.classList.add('rank-item-hover');
        });
        
        item.addEventListener('mouseleave', function() {
            this.classList.remove('rank-item-hover');
        });
    });

    // Position badge rotation effect - using CSS classes
    const positionBadges = document.querySelectorAll('.position-badge');
    positionBadges.forEach(badge => {
        badge.addEventListener('mouseenter', function() {
            this.classList.add('position-badge-hover');
        });
        
        badge.addEventListener('mouseleave', function() {
            this.classList.remove('position-badge-hover');
        });
    });

    // Rank badge pulse effect - using CSS classes
    const rankBadges = document.querySelectorAll('.rank-badge');
    rankBadges.forEach(badge => {
        if (!badge.classList.contains('rank-nema')) {
            badge.addEventListener('mouseenter', function() {
                this.classList.add('rank-badge-hover');
            });
            
            badge.addEventListener('mouseleave', function() {
                this.classList.remove('rank-badge-hover');
            });
        }
    });
    
    // Feedback view all button hover effect
    const feedbackViewAllBtn = document.querySelector('.feedback-view-all-btn');
    if (feedbackViewAllBtn) {
        feedbackViewAllBtn.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-3px)';
            this.style.boxShadow = '0 8px 25px rgba(54, 162, 235, 0.4)';
        });
        
        feedbackViewAllBtn.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(-2px)';
            this.style.boxShadow = '0 6px 20px rgba(54, 162, 235, 0.3)';
        });
    }
}

/**
 * Animate progress bar on page load
 */
function animateProgressBar() {
    const progressFill = document.querySelector('.progress-fill');
    if (progressFill) {
        const targetProgress = progressFill.getAttribute('data-progress');
        if (targetProgress) {
            const targetWidth = parseFloat(targetProgress);
            
            // Set initial width to 0
            progressFill.style.width = '0%';
            
            // Animate to target width after delay
            setTimeout(() => {
                progressFill.style.width = targetWidth + '%';
            }, 500);
        }
    }
}

/**
 * Show modal
 */
function showModal(modal) {
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
    
    // Add show class for animation
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
    
    // Focus first input
    const firstInput = modal.querySelector('input, textarea');
    if (firstInput) {
        setTimeout(() => firstInput.focus(), 100);
    }
}

/**
 * Close modal
 */
function closeModal(modal) {
    modal.classList.remove('show');
    document.body.style.overflow = '';
    
    // Hide after animation
    setTimeout(() => {
        modal.style.display = 'none';
    }, 300);
}

/**
 * Refresh agent statistics (if needed for real-time updates)
 */
function refreshAgentStats() {
    const urlParams = new URLSearchParams(window.location.search);
    const month = urlParams.get('month') || new Date().getMonth() + 1;
    const year = urlParams.get('year') || new Date().getFullYear();
    
    fetch('/agentbonusbodovanje/api/my-stats', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            month: month,
            year: year
        })
    })
        .then(response => response.json())
        .then(data => {
            if (data && !data.error) {
                updateAgentStats(data);
                console.log('Agent stats refreshed successfully');
            } else {
                console.error('Error in response:', data.error);
            }
        })
        .catch(error => {
            console.error('Error refreshing agent stats:', error);
        });
}

/**
 * Update agent statistics with new data
 */
function updateAgentStats(data) {
    // Update main stat cards
    if (data.total_points !== undefined) {
        updateValue('.points-card .stat-value', data.total_points);
    }
    
    if (data.bonus_amount !== undefined) {
        updateEuroValue('.bonus-card .stat-value', data.bonus_amount);
    }
    
    if (data.current_rank) {
        const rankElement = document.querySelector('.rank-card .stat-value');
        if (rankElement) {
            rankElement.textContent = data.current_rank;
            rankElement.className = `stat-value rank-${data.current_rank.toLowerCase()}`;
        }
    }
    
    if (data.group_position !== undefined) {
        updateValue('.position-card .stat-value', data.group_position);
    }
}

/**
 * Helper to update simple values
 */
function updateValue(selector, value) {
    const element = document.querySelector(selector);
    if (element) {
        animateValue(element, 0, value, 800);
    }
}

/**
 * Helper to update EUR values
 */
function updateEuroValue(selector, value) {
    const element = document.querySelector(selector);
    if (element) {
        animateEuroValue(element, 0, value, 800);
    }
}

/**
 * Show success notification
 */
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i>
        <span>${message}</span>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.add('show');
    }, 100);
    
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

/**
 * Format number with thousands separator
 */
function formatNumber(num) {
    return num.toLocaleString('hr-HR');
}

/**
 * Helper function to escape HTML
 */
function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}

/**
 * Auto-resize textareas
 */
function autoResizeTextarea(textarea) {
    textarea.style.height = 'auto';
    textarea.style.height = (textarea.scrollHeight) + 'px';
}

/**
 * Initialize auto-resize for textareas
 */
function initializeAutoResize() {
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        textarea.addEventListener('input', function() {
            autoResizeTextarea(this);
        });
        
        // Initial resize
        autoResizeTextarea(textarea);
    });
}

/**
 * Handle responsive behavior
 */
function handleResponsiveBehavior() {
    const dashboard = document.querySelector('.my-bonus-dashboard');
    if (!dashboard) return;
    
    function checkScreenSize() {
        const isMobile = window.innerWidth <= 768;
        const isTablet = window.innerWidth <= 1200 && window.innerWidth > 768;
        
        if (isMobile) {
            dashboard.classList.add('mobile-view');
            dashboard.classList.remove('tablet-view', 'desktop-view');
        } else if (isTablet) {
            dashboard.classList.add('tablet-view');
            dashboard.classList.remove('mobile-view', 'desktop-view');
        } else {
            dashboard.classList.add('desktop-view');
            dashboard.classList.remove('mobile-view', 'tablet-view');
        }
    }
    
    // Check on load
    checkScreenSize();
    
    // Check on resize
    window.addEventListener('resize', checkScreenSize);
}

/**
 * Initialize keyboard shortcuts
 */
function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // ESC key to close modals
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal[style*="display: block"]');
            if (openModal) {
                closeModal(openModal);
            }
        }
        
        // F key to open feedback modal (if available)
        if (e.key === 'f' || e.key === 'F') {
            if (!e.ctrlKey && !e.altKey && !e.metaKey) {
                const feedbackBtn = document.getElementById('viewAllFeedbacksBtn');
                if (feedbackBtn && feedbackBtn.style.display !== 'none') {
                    e.preventDefault();
                    feedbackBtn.click();
                }
            }
        }
        
        // R key to refresh stats
        if (e.key === 'r' || e.key === 'R') {
            if (e.ctrlKey || e.metaKey) {
                // Allow normal browser refresh
                return;
            }
            if (!e.altKey) {
                e.preventDefault();
                refreshAgentStats();
                showNotification('Statistike osvježene!', 'success');
            }
        }
    });
}

/**
 * Initialize theme detection and adaptation
 */
function initializeThemeAdaptation() {
    // Check for dark mode preference
    const prefersDarkMode = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    if (prefersDarkMode) {
        document.body.classList.add('dark-mode-preferred');
    }
    
    // Listen for theme changes
    if (window.matchMedia) {
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', function(e) {
            if (e.matches) {
                document.body.classList.add('dark-mode-preferred');
            } else {
                document.body.classList.remove('dark-mode-preferred');
            }
            
            // Re-initialize charts if needed
            setTimeout(() => {
                initializeMyBonusCharts();
            }, 100);
        });
    }
}

/**
 * Initialize performance monitoring
 */
function initializePerformanceMonitoring() {
    // Monitor chart loading times
    const chartLoadStart = performance.now();
    
    // Monitor when charts are fully loaded
    setTimeout(() => {
        const chartLoadEnd = performance.now();
        const loadTime = chartLoadEnd - chartLoadStart;
        
        if (loadTime > 2000) {
            console.warn('Charts took longer than expected to load:', loadTime + 'ms');
        }
        
        // Log performance metrics
        console.log('Dashboard performance metrics:', {
            chartLoadTime: loadTime + 'ms',
            totalElements: document.querySelectorAll('*').length,
            memoryUsage: performance.memory ? {
                used: Math.round(performance.memory.usedJSHeapSize / 1024 / 1024) + 'MB',
                total: Math.round(performance.memory.totalJSHeapSize / 1024 / 1024) + 'MB'
            } : 'Not available'
        });
    }, 1000);
}

/**
 * Initialize accessibility features
 */
function initializeAccessibilityFeatures() {
    // Add ARIA labels to interactive elements
    const interactiveElements = document.querySelectorAll('button, .stat-card, .rank-item');
    
    interactiveElements.forEach(element => {
        if (!element.getAttribute('aria-label') && !element.getAttribute('aria-labelledby')) {
            const text = element.textContent.trim();
            if (text) {
                element.setAttribute('aria-label', text);
            }
        }
    });
    
    // Add keyboard navigation for stat cards
    const statCards = document.querySelectorAll('.stat-card');
    statCards.forEach((card, index) => {
        card.setAttribute('tabindex', '0');
        card.setAttribute('role', 'button');
        
        card.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                // Trigger card interaction if any
                card.click();
            }
        });
    });
    
    // Announce dynamic content changes to screen readers
    const announceElement = document.createElement('div');
    announceElement.setAttribute('aria-live', 'polite');
    announceElement.setAttribute('aria-atomic', 'true');
    announceElement.className = 'sr-only';
    document.body.appendChild(announceElement);
    
    // Store reference for announcements
    window.announceToScreenReader = function(message) {
        announceElement.textContent = message;
        setTimeout(() => {
            announceElement.textContent = '';
        }, 1000);
    };
}

/**
 * Initialize tooltips for complex elements
 */
function initializeTooltips() {
    // Add tooltips to rank badges
    const rankBadges = document.querySelectorAll('.rank-badge, .ocjena-badge');
    
    rankBadges.forEach(badge => {
        const rank = badge.textContent.trim();
        
        if (rank && rank !== 'Nema') {
            let tooltipText = '';
            
            switch(rank.toLowerCase()) {
                case 'bronza':
                    tooltipText = 'Osnovni razred - dobra performansa';
                    break;
                case 'srebro':
                    tooltipText = 'Srednji razred - vrlo dobra performansa';
                    break;
                case 'zlato':
                    tooltipText = 'Visoki razred - odlična performansa';
                    break;
                case 'platina':
                    tooltipText = 'Najviši razred - izuzetna performansa';
                    break;
                case 'dijamant':
                    tooltipText = 'Elitni razred - izvanredna performansa';
                    break;
                default:
                    if (rank.includes('.')) {
                        const rating = parseFloat(rank);
                        if (rating >= 4.5) {
                            tooltipText = 'Izuzetna ocjena - nastavite odličan rad!';
                        } else if (rating >= 4.0) {
                            tooltipText = 'Vrlo dobra ocjena - odličan rad!';
                        } else if (rating >= 3.0) {
                            tooltipText = 'Dobra ocjena - ima prostora za napredak';
                        } else {
                            tooltipText = 'Ocjena ispod očekivanja - potrebno poboljšanje';
                        }
                    }
            }
            
            if (tooltipText) {
                badge.setAttribute('title', tooltipText);
                badge.setAttribute('data-tooltip', tooltipText);
            }
        }
    });
}

/**
 * Initialize feedback-specific enhancements
 */
function initializeFeedbackEnhancements() {
    // Add smooth scrolling to feedback timeline
    const feedbackTimeline = document.querySelector('.feedbacks-timeline');
    if (feedbackTimeline) {
        feedbackTimeline.style.scrollBehavior = 'smooth';
    }
    
    // Add reading time estimates for long feedback texts
    const feedbackTexts = document.querySelectorAll('.timeline-feedback, .feedback-text');
    
    feedbackTexts.forEach(text => {
        const wordCount = text.textContent.split(' ').length;
        const readingTime = Math.ceil(wordCount / 200); // Average reading speed: 200 words/minute
        
        if (readingTime > 1) {
            const timeEstimate = document.createElement('small');
            timeEstimate.className = 'reading-time';
            timeEstimate.textContent = `~${readingTime} min čitanja`;
            timeEstimate.style.color = 'var(--text-secondary)';
            timeEstimate.style.fontSize = '0.75rem';
            timeEstimate.style.fontStyle = 'italic';
            
            text.appendChild(timeEstimate);
        }
    });
    
    // Add feedback sentiment indicators
    const sentimentKeywords = {
        positive: ['odličan', 'sjajan', 'izvrsno', 'bravo', 'super', 'fantastično', 'uspješno', 'pohvala'],
        negative: ['loše', 'problem', 'greška', 'nedostatak', 'poboljšanje', 'nedovoljno', 'slabo'],
        neutral: ['napredak', 'razvoj', 'učenje', 'mogućnost', 'prilika']
    };
    
    feedbackTexts.forEach(text => {
        const content = text.textContent.toLowerCase();
        let sentiment = 'neutral';
        let sentimentScore = 0;
        
        // Simple sentiment analysis
        Object.entries(sentimentKeywords).forEach(([type, keywords]) => {
            keywords.forEach(keyword => {
                if (content.includes(keyword)) {
                    if (type === 'positive') sentimentScore += 1;
                    else if (type === 'negative') sentimentScore -= 1;
                }
            });
        });
        
        if (sentimentScore > 0) sentiment = 'positive';
        else if (sentimentScore < 0) sentiment = 'negative';
        
        text.classList.add(`sentiment-${sentiment}`);
        text.setAttribute('data-sentiment', sentiment);
    });
}

/**
 * Initialize error handling and recovery
 */
function initializeErrorHandling() {
    // Global error handler
    window.addEventListener('error', function(event) {
        console.error('Dashboard error:', event.error);
        
        // Try to recover from chart errors
        if (event.error.message && event.error.message.includes('Chart')) {
            setTimeout(() => {
                console.log('Attempting to recover charts...');
                initializeMyBonusCharts();
            }, 1000);
        }
    });
    
    // Promise rejection handler
    window.addEventListener('unhandledrejection', function(event) {
        console.error('Unhandled promise rejection:', event.reason);
        
        // Handle failed feedback loading
        if (event.reason && event.reason.message && event.reason.message.includes('feedback')) {
            showNotification('Greška pri učitavanju feedback-a. Molimo pokušajte ponovno.', 'error');
        }
    });
}

/**
 * Initialize data export functionality
 */
function initializeDataExport() {
    // Add export button if user has high enough privileges
    const currentUser = window.currentUser || {};
    
    if (currentUser.hasExportPermission) {
        const exportBtn = document.createElement('button');
        exportBtn.className = 'export-data-btn';
        exportBtn.innerHTML = '<i class="fas fa-download"></i> Izvezi moje podatke';
        exportBtn.addEventListener('click', exportUserData);
        
        const headerActions = document.querySelector('.header-actions');
        if (headerActions) {
            headerActions.appendChild(exportBtn);
        }
    }
}

/**
 * Export user data (if permitted)
 */
function exportUserData() {
    const urlParams = new URLSearchParams(window.location.search);
    const month = urlParams.get('month') || new Date().getMonth() + 1;
    const year = urlParams.get('year') || new Date().getFullYear();
    
    showNotification('Priprema izvoza podataka...', 'success');
    
    // Simulate data export
    setTimeout(() => {
        const data = {
            agent_name: window.agentName || 'Unknown',
            period: `${year}-${month}`,
            exported_at: new Date().toISOString(),
            // Add more data as needed
        };
        
        const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `my_bonus_data_${year}_${month}.json`;
        a.click();
        URL.revokeObjectURL(url);
        
        showNotification('Podaci uspješno izvezeni!', 'success');
    }, 1500);
}

/**
 * Initialize print functionality
 */
function initializePrintFunctionality() {
    // Add print styles
    const printStyles = `
        @media print {
            .my-bonus-dashboard {
                padding: 0;
                background: white !important;
                color: black !important;
            }
            
            .modal, .notification {
                display: none !important;
            }
            
            .chart-body canvas {
                max-height: 200px !important;
            }
            
            .feedback-view-all-btn {
                display: none !important;
            }
        }
    `;
    
    const styleSheet = document.createElement('style');
    styleSheet.textContent = printStyles;
    document.head.appendChild(styleSheet);
    
    // Add print button
    const printBtn = document.createElement('button');
    printBtn.className = 'print-dashboard-btn';
    printBtn.innerHTML = '<i class="fas fa-print"></i> Ispiši';
    printBtn.addEventListener('click', () => window.print());
    
    const headerActions = document.querySelector('.header-actions');
    if (headerActions) {
        headerActions.appendChild(printBtn);
    }
}

/**
 * Final initialization
 */
document.addEventListener('DOMContentLoaded', function() {
    // Initialize additional features
    handleResponsiveBehavior();
    initializeKeyboardShortcuts();
    initializeThemeAdaptation();
    initializePerformanceMonitoring();
    initializeAccessibilityFeatures();
    initializeTooltips();
    initializeFeedbackEnhancements();
    initializeErrorHandling();
    initializeDataExport();
    initializePrintFunctionality();
    initializeAutoResize();
    
    console.log('All My Bonus Dashboard features initialized successfully!');
});

// Export functions for global access
window.MyBonusDashboard = {
    refreshStats: refreshAgentStats,
    updateValue: updateValue,
    updateEuroValue: updateEuroValue,
    showNotification: showNotification,
    formatNumber: formatNumber,
    escapeHtml: escapeHtml,
    showModal: showModal,
    closeModal: closeModal,
    loadAllFeedbacks: loadAllFeedbacks
};
//# sourceMappingURL=agentbonusbodovanjemybonus.1a8cebba.js.map