/**
 * Complete Osoba Edit with Feedback System - CSRF FIXED VERSION
 */
document.addEventListener('DOMContentLoaded', function() {
    // Initialize existing functionality
    initializeToggleSwitch();
    initializeCharacterCounters();
    initializeCancelButton();
    initializeAutoResize();
    
    // Initialize new feedback functionality
    initializeFeedbackSystem();
    initializeRatingInputs();
    initializeModals();
    
    console.log('Complete Osoba Edit initialized');
});

/**
 * Initialize toggle switch functionality for visibility
 */
function initializeToggleSwitch() {
    const toggleSwitch = document.getElementById('vidljivost-toggle');
    const vidljivostInput = document.getElementById('vidljivost');
    
    if (toggleSwitch && vidljivostInput) {
        // Set initial state
        vidljivostInput.value = toggleSwitch.checked ? '1' : '0';
        
        // Update on change
        toggleSwitch.addEventListener('change', function() {
            vidljivostInput.value = this.checked ? '1' : '0';
        });
    }
}

/**
 * Initialize character counters for all textareas
 */
function initializeCharacterCounters() {
    const textareas = [
        { element: 'newFeedback', counter: 'newFeedbackCounter', max: 2000 },
        { element: 'editFeedback', counter: 'editFeedbackCounter', max: 2000 }
    ];
    
    textareas.forEach(textarea => {
        const element = document.getElementById(textarea.element);
        const counter = document.getElementById(textarea.counter);
        
        if (element && counter) {
            // Update counter on load
            updateCharacterCounter(element, counter, textarea.max);
            
            // Update counter on input
            element.addEventListener('input', function() {
                updateCharacterCounter(this, counter, textarea.max);
            });
        }
    });
}

/**
 * Update character counter
 */
function updateCharacterCounter(element, counter, maxLength) {
    const currentLength = element.value.length;
    
    counter.textContent = `${currentLength}/${maxLength}`;
    
    // Add warning/danger classes based on character count
    counter.classList.remove('warning', 'danger');
    
    if (currentLength > maxLength * 0.9) {
        counter.classList.add('danger');
    } else if (currentLength > maxLength * 0.8) {
        counter.classList.add('warning');
    }
    
    // Prevent typing if max length exceeded
    if (currentLength > maxLength) {
        element.value = element.value.substring(0, maxLength);
        counter.textContent = `${maxLength}/${maxLength}`;
        counter.classList.add('danger');
    }
}

/**
 * Initialize cancel button functionality
 */
function initializeCancelButton() {
    const cancelButton = document.getElementById('cancelButton');
    if (cancelButton) {
        cancelButton.addEventListener('click', function() {
            // Check if form has been modified
            const form = document.querySelector('.modern-form');
            if (form) {
                const formData = new FormData(form);
                let hasChanges = false;
                
                // Simple check for changes
                for (let [key, value] of formData.entries()) {
                    if (value.trim() !== '' && key !== 'edit_token') {
                        hasChanges = true;
                        break;
                    }
                }
                
                if (hasChanges) {
                    if (confirm('Imate nesačuvane promjene. Jeste li sigurni da se želite vratiti?')) {
                        window.history.back();
                    }
                } else {
                    window.history.back();
                }
            } else {
                window.history.back();
            }
        });
    }
}

/**
 * Initialize auto-resize for textareas
 */
function initializeAutoResize() {
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        textarea.addEventListener('input', function() {
            autoResizeTextarea(this);
        });
        
        // Initial resize
        autoResizeTextarea(textarea);
    });
}

/**
 * Auto-resize textarea
 */
function autoResizeTextarea(textarea) {
    textarea.style.height = 'auto';
    textarea.style.height = (textarea.scrollHeight) + 'px';
}

/**
 * Initialize feedback system
 */
function initializeFeedbackSystem() {
    // Add Feedback Button
    const addFeedbackBtn = document.getElementById('addFeedbackBtn');
    if (addFeedbackBtn) {
        addFeedbackBtn.addEventListener('click', function() {
            openAddFeedbackModal();
        });
    }
    
    // View All Feedbacks Button
    const viewAllFeedbacksBtn = document.getElementById('viewAllFeedbacksBtn');
    if (viewAllFeedbacksBtn) {
        viewAllFeedbacksBtn.addEventListener('click', function() {
            const personName = document.querySelector('input[name="osoba[name]"]')?.value;
            if (personName) {
                openAllFeedbacksModal(personName);
            }
        });
    }
    
    // Save Feedback Button
    const saveFeedbackBtn = document.getElementById('saveFeedbackBtn');
    if (saveFeedbackBtn) {
        saveFeedbackBtn.addEventListener('click', function() {
            saveFeedback();
        });
    }
    
    // Update Feedback Button
    const updateFeedbackBtn = document.getElementById('updateFeedbackBtn');
    if (updateFeedbackBtn) {
        updateFeedbackBtn.addEventListener('click', function() {
            updateFeedback();
        });
    }
}

/**
 * Initialize rating inputs with star previews
 */
function initializeRatingInputs() {
    const ratingInputs = [
        { input: 'newOcjena', preview: 'newRatingPreview' },
        { input: 'editOcjena', preview: 'editRatingPreview' }
    ];
    
    ratingInputs.forEach(rating => {
        const input = document.getElementById(rating.input);
        const preview = document.getElementById(rating.preview);
        
        if (input && preview) {
            // Update stars on input change
            input.addEventListener('input', function() {
                updateStarRating(this.value, preview);
            });
            
            // Make stars clickable
            const stars = preview.querySelectorAll('.stars i');
            stars.forEach((star, index) => {
                star.addEventListener('click', function() {
                    const rating = index + 1;
                    input.value = rating.toFixed(1);
                    updateStarRating(rating, preview);
                });
                
                star.addEventListener('mouseenter', function() {
                    const rating = index + 1;
                    updateStarRating(rating, preview, true);
                });
                
                star.addEventListener('mouseleave', function() {
                    updateStarRating(input.value, preview);
                });
            });
        }
    });
}

/**
 * Update star rating display
 */
function updateStarRating(rating, previewElement, isHover = false) {
    if (!previewElement) return;
    
    const stars = previewElement.querySelectorAll('.stars i');
    const ratingValue = parseFloat(rating) || 0;
    
    stars.forEach((star, index) => {
        star.classList.remove('star-filled', 'star-empty', 'star-hover');
        
        if (isHover) {
            if (index < ratingValue) {
                star.classList.add('star-hover');
            } else {
                star.classList.add('star-empty');
            }
        } else {
            if (index < ratingValue) {
                star.classList.add('star-filled');
            } else {
                star.classList.add('star-empty');
            }
        }
    });
}

/**
 * Open add feedback modal
 */
function openAddFeedbackModal() {
    const modal = document.getElementById('addFeedbackModal');
    const personName = document.querySelector('input[name="osoba[name]"]')?.value;
    
    if (modal && personName) {
        // Clear form
        const form = document.getElementById('addFeedbackForm');
        if (form) {
            form.reset();
        }
        
        const counter = document.getElementById('newFeedbackCounter');
        if (counter) {
            counter.textContent = '0/2000';
            counter.classList.remove('warning', 'danger');
        }
        
        const preview = document.getElementById('newRatingPreview');
        if (preview) {
            updateStarRating(0, preview);
        }
        
        // Set person name
        const personInput = document.querySelector('input[name="person_name"]');
        if (personInput) {
            personInput.value = personName;
        }
        
        showModal(modal);
    }
}

/**
 * Open all feedbacks modal - FIXED to use same modal as list page
 */
function openAllFeedbacksModal(personName) {
    // Try to find the timeline modal first (same as list page)
    let modal = document.getElementById('viewFeedbacksModal');
    
    // If timeline modal doesn't exist, use the simple modal
    if (!modal) {
        modal = document.getElementById('allFeedbacksModal');
    }
    
    if (!modal) {
        console.error('No feedback modal found!');
        return;
    }
    
    // Set person name in modal title
    const personNameElement = modal.querySelector('#viewFeedbacksPersonName, h3');
    if (personNameElement) {
        if (personNameElement.id === 'viewFeedbacksPersonName') {
            personNameElement.textContent = personName;
        } else {
            // Update the h3 title
            personNameElement.innerHTML = `<i class="fas fa-history"></i> Svi Feedbackovi - ${personName}`;
        }
    }
    
    showModal(modal);
    
    // Load feedbacks using appropriate container
    if (modal.id === 'viewFeedbacksModal') {
        loadFeedbacksTimelineForViewModal(personName);
    } else {
        loadFeedbacksTimeline(personName);
    }
}

/**
 * Load feedbacks timeline for view modal (timeline style)
 */
function loadFeedbacksTimelineForViewModal(personName) {
    const container = document.getElementById('feedbacksTimelineContainer');
    if (!container) {
        console.error('Timeline container not found, falling back to simple modal');
        loadFeedbacksTimeline(personName);
        return;
    }
    
    console.log('Loading timeline feedbacks for:', personName);
    
    // Show loading
    container.innerHTML = '<div class="loading-state"><i class="fas fa-spinner fa-spin"></i> Učitavanje feedbackova...</div>';
    
    // Use fetchWithCsrf
    window.fetchWithCsrf(`/saleosoba/feedbacks?person_name=${encodeURIComponent(personName)}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        console.log('Timeline feedbacks data received:', data);
        if (data.feedbacks && data.feedbacks.length > 0) {
            renderFeedbacksTimeline(data.feedbacks, container, data.can_manage);
        } else {
            container.innerHTML = '<div class="empty-state"><i class="fas fa-comment-slash"></i><p>Nema feedbackova za ovu osobu</p></div>';
        }
    })
    .catch(error => {
        console.error('Error loading timeline feedbacks:', error);
        container.innerHTML = '<div class="error-state"><i class="fas fa-exclamation-triangle"></i><p>Greška pri učitavanju feedbackova</p></div>';
    });
}

/**
 * Load feedbacks timeline for all feedbacks modal
 */
function loadFeedbacksTimeline(personName) {
    const container = document.getElementById('feedbacksList');
    if (!container) {
        console.error('Feedbacks list container not found!');
        return;
    }
    
    console.log('Loading feedbacks timeline for:', personName);
    
    // Show loading
    container.innerHTML = '<div class="loading-state"><i class="fas fa-spinner fa-spin"></i> Učitavanje feedbackova...</div>';
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf(`/saleosoba/feedbacks?person_name=${encodeURIComponent(personName)}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        console.log('Feedbacks timeline data received:', data);
        if (data.feedbacks && data.feedbacks.length > 0) {
            renderFeedbacksTimeline(data.feedbacks, container, data.can_manage);
        } else {
            container.innerHTML = '<div class="empty-state"><i class="fas fa-comment-slash"></i><p>Nema feedbackova za ovu osobu</p></div>';
        }
    })
    .catch(error => {
        console.error('Error loading feedbacks timeline:', error);
        container.innerHTML = '<div class="error-state"><i class="fas fa-exclamation-triangle"></i><p>Greška pri učitavanju feedbackova</p></div>';
    });
}

/**
 * Render feedbacks timeline
 */
function renderFeedbacksTimeline(feedbacks, container, canManage) {
    const timeline = document.createElement('div');
    timeline.className = 'feedbacks-timeline';
    
    feedbacks.forEach(feedback => {
        const timelineItem = document.createElement('div');
        timelineItem.className = 'timeline-item';
        timelineItem.setAttribute('data-feedback-id', feedback.id);
        
        const date = new Date(feedback.datum);
        const formattedDate = date.toLocaleDateString('hr-HR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        
        let ratingHtml = '';
        if (feedback.ocjena) {
            const rating = parseFloat(feedback.ocjena);
            const ratingClass = rating >= 4 ? 'excellent' : (rating >= 3 ? 'good' : 'poor');
            const stars = Array.from({length: 5}, (_, i) => 
                `<i class="fas fa-star ${i < rating ? 'star-filled' : 'star-empty'}"></i>`
            ).join('');
            
            ratingHtml = `
                <div class="timeline-rating">
                    <span class="timeline-rating-value rating-${ratingClass}">${rating.toFixed(2)}/5.00</span>
                    <div class="timeline-stars">${stars}</div>
                </div>
            `;
        }
        
        let actionsHtml = '';
        if (canManage) {
            actionsHtml = `
                <div class="timeline-actions">
                    <button type="button" class="timeline-edit-btn" data-feedback-id="${feedback.id}">
                        <i class="fas fa-edit"></i> Uredi
                    </button>
                    <button type="button" class="timeline-delete-btn" data-feedback-id="${feedback.id}">
                        <i class="fas fa-trash"></i> Obriši
                    </button>
                </div>
            `;
        }
        
        timelineItem.innerHTML = `
            <div class="timeline-content">
                <div class="timeline-header">
                    <div class="timeline-date">${formattedDate}</div>
                    ${actionsHtml}
                </div>
                ${feedback.feedback ? `<div class="timeline-feedback">${escapeHtml(feedback.feedback).replace(/\n/g, '<br>')}</div>` : ''}
                ${ratingHtml}
                <div class="timeline-meta">
                    <i class="fas fa-user"></i>
                    <span>${escapeHtml(feedback.created_by)}</span>
                </div>
            </div>
        `;
        
        timeline.appendChild(timelineItem);
    });
    
    container.innerHTML = '';
    container.appendChild(timeline);
    
    // Add event listeners to timeline action buttons
    container.querySelectorAll('.timeline-edit-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const feedbackId = this.getAttribute('data-feedback-id');
            editFeedback(feedbackId);
        });
    });
    
    container.querySelectorAll('.timeline-delete-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const feedbackId = this.getAttribute('data-feedback-id');
            deleteFeedback(feedbackId);
        });
    });
}

/**
 * Save new feedback - CSRF FIXED VERSION
 */
function saveFeedback() {
    const form = document.getElementById('addFeedbackForm');
    if (!form) return;
    
    const formData = new FormData(form);
    
    // Validate form
    const feedback = formData.get('feedback');
    const ocjena = formData.get('ocjena');
    
    if (!feedback && !ocjena) {
        showNotification('Morate unijeti feedback ili ocjenu', 'error');
        return;
    }
    
    // Show loading
    const saveBtn = document.getElementById('saveFeedbackBtn');
    if (saveBtn) {
        const originalText = saveBtn.innerHTML;
        saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Spremanje...';
        saveBtn.disabled = true;
        
        // Use fetchWithCsrf instead of fetch
        window.fetchWithCsrf('/saleosoba/add-feedback', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('Feedback je uspješno dodan', 'success');
                closeModal(document.getElementById('addFeedbackModal'));
                // Refresh the page to show new feedback
                setTimeout(() => {
                    location.reload();
                }, 1000);
            } else {
                showNotification(data.error || 'Greška pri dodavanju feedback-a', 'error');
            }
        })
        .catch(error => {
            console.error('Error saving feedback:', error);
            showNotification('Greška pri dodavanju feedback-a', 'error');
        })
        .finally(() => {
            saveBtn.innerHTML = originalText;
            saveBtn.disabled = false;
        });
    }
}

/**
 * Edit feedback - CSRF FIXED VERSION
 */
function editFeedback(feedbackId) {
    // Get feedback data using fetchWithCsrf
    window.fetchWithCsrf(`/saleosoba/feedback/${feedbackId}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.feedback) {
            openEditFeedbackModal(data.feedback);
        } else {
            showNotification('Greška pri učitavanju feedback-a', 'error');
        }
    })
    .catch(error => {
        console.error('Error loading feedback:', error);
        showNotification('Greška pri učitavanju feedback-a', 'error');
    });
}

/**
 * Open edit feedback modal
 */
function openEditFeedbackModal(feedbackData) {
    const modal = document.getElementById('editFeedbackModal');
    if (!modal) return;
    
    // Populate form
    const feedbackIdInput = document.getElementById('editFeedbackId');
    const feedbackTextarea = document.getElementById('editFeedback');
    const ocjenaInput = document.getElementById('editOcjena');
    
    if (feedbackIdInput) feedbackIdInput.value = feedbackData.id;
    if (feedbackTextarea) feedbackTextarea.value = feedbackData.feedback || '';
    if (ocjenaInput) ocjenaInput.value = feedbackData.ocjena || '';
    
    // Update character counter
    const counter = document.getElementById('editFeedbackCounter');
    if (counter && feedbackTextarea) {
        updateCharacterCounter(feedbackTextarea, counter, 2000);
    }
    
    // Update rating preview
    const preview = document.getElementById('editRatingPreview');
    if (preview) {
        updateStarRating(feedbackData.ocjena || 0, preview);
    }
    
    showModal(modal);
}

/**
 * Update feedback - CSRF FIXED VERSION
 */
function updateFeedback() {
    const form = document.getElementById('editFeedbackForm');
    if (!form) return;
    
    const formData = new FormData(form);
    
    // Validate form
    const feedback = formData.get('feedback');
    const ocjena = formData.get('ocjena');
    
    if (!feedback && !ocjena) {
        showNotification('Morate unijeti feedback ili ocjenu', 'error');
        return;
    }
    
    // Show loading
    const updateBtn = document.getElementById('updateFeedbackBtn');
    if (updateBtn) {
        const originalText = updateBtn.innerHTML;
        updateBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Ažuriranje...';
        updateBtn.disabled = true;
        
        // Use fetchWithCsrf instead of fetch
        window.fetchWithCsrf('/saleosoba/update-feedback', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('Feedback je uspješno ažuriran', 'success');
                closeModal(document.getElementById('editFeedbackModal'));
                // Refresh the feedbacks timeline
                const personName = document.querySelector('input[name="osoba[name]"]')?.value;
                if (personName) {
                    loadAllFeedbacks(personName);
                }
            } else {
                showNotification(data.error || 'Greška pri ažuriranju feedback-a', 'error');
            }
        })
        .catch(error => {
            console.error('Error updating feedback:', error);
            showNotification('Greška pri ažuriranju feedback-a', 'error');
        })
        .finally(() => {
            updateBtn.innerHTML = originalText;
            updateBtn.disabled = false;
        });
    }
}

/**
 * Delete feedback - CSRF FIXED VERSION
 */
function deleteFeedback(feedbackId) {
    if (!confirm('Jeste li sigurni da želite obrisati ovaj feedback?')) {
        return;
    }
    
    const formData = new FormData();
    formData.append('feedback_id', feedbackId);
    
    // Use fetchWithCsrf instead of fetch
    window.fetchWithCsrf('/saleosoba/delete-feedback', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Feedback je uspješno obrisan', 'success');
            // Remove from timeline
            const timelineItem = document.querySelector(`[data-feedback-id="${feedbackId}"]`);
            if (timelineItem) {
                timelineItem.remove();
            }
            // Refresh the page to update feedback counts
            setTimeout(() => {
                location.reload();
            }, 1000);
        } else {
            showNotification(data.error || 'Greška pri brisanju feedback-a', 'error');
        }
    })
    .catch(error => {
        console.error('Error deleting feedback:', error);
        showNotification('Greška pri brisanju feedback-a', 'error');
    });
}

/**
 * Show modal - COMPLETELY FIXED VERSION
 */
function showModal(modal) {
    if (!modal) {
        console.error('Cannot show modal - modal element is null');
        return;
    }
    
    console.log('Showing modal:', modal.id);
    
    // Ukloni sve postojeće stilove koji mogu interferovati
    modal.style.removeProperty('height');
    modal.style.removeProperty('visibility');
    modal.style.removeProperty('opacity');
    
    // Dodaj show klasu koja koristi flex
    modal.classList.add('show');
    
    // Force reflow
    modal.offsetHeight;
    
    // Prevent body scroll
    document.body.style.overflow = 'hidden';
    
    // Popravi textarea visinu u modalu
    const textareas = modal.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        textarea.style.height = 'auto';
        textarea.style.height = Math.max(120, textarea.scrollHeight) + 'px';
    });
    
    // Focus first input
    const firstInput = modal.querySelector('input:not([type="hidden"]), textarea, select');
    if (firstInput) {
        setTimeout(() => {
            firstInput.focus();
            console.log('Focused first input:', firstInput);
        }, 200); // Povećao timeout
    }
    
    console.log('Modal should now be visible');
}

/**
 * Close modal - IMPROVED VERSION
 */
function closeModal(modal) {
    if (!modal) return;
    
    console.log('Closing modal:', modal.id);
    
    // Ukloni show klasu
    modal.classList.remove('show');
    
    // Vrati body scroll
    document.body.style.overflow = '';
    
    // Clear any form data if needed
    const form = modal.querySelector('form');
    if (form && modal.id === 'addFeedbackModal') {
        // Reset form only for add modal, not edit modal
        setTimeout(() => {
            form.reset();
            // Reset character counters
            const counters = modal.querySelectorAll('.char-counter');
            counters.forEach(counter => {
                counter.textContent = '0/2000';
                counter.classList.remove('warning', 'danger');
            });
        }, 300);
    }
}

/**
 * Initialize modal functionality - IMPROVED VERSION
 */
function initializeModals() {
    console.log('Initializing modals...');
    
    // Modal close buttons
    document.querySelectorAll('.modal-close, .modal-cancel').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Modal close button clicked');
            const modal = this.closest('.modal');
            if (modal) {
                closeModal(modal);
            }
        });
    });
    
    // Close modal when clicking outside
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', function(e) {
            // Provjeri da li je klik direktno na modal pozadinu
            if (e.target === this) {
                console.log('Modal background clicked');
                closeModal(this);
            }
        });
        
        // Sprječava zatvaranje modala kada se klikne na modal-content
        const modalContent = modal.querySelector('.modal-content');
        if (modalContent) {
            modalContent.addEventListener('click', function(e) {
                e.stopPropagation();
            });
        }
    });
    
    // ESC key za zatvaranje modala
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                closeModal(openModal);
            }
        }
    });
    
    // Close all feedbacks modal button
    const closeAllFeedbacksBtn = document.getElementById('closeAllFeedbacksBtn');
    if (closeAllFeedbacksBtn) {
        closeAllFeedbacksBtn.addEventListener('click', function(e) {
            e.preventDefault();
            closeModal(document.getElementById('allFeedbacksModal'));
        });
    }
    
    const modals = document.querySelectorAll('.modal');
    console.log(`Found ${modals.length} modals`);
}

/**
 * Show notification
 */
function showNotification(message, type = 'success') {
    let notification = document.getElementById('notification');
    
    // Create notification if it doesn't exist
    if (!notification) {
        const notificationHtml = `
            <div id="notification" class="notification">
                <div class="notification-content">
                    <i class="notification-icon"></i>
                    <span class="notification-message"></span>
                </div>
            </div>
        `;
        document.body.insertAdjacentHTML('beforeend', notificationHtml);
        notification = document.getElementById('notification');
    }
    
    if (!notification) return;
    
    const icon = type === 'success' ? 'fas fa-check-circle' : 'fas fa-exclamation-triangle';
    
    notification.className = `notification ${type}`;
    
    const iconElement = notification.querySelector('.notification-icon');
    const messageElement = notification.querySelector('.notification-message');
    
    if (iconElement) iconElement.className = `notification-icon ${icon}`;
    if (messageElement) messageElement.textContent = message;
    
    notification.classList.add('show');
    
    setTimeout(() => {
        notification.classList.remove('show');
    }, 4000);
}

/**
 * Escape HTML to prevent XSS
 */
function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}

// Global functions for timeline actions
window.editFeedback = editFeedback;
window.deleteFeedback = deleteFeedback;
//# sourceMappingURL=osobaedit.67dd9905.js.map