/**
 * Real Data Sales Admin Dashboard Charts
 * Handles chart creation functionality with real database data
 */
document.addEventListener('DOMContentLoaded', function() {
  // Get data elements
  const chartsData = document.getElementById('sales-charts-data');
  
  if (!chartsData) {
    console.error('Sales charts data element not found!');
    return;
  }
  
  try {
    // Parse real data from the backend
    const funnelData = JSON.parse(chartsData.getAttribute('data-funnel') || '[]');
    const monthlyData = JSON.parse(chartsData.getAttribute('data-monthly') || '[]');
    const servicesData = JSON.parse(chartsData.getAttribute('data-services') || '[]');
    const agentsData = JSON.parse(chartsData.getAttribute('data-agents') || '[]');
    const statusData = JSON.parse(chartsData.getAttribute('data-status') || '[]');
    
    // Create charts with real data
    createSalesFunnelChart(funnelData);
    createMonthlyTrendsChart(monthlyData);
    createStatusDistributionChart(statusData);
    
  } catch (error) {
    console.error('Error initializing sales admin charts:', error);
  }
});

/**
 * Create the sales funnel chart
 * @param {Array} funnelData - Real funnel data from database
 */
function createSalesFunnelChart(funnelData) {
  if (!document.getElementById('salesFunnelChart')) {
    console.error('Sales funnel chart element not found');
    return;
  }
  
  if (!funnelData || funnelData.length === 0) {
    console.warn('No funnel data available');
    return;
  }
  
  // Define funnel stages in order
  const funnelStages = ['Potencijal', 'Uneseno', 'Aktivno', 'Storno'];
  const funnelColors = {
    'Potencijal': '#2196f3',
    'Uneseno': '#ff9800', 
    'Aktivno': '#4caf50',
    'Storno': '#f44336'
  };
  
  // Map data to ordered stages
  const labels = [];
  const values = [];
  const colors = [];
  
  funnelStages.forEach(stage => {
    const found = funnelData.find(item => item.status === stage);
    if (found) {
      labels.push(stage);
      values.push(found.count);
      colors.push(funnelColors[stage]);
    }
  });
  
  const ctx = document.getElementById('salesFunnelChart').getContext('2d');
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: labels,
      datasets: [{
        label: 'Broj prodaja',
        data: values,
        backgroundColor: colors,
        borderColor: colors.map(color => color.replace('1)', '1)')),
        borderWidth: 2,
        borderRadius: 8,
        borderSkipped: false
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      indexAxis: 'y', // Horizontal bar chart
      scales: {
        x: {
          beginAtZero: true,
          ticks: {
            precision: 0
          }
        },
        y: {
          grid: {
            display: false
          }
        }
      },
      plugins: {
        legend: {
          display: false
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = ((context.raw / total) * 100).toFixed(1);
              return `${context.label}: ${context.raw} prodaja (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the monthly trends chart
 * @param {Array} monthlyData - Real monthly data from database
 */
function createMonthlyTrendsChart(monthlyData) {
  if (!document.getElementById('monthlyTrendsChart')) {
    console.error('Monthly trends chart element not found');
    return;
  }
  
  if (!monthlyData || monthlyData.length === 0) {
    console.warn('No monthly data available');
    return;
  }
  
  // Sort data by year and month
  const sortedData = monthlyData.sort((a, b) => {
    return new Date(a.year, a.month - 1) - new Date(b.year, b.month - 1);
  });
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  
  const labels = sortedData.map(item => 
    `${monthNames[item.month - 1]} ${item.year}`
  );
  
  const totalSales = sortedData.map(item => parseInt(item.total_sales) || 0);
  const activeSales = sortedData.map(item => parseInt(item.active_sales) || 0);
  const potentialSales = sortedData.map(item => parseInt(item.potential_sales) || 0);
  
  const ctx = document.getElementById('monthlyTrendsChart').getContext('2d');
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Ukupno prodaja',
          data: totalSales,
          borderColor: '#2196f3',
          backgroundColor: 'rgba(33, 150, 243, 0.1)',
          borderWidth: 3,
          fill: true,
          tension: 0.4,
          pointRadius: 4,
          pointHoverRadius: 6
        },
        {
          label: 'Aktivne prodaje',
          data: activeSales,
          borderColor: '#4caf50',
          backgroundColor: 'rgba(76, 175, 80, 0.1)',
          borderWidth: 3,
          fill: false,
          tension: 0.4,
          pointRadius: 4,
          pointHoverRadius: 6
        },
        {
          label: 'Potencijalne prodaje',
          data: potentialSales,
          borderColor: '#ff9800',
          backgroundColor: 'rgba(255, 152, 0, 0.1)',
          borderWidth: 2,
          fill: false,
          tension: 0.4,
          pointRadius: 3,
          pointHoverRadius: 5,
          borderDash: [5, 5]
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'index',
        intersect: false,
      },
      scales: {
        x: {
          ticks: {
            maxRotation: 45
          }
        },
        y: {
          beginAtZero: true,
          ticks: {
            precision: 0
          }
        }
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            boxWidth: 12,
            font: {
              size: 11
            },
            usePointStyle: true
          }
        },
        tooltip: {
          callbacks: {
            title: function(context) {
              return context[0].label;
            },
            label: function(context) {
              return `${context.dataset.label}: ${context.raw} prodaja`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the status distribution chart
 * @param {Array} statusData - Real status data from database
 */
function createStatusDistributionChart(statusData) {
  if (!document.getElementById('statusDistributionChart')) {
    console.error('Status distribution chart element not found');
    return;
  }
  
  if (!statusData || statusData.length === 0) {
    console.warn('No status data available');
    return;
  }
  
  const statusColors = {
    'Aktivno': '#4caf50',
    'Potencijal': '#2196f3',
    'Uneseno': '#ff9800',
    'Storno': '#f44336'
  };
  
  const labels = statusData.map(item => item.status || 'Nepoznato');
  const values = statusData.map(item => parseInt(item.count) || 0);
  const colors = labels.map(label => statusColors[label] || '#757575');
  
  const ctx = document.getElementById('statusDistributionChart').getContext('2d');
  new Chart(ctx, {
    type: 'doughnut',
    data: {
      labels: labels,
      datasets: [{
        data: values,
        backgroundColor: colors,
        borderColor: '#ffffff',
        borderWidth: 2,
        hoverBorderWidth: 3
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      cutout: '50%',
      plugins: {
        legend: {
          position: 'right',
          labels: {
            boxWidth: 12,
            font: {
              size: 11
            },
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
                  return {
                    text: `${label} (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = total > 0 ? ((value / total) * 100).toFixed(1) : '0.0';
              return `${context.label}: ${value} prodaja (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}
//# sourceMappingURL=salesadmincharts.76e5a8c7.js.map