/**
 * Target Schema List JavaScript
 * Specific functionality for target schema list view
 * NOTE: This script depends on table.js and should be loaded AFTER it
 */
document.addEventListener('DOMContentLoaded', function() {
    // Apply titles to cells for tooltip display on hover
    addCellTitles();
    
    /**
     * Add title attributes to all cells for better display of truncated content
     */
    function addCellTitles() {
      const dataCells = document.querySelectorAll('.data-table td');
      dataCells.forEach(cell => {
        if (cell.textContent.trim()) {
          cell.setAttribute('title', cell.textContent.trim());
        }
      });
    }
    // Find the table (already initialized by table.js)
    const dataTable = document.getElementById('dataTable');
    
    if (!dataTable) {
      console.warn('Target table not found');
      return;
    }
    
    // Add target-specific enhancements
    addHoverEffects();
    initFilters(); // Add filters initialization
    initDeleteModal(); // Initialize delete confirmation modal
    initExportPanel(); // Initialize export panel
    
    /**
     * Add hover effects to rows and elements
     */
    function addHoverEffects() {
      const rows = dataTable.querySelectorAll('tbody tr');
      
      rows.forEach(row => {
        row.addEventListener('mouseenter', function() {
          // Highlight the whole row on hover
          this.style.backgroundColor = 'var(--hover)';
        });
        
        row.addEventListener('mouseleave', function() {
          // Remove highlight on mouse leave
          this.style.backgroundColor = '';
        });
      });
    }
    
    /**
     * Initialize filter functionality - AŽURIRANO za nove kolone
     */
    function initFilters() {
      const prijenosFilterContainer = document.getElementById('prijenosFilter');
      const p2pFilterContainer = document.getElementById('p2pFilter');
      
      if (!prijenosFilterContainer || !p2pFilterContainer) {
        console.warn('Filter containers not found');
        return;
      }
      
      // Get all unique "prijenos" values from the table (kolona 2)
      const prijenosValues = new Set();
      const prijenosCells = dataTable.querySelectorAll('tbody tr td:nth-child(2)');
      
      prijenosCells.forEach(cell => {
        const prijenosValue = cell.textContent.trim();
        if (prijenosValue) {
          prijenosValues.add(prijenosValue);
        }
      });
      
      // Get all unique "p2p" values from the table (kolona 3)
      const p2pValues = new Set();
      const p2pCells = dataTable.querySelectorAll('tbody tr td:nth-child(3)');
      
      p2pCells.forEach(cell => {
        const p2pValue = cell.textContent.trim();
        if (p2pValue) {
          p2pValues.add(p2pValue);
        }
      });
      
      // Clear existing content
      prijenosFilterContainer.innerHTML = '';
      p2pFilterContainer.innerHTML = '';
      
      // Create filter checkboxes for prijenos values
      prijenosValues.forEach(prijenosValue => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'prijenos');
        checkbox.setAttribute('data-value', prijenosValue);
        checkbox.checked = true; // Default to checked
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + prijenosValue));
        
        prijenosFilterContainer.appendChild(label);
      });
      
      // Create filter checkboxes for p2p values
      p2pValues.forEach(p2pValue => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'p2p');
        checkbox.setAttribute('data-value', p2pValue);
        checkbox.checked = true; // Default to checked
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + p2pValue));
        
        p2pFilterContainer.appendChild(label);
      });
      
      // Add event listeners to the new checkboxes
      const filterCheckboxes = document.querySelectorAll('.filter-checkbox input[type="checkbox"]');
      filterCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', applyFilters);
      });
      
      // Add search functionality
      const searchInput = document.getElementById('dataSearch');
      if (searchInput) {
        searchInput.addEventListener('input', applyFilters);
      }
    }
    
    /**
     * Apply filters to the table - AŽURIRANO za nove kolone
     */
    function applyFilters() {
      if (!dataTable) return;
      
      const rows = dataTable.querySelectorAll('tbody tr');
      const activeFilters = {};
      const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
      const searchInput = document.getElementById('dataSearch');
      const searchValue = searchInput ? searchInput.value.toLowerCase() : '';
      
      console.log('Applying filters, checked filters:', filterCheckboxes.length);
      
      // Build active filters map
      filterCheckboxes.forEach(checkbox => {
        const filterType = checkbox.getAttribute('data-filter');
        const filterValue = checkbox.getAttribute('data-value');
        
        if (!activeFilters[filterType]) {
          activeFilters[filterType] = [];
        }
        
        activeFilters[filterType].push(filterValue);
      });
      
      console.log('Active filters:', activeFilters);
      
      // Apply filtering
      let visibleCount = 0;
      
      rows.forEach(row => {
        let shouldShow = true;
        
        // Apply search filter
        if (searchValue) {
          const rowText = row.textContent.toLowerCase();
          if (!rowText.includes(searchValue)) {
            shouldShow = false;
          }
        }
        
        // Apply category filters if row is still visible
        if (shouldShow) {
          // Check each filter type
          for (const filterType in activeFilters) {
            let matches = false;
            
            if (filterType === 'prijenos') {
              // Get the prijenos from the row (kolona 2)
              const prijenosCell = row.querySelector('td:nth-child(2)');
              if (prijenosCell) {
                const prijenosValue = prijenosCell.textContent.trim();
                
                matches = activeFilters[filterType].some(value => 
                  prijenosValue === value
                );
              }
            } else if (filterType === 'p2p') {
              // Get the p2p from the row (kolona 3)
              const p2pCell = row.querySelector('td:nth-child(3)');
              if (p2pCell) {
                const p2pValue = p2pCell.textContent.trim();
                
                matches = activeFilters[filterType].some(value => 
                  p2pValue === value
                );
              }
            }
            
            if (!matches) {
              shouldShow = false;
              break;
            }
          }
        }
        
        row.style.display = shouldShow ? '' : 'none';
        
        if (shouldShow) {
          visibleCount++;
        }
      });
      
      console.log('Filter applied, visible rows:', visibleCount);
    }
    
    /**
     * Initialize delete modal functionality
     */
    function initDeleteModal() {
      const deleteButtons = document.querySelectorAll('.action-btn.delete');
      const deleteModal = document.getElementById('deleteModal');
      const deleteForm = document.getElementById('deleteForm');
      const deleteId = document.getElementById('deleteId');
      const deleteName = document.getElementById('deleteName');
      const modalClose = document.querySelector('.modal-close');
      const modalCancel = document.querySelector('.modal-cancel');
      
      if (!deleteModal) {
        console.warn('Delete modal not found');
        return;
      }
      
      deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
          const id = this.getAttribute('data-id');
          const name = this.getAttribute('data-name');
          
          deleteId.value = id;
          deleteName.textContent = name;
          deleteModal.style.display = 'block';
        });
      });
      
      // Close modal when clicking on X or Cancel
      modalClose.addEventListener('click', function() {
        deleteModal.style.display = 'none';
      });
      
      modalCancel.addEventListener('click', function() {
        deleteModal.style.display = 'none';
      });
      
      // Close modal when clicking outside
      window.addEventListener('click', function(event) {
        if (event.target == deleteModal) {
          deleteModal.style.display = 'none';
        }
      });
    }
    
    /**
     * Initialize export panel functionality
     */
    function initExportPanel() {
      const exportToggle = document.getElementById('exportToggle');
      const exportPanel = document.getElementById('exportPanel');
      const closeExport = document.getElementById('closeExport');
      
      if (!exportPanel || !exportToggle) {
        console.warn('Export panel elements not found');
        return;
      }
      
      exportToggle.addEventListener('click', function() {
        exportPanel.classList.toggle('visible');
      });
      
      closeExport.addEventListener('click', function() {
        exportPanel.classList.remove('visible');
      });
    }
    
    // Listen for theme changes
    document.addEventListener('themechange', function(event) {
      console.log('Theme changed in target list page:', event.detail.theme);
    });
  });
//# sourceMappingURL=targetlist.eb3ad7f7.js.map