// teambonus.js
document.addEventListener('DOMContentLoaded', function() {
  // Card value animation
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach(valueElement => {
    let finalValue = parseFloat(valueElement.textContent.replace(/[^\d.-]/g, ''));
    if (!isNaN(finalValue)) {
      // If the value has decimal places (like percentages)
      if (valueElement.textContent.includes('%')) {
        const decimalPlaces = 1;
        animateValue(valueElement, 0, finalValue, 1000, finalValue, decimalPlaces, false, true);
      } else {
        // Simple integer animation
        animateValue(valueElement, 0, finalValue, 1000);
      }
    }
  });
  
  // Initialize month counts animation
  const monthCounts = document.querySelectorAll('.month-count');
  monthCounts.forEach(countElement => {
    let finalValue = parseInt(countElement.textContent);
    if (!isNaN(finalValue)) {
      animateValue(countElement, 0, finalValue, 800);
    }
  });
  
  // Refresh stats periodically (every 5 minutes)
  if (typeof refreshTeamBonusInterval === 'undefined') {
    let refreshTeamBonusInterval = setInterval(refreshTeamBonusStats, 300000);
  }

  // Add hover effects to comparison rows
  const comparisonRows = document.querySelectorAll('.comparison-row');
  comparisonRows.forEach(row => {
    row.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
    });
    
    row.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
    });
  });

  // Add click handlers for month cards
  const monthCards = document.querySelectorAll('.month-card');
  monthCards.forEach(card => {
    card.addEventListener('click', function() {
      // You can add functionality to show detailed view for specific month
      const monthName = this.querySelector('.month-name').textContent;
      console.log('Clicked on month:', monthName);
      // Example: redirect to detailed view
      // window.location.href = `/teambonus/detail?month=${monthName}`;
    });
  });
});

// Animate counter value from start to end with optional decimal formatting
function animateValue(element, start, end, duration, finalValue, decimalPlaces, hasComma, hasPercent) {
  let startTimestamp = null;
  const originalText = element.textContent;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    const value = Math.floor(progress * (end - start) + start);
    
    if (decimalPlaces !== undefined) {
      // For values with decimal places
      const decimalSeparator = hasComma ? ',' : '.';
      let formattedValue;
      
      if (progress >= 1) {
        // On animation completion, use the exact final value
        formattedValue = typeof finalValue === 'number' ? 
          finalValue.toFixed(decimalPlaces).replace('.', decimalSeparator) : 
          value.toString();
      } else {
        formattedValue = value.toString();
      }
      
      // Add percent symbol if needed
      if (hasPercent) {
        formattedValue += '%';
      }
      
      element.textContent = formattedValue;
    } else {
      // For simple integers
      element.textContent = value;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    }
  };
  
  window.requestAnimationFrame(step);
}

// Function to refresh team bonus stats via AJAX
function refreshTeamBonusStats() {
  fetch('/teambonus/api/stats', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    }
  })
    .then(response => response.json())
    .then(data => {
      // Update card values
      updateElementText('.dashboard-card:nth-child(1) .card-value', data.currentMonthActives);
      updateElementText('.dashboard-card:nth-child(2) .card-value', data.lastMonthActives);
      updateElementText('.dashboard-card:nth-child(3) .card-value', data.totalActiveServices);
      
      // Update growth percentage
      if (data.currentMonthActives && data.lastMonthActives) {
        const growth = data.lastMonthActives > 0 ? 
          ((data.currentMonthActives - data.lastMonthActives) / data.lastMonthActives) * 100 : 0;
        updateElementText('.dashboard-card:nth-child(4) .card-value', `${growth.toFixed(1)}%`);
      }
      
      // Update top performers if needed
      updateTopPerformers(data.topPerformingServices);
      
      // Update monthly comparison if needed
      updateMonthlyComparison(data.monthlyComparison);
    })
    .catch(error => {
      console.error('Error refreshing team bonus stats:', error);
    });
}

// Helper to update element text
function updateElementText(selector, value) {
  const element = document.querySelector(selector);
  if (element) {
    element.textContent = value;
  }
}

// Update top performers section
function updateTopPerformers(performersData) {
  if (!performersData || !Array.isArray(performersData)) return;
  
  const performersList = document.querySelector('.top-performers-list');
  if (!performersList) return;
  
  // Clear existing content
  performersList.innerHTML = '';
  
  if (performersData.length === 0) {
    performersList.innerHTML = '<div class="empty-state"><p>Nema dostupnih podataka o aktivnim uslugama</p></div>';
    return;
  }
  
  // Add updated performers
  performersData.forEach((performer, index) => {
    const performerItem = document.createElement('div');
    performerItem.className = 'performer-item';
    performerItem.innerHTML = `
      <div class="performer-rank">${index + 1}</div>
      <div class="performer-info">
        <span class="performer-name">${escapeHtml(performer.service_type)}</span>
        <div class="performer-stats">
          <span class="performer-tasks"><i class="fas fa-chart-line"></i> ${performer.active_count} aktivnih</span>
          <span class="performer-work"><i class="fas fa-calendar"></i> ${performer.current_month_count} ovaj mjesec</span>
        </div>
      </div>
    `;
    performersList.appendChild(performerItem);
  });
}

// Update monthly comparison section
function updateMonthlyComparison(comparisonData) {
  if (!comparisonData || !Array.isArray(comparisonData)) return;
  
  const comparisonTable = document.querySelector('.comparison-table');
  if (!comparisonTable) return;
  
  // Keep header, update rows
  const header = comparisonTable.querySelector('.comparison-header');
  comparisonTable.innerHTML = '';
  comparisonTable.appendChild(header);
  
  if (comparisonData.length === 0) {
    const emptyState = document.createElement('div');
    emptyState.className = 'empty-state';
    emptyState.innerHTML = '<p>Nema dostupnih podataka za usporedbu</p>';
    comparisonTable.appendChild(emptyState);
    return;
  }
  
  // Add updated comparison rows
  comparisonData.forEach(comparison => {
    const current = comparison.current_month;
    const previous = comparison.previous_month;
    const change = previous > 0 ? ((current - previous) / previous) * 100 : 0;
    const changeClass = change > 0 ? 'positive' : (change < 0 ? 'negative' : 'neutral');
    
    const row = document.createElement('div');
    row.className = 'comparison-row';
    row.innerHTML = `
      <div class="comparison-col service-name">${escapeHtml(comparison.service_type)}</div>
      <div class="comparison-col">${current}</div>
      <div class="comparison-col">${previous}</div>
      <div class="comparison-col change ${changeClass}">
        ${change > 0 ? '+' : ''}${change.toFixed(1)}%
      </div>
    `;
    comparisonTable.appendChild(row);
  });
}

// Helper function to escape HTML
function escapeHtml(text) {
  const map = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#039;'
  };
  return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}
//# sourceMappingURL=teambonus.4aee5b04.js.map