/**
 * JavaScript for Team Bonus Dashboard
 * Handles chart creation functionality
 */
document.addEventListener('DOMContentLoaded', function() {
  // Get data elements
  const chartsData = document.getElementById('teambonus-charts-data');
  
  if (!chartsData) {
    console.error('Team bonus charts data element not found!');
    return;
  }
  
  try {
    // Parse data for the charts
    const monthlyData = JSON.parse(chartsData.getAttribute('data-monthly') || '[]');
    const servicesData = JSON.parse(chartsData.getAttribute('data-services') || '[]');
    const trendsData = JSON.parse(chartsData.getAttribute('data-trends') || '[]');
    const comparisonData = JSON.parse(chartsData.getAttribute('data-comparison') || '[]');
    
    // Create Monthly Actives chart
    createMonthlyActivesChart(monthlyData);
    
    // Create Service Actives chart
    createServiceActivesChart(servicesData);
    
    // Create Service Growth Trends chart
    createServiceGrowthChart(trendsData);
    
  } catch (error) {
    console.error('Error initializing team bonus charts:', error);
  }
});

/**
 * Create the monthly active sales chart
 * @param {Array} monthlyData - Data for the monthly chart
 */
function createMonthlyActivesChart(monthlyData) {
  if (!document.getElementById('monthlyActivesChart')) {
    console.error('Monthly actives chart element not found');
    return;
  }
  
  // Group data by month and calculate totals
  const monthlyTotals = {};
  monthlyData.forEach(item => {
    const key = `${item.year}-${String(item.month).padStart(2, '0')}`;
    if (!monthlyTotals[key]) {
      monthlyTotals[key] = {
        year: item.year,
        month: item.month,
        total: 0
      };
    }
    monthlyTotals[key].total += parseInt(item.active_count || 0);
  });
  
  // Sort by date and get last 12 months
  const sortedMonths = Object.values(monthlyTotals)
    .sort((a, b) => new Date(a.year, a.month - 1) - new Date(b.year, b.month - 1))
    .slice(-12);
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  const labels = sortedMonths.map(item => `${monthNames[item.month - 1]} ${item.year}`);
  const values = sortedMonths.map(item => item.total);
  
  const monthlyCtx = document.getElementById('monthlyActivesChart').getContext('2d');
  new Chart(monthlyCtx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [{
        label: 'Aktivne prodaje',
        data: values,
        backgroundColor: 'rgba(54, 162, 235, 0.2)',
        borderColor: 'rgba(54, 162, 235, 1)',
        borderWidth: 3,
        fill: true,
        tension: 0.4,
        pointBackgroundColor: 'rgba(54, 162, 235, 1)',
        pointBorderColor: '#fff',
        pointBorderWidth: 2,
        pointRadius: 5
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      scales: {
        y: {
          beginAtZero: true,
          ticks: {
            precision: 0
          }
        },
        x: {
          ticks: {
            maxRotation: 45
          }
        }
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            boxWidth: 12,
            font: {
              size: 11
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              return `${context.dataset.label}: ${context.raw} aktivnih prodaja`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the active sales by service chart
 * @param {Array} servicesData - Data for the services chart
 */
function createServiceActivesChart(servicesData) {
  if (!document.getElementById('serviceActivesChart')) {
    console.error('Service actives chart element not found');
    return;
  }
  
  // Limit to top 8 services if there are many
  let limitedData = servicesData;
  if (servicesData.length > 8) {
    const sortedData = [...servicesData].sort((a, b) => b.active_count - a.active_count);
    const topData = sortedData.slice(0, 7);
    const otherCount = sortedData.slice(7).reduce((sum, item) => sum + parseInt(item.active_count), 0);
    limitedData = [
      ...topData,
      { service_type: 'Ostalo', active_count: otherCount }
    ];
  }
  
  const serviceLabels = limitedData.map(item => item.service_type || 'Nepoznato');
  const serviceValues = limitedData.map(item => parseInt(item.active_count));
  const serviceColors = [
    '#ff6384', '#36a2eb', '#ffce56', '#4bc0c0', 
    '#9966ff', '#ff9f40', '#ff6384', '#c9cbcf'
  ];
  
  const serviceCtx = document.getElementById('serviceActivesChart').getContext('2d');
  new Chart(serviceCtx, {
    type: 'doughnut',
    data: {
      labels: serviceLabels,
      datasets: [{
        data: serviceValues,
        backgroundColor: serviceColors.slice(0, serviceLabels.length),
        borderWidth: 2,
        borderColor: '#fff'
      }]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          position: 'right',
          labels: {
            boxWidth: 12,
            font: {
              size: 10
            },
            generateLabels: function(chart) {
              const data = chart.data;
              if (data.labels.length && data.datasets.length) {
                return data.labels.map((label, i) => {
                  const value = data.datasets[0].data[i];
                  const total = data.datasets[0].data.reduce((acc, val) => acc + val, 0);
                  const percentage = ((value / total) * 100).toFixed(1);
                  return {
                    text: `${label} (${percentage}%)`,
                    fillStyle: data.datasets[0].backgroundColor[i],
                    hidden: false,
                    index: i
                  };
                });
              }
              return [];
            }
          }
        },
        tooltip: {
          callbacks: {
            label: function(context) {
              const value = context.raw;
              const total = context.dataset.data.reduce((acc, val) => acc + val, 0);
              const percentage = ((value / total) * 100).toFixed(1);
              return `${context.label}: ${value} aktivnih (${percentage}%)`;
            }
          }
        }
      }
    }
  });
}

/**
 * Create the service growth trends chart
 * @param {Array} trendsData - Data for the trends chart
 */
function createServiceGrowthChart(trendsData) {
  if (!document.getElementById('serviceGrowthChart')) {
    console.error('Service growth chart element not found');
    return;
  }
  
  if (!trendsData || trendsData.length === 0) {
    const ctx = document.getElementById('serviceGrowthChart').getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: [],
        datasets: []
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            display: false
          }
        },
        elements: {
          point: {
            radius: 0
          }
        }
      }
    });
    return;
  }
  
  // Group by service type
  const serviceGroups = {};
  trendsData.forEach(item => {
    const service = item.service_type;
    if (!serviceGroups[service]) {
      serviceGroups[service] = [];
    }
    serviceGroups[service].push({
      year: item.year,
      month: item.month,
      count: parseInt(item.active_count || 0),
      date: new Date(item.year, item.month - 1, 1)
    });
  });
  
  // Sort each service's data by date
  Object.keys(serviceGroups).forEach(service => {
    serviceGroups[service].sort((a, b) => a.date - b.date);
  });
  
  // Get all unique months for labels
  const allDates = [];
  Object.values(serviceGroups).forEach(serviceData => {
    serviceData.forEach(item => {
      const dateKey = `${item.year}-${String(item.month).padStart(2, '0')}`;
      if (!allDates.includes(dateKey)) {
        allDates.push(dateKey);
      }
    });
  });
  
  allDates.sort();
  
  const monthNames = ['Sij', 'Velj', 'Ožu', 'Tra', 'Svi', 'Lip', 'Srp', 'Kol', 'Ruj', 'Lis', 'Stu', 'Pro'];
  const labels = allDates.map(dateKey => {
    const [year, month] = dateKey.split('-');
    return `${monthNames[parseInt(month) - 1]} ${year}`;
  });
  
  // Create datasets for top 5 services only
  const topServices = Object.keys(serviceGroups)
    .map(service => ({
      service,
      total: serviceGroups[service].reduce((sum, item) => sum + item.count, 0)
    }))
    .sort((a, b) => b.total - a.total)
    .slice(0, 5);
  
  const colors = [
    'rgba(255, 99, 132, 1)',
    'rgba(54, 162, 235, 1)', 
    'rgba(255, 206, 86, 1)',
    'rgba(75, 192, 192, 1)',
    'rgba(153, 102, 255, 1)'
  ];
  
  const datasets = topServices.map((serviceInfo, index) => {
    const service = serviceInfo.service;
    const serviceData = serviceGroups[service];
    
    // Map data to all months
    const data = allDates.map(dateKey => {
      const [year, month] = dateKey.split('-');
      const found = serviceData.find(item => 
        item.year === parseInt(year) && item.month === parseInt(month)
      );
      return found ? found.count : 0;
    });
    
    return {
      label: service,
      data: data,
      borderColor: colors[index],
      backgroundColor: colors[index].replace('1)', '0.1)'),
      borderWidth: 2,
      fill: false,
      tension: 0.3,
      pointRadius: 3,
      pointHoverRadius: 5
    };
  });
  
  const trendsCtx = document.getElementById('serviceGrowthChart').getContext('2d');
  new Chart(trendsCtx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: datasets
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'index',
        intersect: false,
      },
      scales: {
        y: {
          beginAtZero: true,
          ticks: {
            precision: 0
          }
        },
        x: {
          ticks: {
            maxRotation: 45
          }
        }
      },
      plugins: {
        legend: {
          position: 'top',
          labels: {
            boxWidth: 12,
            font: {
              size: 10
            },
            usePointStyle: true
          }
        },
        tooltip: {
          callbacks: {
            title: function(context) {
              return context[0].label;
            },
            label: function(context) {
              return `${context.dataset.label}: ${context.raw} aktivnih`;
            }
          }
        }
      }
    }
  });
}
//# sourceMappingURL=teambonuscharts.6eddaebe.js.map