/**
 * Osobe (Person) List JavaScript
 * Specific functionality for persons list view
 * NOTE: This script depends on table.js and should be loaded AFTER it
 */
document.addEventListener('DOMContentLoaded', function() {
  // Find the table (already initialized by table.js)
  const dataTable = document.getElementById('dataTable');
  
  if (!dataTable) {
    console.warn('Person table not found');
    return;
  }
  
  // Add person-specific enhancements
  updateBadgeColors();
  updateStatusIndicators();
  addHoverEffects();
  initFilters(); // Add filters initialization
  initDeleteModal(); // Initialize delete confirmation modal
  initExportPanel(); // Initialize export panel
  
  /**
   * Update status indicators (active/inactive)
   */
  function updateStatusIndicators() {
    const rows = dataTable.querySelectorAll('tbody tr');
    
    rows.forEach(row => {
      const statusCell = row.querySelector('td:nth-child(7)');
      if (!statusCell) return;
      
      const statusBadge = statusCell.querySelector('.status-badge');
      if (!statusBadge) return;
      
      const isActive = statusBadge.textContent.trim().includes('Active');
      row.setAttribute('data-status', isActive ? '1' : '0');
      
      // Add title for accessibility
      statusBadge.setAttribute('title', isActive ? 'Active user' : 'Inactive user');
      
      // Add pulse animation to active status
      if (isActive && !statusBadge.classList.contains('pulse-added')) {
        statusBadge.classList.add('pulse-added');
        const pulseElement = document.createElement('span');
        pulseElement.className = 'status-pulse';
        statusBadge.appendChild(pulseElement);
      }
    });
  }
  
  /**
   * Add hover effects to rows and elements
   */
  function addHoverEffects() {
    const rows = dataTable.querySelectorAll('tbody tr');
    
    rows.forEach(row => {
      const personIcon = row.querySelector('.person-icon');
      const companyBadge = row.querySelector('.company-badge');
      const orgBadge = row.querySelector('.org-badge');
      
      row.addEventListener('mouseenter', function() {
        if (personIcon) personIcon.classList.add('icon-highlight');
        if (companyBadge) companyBadge.classList.add('badge-highlight');
        if (orgBadge) orgBadge.classList.add('badge-highlight');
      });
      
      row.addEventListener('mouseleave', function() {
        if (personIcon) personIcon.classList.remove('icon-highlight');
        if (companyBadge) companyBadge.classList.remove('badge-highlight');
        if (orgBadge) orgBadge.classList.remove('badge-highlight');
      });
    });
  }
  
  /**
   * Set different colors for different badges
   */
  function updateBadgeColors() {
    // Update company badges
    const companyBadges = document.querySelectorAll('.company-badge');
    companyBadges.forEach(badge => {
      const companyText = badge.textContent.trim();
      badge.setAttribute('data-company', companyText);
    });
    
    // Update org badges
    const orgBadges = document.querySelectorAll('.org-badge');
    orgBadges.forEach(badge => {
      const orgText = badge.textContent.trim();
      badge.setAttribute('data-org', orgText);
    });
  }
  
  /**
   * Initialize filter functionality for organization units, companies, and status
   */
  function initFilters() {
    // Initialize organization unit filters
    initOrgFilters();
    
    // Initialize company filters
    initCompanyFilters();
    
    // Status filters are already defined in HTML, so we just need to add the event listeners
    const statusCheckboxes = document.querySelectorAll('input[data-filter="status"]');
    statusCheckboxes.forEach(checkbox => {
      checkbox.addEventListener('change', applyFilters);
    });
  }
  
  /**
   * Initialize organization unit filters
   */
  function initOrgFilters() {
    const orgFilterContainer = document.getElementById('orgFilter');
    if (!orgFilterContainer) {
      console.warn('Organization filter container not found');
      return;
    }
    
    // Get all unique org units from the table
    const orgUnits = new Set();
    const orgBadges = document.querySelectorAll('.org-badge');
    
    orgBadges.forEach(badge => {
      const orgText = badge.textContent.trim();
      if (orgText) {
        orgUnits.add(orgText);
      }
    });
    
    // Clear existing content
    orgFilterContainer.innerHTML = '';
    
    // Create filter checkboxes for each org unit
    orgUnits.forEach(orgUnit => {
      const label = document.createElement('label');
      label.className = 'filter-checkbox';
      
      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.setAttribute('data-filter', 'org');
      checkbox.setAttribute('data-value', orgUnit);
      
      label.appendChild(checkbox);
      label.appendChild(document.createTextNode(' ' + orgUnit));
      
      orgFilterContainer.appendChild(label);
    });
    
    // Add event listeners to the new checkboxes
    const filterCheckboxes = orgFilterContainer.querySelectorAll('input[type="checkbox"]');
    filterCheckboxes.forEach(checkbox => {
      checkbox.addEventListener('change', applyFilters);
    });
  }
  
  /**
   * Initialize company filters
   */
  function initCompanyFilters() {
    const companyFilterContainer = document.getElementById('companyFilter');
    if (!companyFilterContainer) {
      console.warn('Company filter container not found');
      return;
    }
    
    // Get all unique companies from the table
    const companies = new Set();
    const companyBadges = document.querySelectorAll('.company-badge');
    
    companyBadges.forEach(badge => {
      const companyText = badge.textContent.trim();
      if (companyText) {
        companies.add(companyText);
      }
    });
    
    // Clear existing content
    companyFilterContainer.innerHTML = '';
    
    // Create filter checkboxes for each company
    companies.forEach(company => {
      const label = document.createElement('label');
      label.className = 'filter-checkbox';
      
      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.setAttribute('data-filter', 'company');
      checkbox.setAttribute('data-value', company);
      
      label.appendChild(checkbox);
      label.appendChild(document.createTextNode(' ' + company));
      
      companyFilterContainer.appendChild(label);
    });
    
    // Add event listeners to the new checkboxes
    const filterCheckboxes = companyFilterContainer.querySelectorAll('input[type="checkbox"]');
    filterCheckboxes.forEach(checkbox => {
      checkbox.addEventListener('change', applyFilters);
    });
  }
  
  /**
   * Apply filters to the table
   */
  function applyFilters() {
    if (!dataTable) return;
    
    const rows = dataTable.querySelectorAll('tbody tr');
    const activeFilters = {};
    const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
    
    console.log('Applying filters, checked filters:', filterCheckboxes.length);
    
    // Build active filters map
    filterCheckboxes.forEach(checkbox => {
      const filterType = checkbox.getAttribute('data-filter');
      const filterValue = checkbox.getAttribute('data-value');
      
      if (!activeFilters[filterType]) {
        activeFilters[filterType] = [];
      }
      
      activeFilters[filterType].push(filterValue);
    });
    
    console.log('Active filters:', activeFilters);
    
    // Show all rows if no filters are active
    if (Object.keys(activeFilters).length === 0) {
      rows.forEach(row => {
        row.style.display = '';
      });
      return;
    }
    
    // Apply filtering
    let visibleCount = 0;
    
    rows.forEach(row => {
      let shouldShow = true;
      
      // Check each filter type
      for (const filterType in activeFilters) {
        let matches = false;
        
        if (filterType === 'org') {
          // Get the org badge from the row
          const orgBadge = row.querySelector('.org-badge');
          if (orgBadge) {
            const orgText = orgBadge.textContent.trim();
            
            matches = activeFilters[filterType].some(value => 
              orgText === value
            );
          }
        }
        else if (filterType === 'company') {
          // Get the company badge from the row
          const companyBadge = row.querySelector('.company-badge');
          if (companyBadge) {
            const companyText = companyBadge.textContent.trim();
            
            matches = activeFilters[filterType].some(value => 
              companyText === value
            );
          }
        }
        else if (filterType === 'status') {
          // Get the status from the row attribute
          const rowStatus = row.getAttribute('data-status');
          
          matches = activeFilters[filterType].some(value => 
            rowStatus === value
          );
        }
        
        if (!matches) {
          shouldShow = false;
          break;
        }
      }
      
      row.style.display = shouldShow ? '' : 'none';
      
      if (shouldShow) {
        visibleCount++;
      }
    });
    
    console.log('Filter applied, visible rows:', visibleCount);
  }
    
  /**
   * Initialize delete modal functionality
   */
  function initDeleteModal() {
    const deleteButtons = document.querySelectorAll('.action-btn.delete');
    const deleteModal = document.getElementById('deleteModal');
    const deleteForm = document.getElementById('deleteForm');
    const deleteId = document.getElementById('deleteId');
    const deleteName = document.getElementById('deleteName');
    const modalClose = document.querySelector('.modal-close');
    const modalCancel = document.querySelector('.modal-cancel');
    
    if (!deleteModal) {
      console.warn('Delete modal not found');
      return;
    }
    
    deleteButtons.forEach(button => {
      button.addEventListener('click', function() {
        const id = this.getAttribute('data-id');
        const name = this.getAttribute('data-name');
        
        deleteId.value = id;
        deleteName.textContent = name;
        deleteModal.style.display = 'block';
      });
    });
    
    // Close modal when clicking on X or Cancel
    modalClose.addEventListener('click', function() {
      deleteModal.style.display = 'none';
    });
    
    modalCancel.addEventListener('click', function() {
      deleteModal.style.display = 'none';
    });
    
    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
      if (event.target == deleteModal) {
        deleteModal.style.display = 'none';
      }
    });
  }
  
  /**
   * Initialize export panel functionality
   */
  function initExportPanel() {
    const exportToggle = document.getElementById('exportToggle');
    const exportPanel = document.getElementById('exportPanel');
    const closeExport = document.getElementById('closeExport');
    
    if (!exportPanel || !exportToggle) {
      console.warn('Export panel elements not found');
      return;
    }
    
    exportToggle.addEventListener('click', function() {
      exportPanel.classList.toggle('visible');
    });
    
    closeExport.addEventListener('click', function() {
      exportPanel.classList.remove('visible');
    });
  }
  
  // Listen for theme changes
  document.addEventListener('themechange', function(event) {
    console.log('Theme changed in osoba list page:', event.detail.theme);
    // Update badge colors when theme changes
    updateBadgeColors();
    updateStatusIndicators();
  });
});
//# sourceMappingURL=listosoba.40b9e0eb.js.map