/**
 * Vrste Zadataka (Task Types) List JavaScript
 * Specific functionality for task types list view
 * NOTE: This script depends on table.js and should be loaded AFTER it
 */
document.addEventListener('DOMContentLoaded', function() {
  // Find the table (already initialized by table.js)
  const dataTable = document.getElementById('dataTable');
  
  if (!dataTable) {
    console.warn('Task types table not found');
    return;
  }
  
  // Add task type-specific enhancements
  updateOrgBadgeColors();
  addHoverEffects();
  initFilters(); // Add filters initialization
  initDeleteModal(); // Initialize delete confirmation modal
  initExportPanel(); // Initialize export panel
  
  /**
   * Add hover effects to rows and elements
   */
  function addHoverEffects() {
    const rows = dataTable.querySelectorAll('tbody tr');
    
    rows.forEach(row => {
      const orgBadge = row.querySelector('.org-badge');
      
      row.addEventListener('mouseenter', function() {
        if (orgBadge) orgBadge.classList.add('badge-highlight');
      });
      
      row.addEventListener('mouseleave', function() {
        if (orgBadge) orgBadge.classList.remove('badge-highlight');
      });
    });
  }
  
  /**
   * Set different colors for different org units
   */
  function updateOrgBadgeColors() {
    const orgBadges = document.querySelectorAll('.org-badge');
    
    orgBadges.forEach(badge => {
      const orgText = badge.textContent.trim();
      badge.setAttribute('data-org', orgText);
      
      // Add title attribute for better accessibility
      badge.setAttribute('title', 'Organizacijska jedinica: ' + orgText);
    });
  }
  
  /**
   * Initialize filter functionality for organization units
   */
  function initFilters() {
    const orgFilterContainer = document.getElementById('orgFilter');
    if (!orgFilterContainer) {
      console.warn('Organization filter container not found');
      return;
    }
    
    // Get all unique org units from the table
    const orgUnits = new Set();
    const orgBadges = document.querySelectorAll('.org-badge');
    
    orgBadges.forEach(badge => {
      const orgText = badge.textContent.trim();
      if (orgText) {
        orgUnits.add(orgText);
      }
    });
    
    // Clear existing content
    orgFilterContainer.innerHTML = '';
    
    // Create filter checkboxes for each org unit
    orgUnits.forEach(orgUnit => {
      const label = document.createElement('label');
      label.className = 'filter-checkbox';
      
      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.setAttribute('data-filter', 'org');
      checkbox.setAttribute('data-value', orgUnit);
      
      label.appendChild(checkbox);
      label.appendChild(document.createTextNode(' ' + orgUnit));
      
      orgFilterContainer.appendChild(label);
    });
    
    // Add event listeners to the new checkboxes
    const filterCheckboxes = orgFilterContainer.querySelectorAll('input[type="checkbox"]');
    filterCheckboxes.forEach(checkbox => {
      checkbox.addEventListener('change', applyFilters);
    });
  }
  
  /**
   * Apply filters to the table
   */
  function applyFilters() {
    if (!dataTable) return;
    
    const rows = dataTable.querySelectorAll('tbody tr');
    const activeFilters = {};
    const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
    
    console.log('Applying filters, checked filters:', filterCheckboxes.length);
    
    // Build active filters map
    filterCheckboxes.forEach(checkbox => {
      const filterType = checkbox.getAttribute('data-filter');
      const filterValue = checkbox.getAttribute('data-value');
      
      if (!activeFilters[filterType]) {
        activeFilters[filterType] = [];
      }
      
      activeFilters[filterType].push(filterValue);
    });
    
    console.log('Active filters:', activeFilters);
    
    // Show all rows if no filters are active
    if (Object.keys(activeFilters).length === 0) {
      rows.forEach(row => {
        row.style.display = '';
      });
      return;
    }
    
    // Apply filtering
    let visibleCount = 0;
    
    rows.forEach(row => {
      let shouldShow = true;
      
      // Check each filter type
      for (const filterType in activeFilters) {
        let matches = false;
        
        if (filterType === 'org') {
          // Get the org badge from the row
          const orgBadge = row.querySelector('.org-badge');
          if (orgBadge) {
            const orgText = orgBadge.textContent.trim();
            
            matches = activeFilters[filterType].some(value => 
              orgText === value
            );
          }
        }
        
        if (!matches) {
          shouldShow = false;
          break;
        }
      }
      
      row.style.display = shouldShow ? '' : 'none';
      
      if (shouldShow) {
        visibleCount++;
      }
    });
    
    console.log('Filter applied, visible rows:', visibleCount);
  }
    
  /**
   * Initialize delete modal functionality
   */
  function initDeleteModal() {
    const deleteButtons = document.querySelectorAll('.action-btn.delete');
    const deleteModal = document.getElementById('deleteModal');
    const deleteForm = document.getElementById('deleteForm');
    const deleteId = document.getElementById('deleteId');
    const deleteName = document.getElementById('deleteName');
    const modalClose = document.querySelector('.modal-close');
    const modalCancel = document.querySelector('.modal-cancel');
    
    if (!deleteModal) {
      console.warn('Delete modal not found');
      return;
    }
    
    deleteButtons.forEach(button => {
      button.addEventListener('click', function() {
        const id = this.getAttribute('data-id');
        const name = this.getAttribute('data-name');
        
        deleteId.value = id;
        deleteName.textContent = name;
        deleteModal.style.display = 'block';
      });
    });
    
    // Close modal when clicking on X or Cancel
    modalClose.addEventListener('click', function() {
      deleteModal.style.display = 'none';
    });
    
    modalCancel.addEventListener('click', function() {
      deleteModal.style.display = 'none';
    });
    
    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
      if (event.target == deleteModal) {
        deleteModal.style.display = 'none';
      }
    });
  }
  
  /**
   * Initialize export panel functionality
   */
  function initExportPanel() {
    const exportToggle = document.getElementById('exportToggle');
    const exportPanel = document.getElementById('exportPanel');
    const closeExport = document.getElementById('closeExport');
    
    if (!exportPanel || !exportToggle) {
      console.warn('Export panel elements not found');
      return;
    }
    
    exportToggle.addEventListener('click', function() {
      exportPanel.classList.toggle('visible');
    });
    
    closeExport.addEventListener('click', function() {
      exportPanel.classList.remove('visible');
    });
  }
  
  // Listen for theme changes
  document.addEventListener('themechange', function(event) {
    console.log('Theme changed in vrsta list page:', event.detail.theme);
    // Update org badge colors when theme changes
    updateOrgBadgeColors();
  });
});
//# sourceMappingURL=listvrsta.18f55048.js.map