// Real Data Tasks Admin Dashboard JavaScript
document.addEventListener('DOMContentLoaded', function() {
  // Initialize all dashboard functionality
  initializeCounterAnimations();
  initializeInteractiveElements();
  initializeRealTimeUpdates();
  
  // Set up periodic refresh for real-time data
  if (typeof refreshTasksAdminInterval === 'undefined') {
    let refreshTasksAdminInterval = setInterval(refreshRealTimeData, 300000); // Every 5 minutes
  }
});

/**
 * Initialize counter animations for KPI values
 */
function initializeCounterAnimations() {
  // Animate dashboard cards
  const cardValues = document.querySelectorAll('.card-value');
  cardValues.forEach(valueElement => {
    animateCounter(valueElement, 1200);
  });
  
  // Animate metric values
  const metricValues = document.querySelectorAll('.metric-value');
  metricValues.forEach(valueElement => {
    animateCounter(valueElement, 1000);
  });
}

/**
 * Animate a counter element from 0 to its final value
 */
function animateCounter(element, duration) {
  const text = element.textContent.trim();
  let finalValue, suffix = '';
  
  // Handle different formats
  if (text.includes('€')) {
    finalValue = parseFloat(text.replace('€', '').replace(',', '.'));
    suffix = '€';
  }
  else if (text.includes('h')) {
    finalValue = parseFloat(text.replace('h', '').trim());
    suffix = ' h';
  }
  // Handle pure numbers
  else if (/^\d+$/.test(text)) {
    finalValue = parseInt(text);
  }
  // Handle decimal numbers
  else if (/^\d+\.\d+$/.test(text)) {
    finalValue = parseFloat(text);
  }
  // Skip non-numeric values
  else {
    return;
  }
  
  if (isNaN(finalValue)) return;
  
  let startTimestamp = null;
  const startValue = 0;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    const currentValue = progress * (finalValue - startValue) + startValue;
    
    if (suffix === '€') {
      element.textContent = currentValue.toFixed(2) + suffix;
    } else if (suffix === ' h') {
      element.textContent = currentValue.toFixed(1) + suffix;
    } else if (finalValue % 1 === 0) {
      element.textContent = Math.floor(currentValue) + suffix;
    } else {
      element.textContent = currentValue.toFixed(1) + suffix;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      if (suffix === '€') {
        element.textContent = finalValue.toFixed(2) + suffix;
      } else if (suffix === ' h') {
        element.textContent = finalValue.toFixed(1) + suffix;
      } else if (finalValue % 1 === 0) {
        element.textContent = finalValue + suffix;
      } else {
        element.textContent = finalValue.toFixed(1) + suffix;
      }
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Initialize interactive elements
 */
function initializeInteractiveElements() {
  // Add hover effects to performer items
  const performerItems = document.querySelectorAll('.performer-item');
  performerItems.forEach(item => {
    item.addEventListener('mouseenter', function() {
      this.style.transform = 'translateX(5px)';
    });
    
    item.addEventListener('mouseleave', function() {
      this.style.transform = 'translateX(0)';
    });
  });
  
  // Add click handlers to recent task items for potential drill-down
  const taskItems = document.querySelectorAll('.recent-task-item');
  taskItems.forEach(item => {
    item.addEventListener('click', function() {
      const taskAddress = this.querySelector('.task-address').textContent;
      console.log('Clicked on task:', taskAddress);
      // Potential navigation to task detail view
      // window.location.href = `/tasks/edit?address=${encodeURIComponent(taskAddress)}`;
    });
  });
  
  // Add click handlers to metric items for potential drill-down
  const metricItems = document.querySelectorAll('.metric-item');
  metricItems.forEach(item => {
    item.addEventListener('click', function() {
      const label = this.querySelector('.metric-label').textContent;
      console.log('Clicked on metric:', label);
      // Potential navigation based on metric type
    });
  });
}

/**
 * Initialize real-time updates for dashboard data
 */
function initializeRealTimeUpdates() {
  // Add visual indicators for real-time data
  const dashboardCards = document.querySelectorAll('.dashboard-card');
  dashboardCards.forEach(card => {
    card.addEventListener('mouseenter', function() {
      this.style.transform = 'translateY(-8px)';
    });
    
    card.addEventListener('mouseleave', function() {
      this.style.transform = 'translateY(0)';
    });
  });
  
  // Initialize pulse animation for tasks in progress if count > 0
  const inProgressCard = document.querySelector('.dashboard-card:nth-child(4) .card-value');
  if (inProgressCard && parseInt(inProgressCard.textContent) > 0) {
    inProgressCard.parentElement.parentElement.classList.add('urgent-attention');
  }
}

/**
 * Refresh real-time data via AJAX
 */
function refreshRealTimeData() {
  fetch('/tasks/admin-stats', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    }
  })
    .then(response => {
      if (!response.ok) {
        throw new Error('Network response was not ok');
      }
      return response.json();
    })
    .then(data => {
      updateDashboardValues(data);
      showUpdateIndicator();
    })
    .catch(error => {
      console.error('Error refreshing real-time data:', error);
      showErrorIndicator();
    });
}

/**
 * Update dashboard values with new data
 */
function updateDashboardValues(data) {
  // Update main dashboard cards
  updateElementValue('.dashboard-card:nth-child(1) .card-value', data.taskTypesCount);
  updateElementValue('.dashboard-card:nth-child(2) .card-value', data.usersCount);
  updateElementValue('.dashboard-card:nth-child(3) .card-value', data.completedTasksCount);
  updateElementValue('.dashboard-card:nth-child(4) .card-value', data.inProgressTasksCount);
  
  // Update key metrics
  updateElementValue('.metric-item:nth-child(1) .metric-value', data.totalRevenue, '€');
  updateElementValue('.metric-item:nth-child(2) .metric-value', data.averageResolutionTime, ' h');
  updateElementValue('.metric-item:nth-child(3) .metric-value', 
    data.tasksByStatus ? data.tasksByStatus.reduce((sum, item) => sum + parseInt(item.count), 0) : 0);
  
  // High priority tasks
  const highPriorityCount = data.tasksByPriority ? 
    (data.tasksByPriority.find(p => p.priority === 'High')?.count || 0) : 0;
  updateElementValue('.metric-item:nth-child(4) .metric-value', highPriorityCount);
  
  // Update urgency indicator for tasks in progress
  const inProgressElement = document.querySelector('.dashboard-card:nth-child(4)');
  if (data.inProgressTasksCount > 0) {
    inProgressElement.classList.add('urgent-attention');
  } else {
    inProgressElement.classList.remove('urgent-attention');
  }
}

/**
 * Update a specific element's value with animation
 */
function updateElementValue(selector, newValue, suffix = '') {
  const element = document.querySelector(selector);
  if (!element || newValue === undefined) return;
  
  const currentText = element.textContent.trim();
  const currentValue = parseFloat(currentText.replace(/[^\d.-]/g, '')) || 0;
  const targetValue = parseFloat(newValue) || 0;
  
  // Only animate if value has changed
  if (currentValue !== targetValue) {
    // Add update flash effect
    element.classList.add('value-updated');
    
    // Animate to new value
    animateValueChange(element, currentValue, targetValue, suffix, 800);
    
    // Remove flash effect after animation
    setTimeout(() => {
      element.classList.remove('value-updated');
    }, 1000);
  }
}

/**
 * Animate value change from current to new value
 */
function animateValueChange(element, startValue, endValue, suffix, duration) {
  let startTimestamp = null;
  
  const step = (timestamp) => {
    if (!startTimestamp) startTimestamp = timestamp;
    const progress = Math.min((timestamp - startTimestamp) / duration, 1);
    
    const currentValue = startValue + (endValue - startValue) * progress;
    
    if (suffix === '€') {
      element.textContent = currentValue.toFixed(2) + suffix;
    } else if (suffix === ' h') {
      element.textContent = currentValue.toFixed(1) + suffix;
    } else {
      element.textContent = Math.round(currentValue) + suffix;
    }
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    } else {
      // Ensure final value is exact
      if (suffix === '€') {
        element.textContent = endValue.toFixed(2) + suffix;
      } else if (suffix === ' h') {
        element.textContent = endValue.toFixed(1) + suffix;
      } else {
        element.textContent = Math.round(endValue) + suffix;
      }
    }
  };
  
  window.requestAnimationFrame(step);
}

/**
 * Show update indicator
 */
function showUpdateIndicator() {
  // Create or update a small indicator showing last update time
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  const now = new Date();
  indicator.innerHTML = `
    <i class="fas fa-sync-alt"></i>
    <span>Ažurirano: ${now.toLocaleTimeString('hr-HR')}</span>
  `;
  
  // Add pulse animation
  indicator.classList.add('updated');
  setTimeout(() => {
    indicator.classList.remove('updated');
  }, 1000);
}

/**
 * Show error indicator
 */
function showErrorIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator error';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-exclamation-triangle"></i>
    <span>Greška pri ažuriranju</span>
  `;
  
  indicator.classList.add('error');
  setTimeout(() => {
    indicator.classList.remove('error');
  }, 3000);
}

/**
 * Manual refresh function that can be called by user
 */
function manualRefresh() {
  showLoadingIndicator();
  refreshRealTimeData();
}

/**
 * Show loading indicator
 */
function showLoadingIndicator() {
  let indicator = document.querySelector('.update-indicator');
  if (!indicator) {
    indicator = document.createElement('div');
    indicator.className = 'update-indicator';
    document.querySelector('.content-header').appendChild(indicator);
  }
  
  indicator.innerHTML = `
    <i class="fas fa-spinner fa-spin"></i>
    <span>Ažuriranje...</span>
  `;
}

// Export functions for external use
window.TasksAdminDashboard = {
  refreshStats: refreshRealTimeData,
  updateData: updateDashboardValues,
  manualRefresh: manualRefresh,
  animateCounter: animateCounter
};
//# sourceMappingURL=tasksadmin.2aa1af41.js.map