/**
 * Enhanced Putni Nalozi (Travel Orders) List JavaScript
 * Updated with month, driver, vehicle, and registration filters + export functionality + top 10 view
 */
document.addEventListener('DOMContentLoaded', function() {
    // Find the table (already initialized by table.js)
    const dataTable = document.getElementById('dataTable');
    
    if (!dataTable) {
      console.warn('Travel orders table not found');
      return;
    }
    
    // Global variables for managing view state
    let allRows = [];
    let isShowingAll = false;
    let activeFilters = {};
    
    // Store all rows at initialization
    storeAllRows();
    
    // Add travel order-specific enhancements
    updateVehicleBadgeColors();
    addHoverEffects();
    initFilters(); // Add filters initialization
    initDeleteModal(); // Initialize delete confirmation modal
    initExportPanel(); // Initialize export panel
    initShowAllButton(); // Initialize show all functionality
    
    // Set default sorting and show top 10
    setDefaultSorting();
    showTopTen();
    
    /**
     * Add hover effects to rows and elements
     */
    function addHoverEffects() {
      const rows = dataTable.querySelectorAll('tbody tr');
      
      rows.forEach(row => {
        const vehicleBadge = row.querySelector('.vehicle-badge');
        
        row.addEventListener('mouseenter', function() {
          if (vehicleBadge) vehicleBadge.classList.add('badge-highlight');
        });
        
        row.addEventListener('mouseleave', function() {
          if (vehicleBadge) vehicleBadge.classList.remove('badge-highlight');
        });
      });
    }
    
    /**
     * Set different colors for different vehicles
     */
    function updateVehicleBadgeColors() {
      const vehicleBadges = document.querySelectorAll('.vehicle-badge');
      
      vehicleBadges.forEach(badge => {
        const vehicleText = badge.textContent.trim();
        badge.setAttribute('data-vehicle', vehicleText);
        
        // Add title attribute for better accessibility
        badge.setAttribute('title', 'Vozilo: ' + vehicleText);
      });
    }
    
    /**
     * Initialize filter functionality for months, drivers, vehicles, and registrations
     */
    function initFilters() {
      const monthFilterContainer = document.getElementById('monthFilter');
      const driverFilterContainer = document.getElementById('driverFilter');
      const vehicleFilterContainer = document.getElementById('vehicleFilter');
      const registrationFilterContainer = document.getElementById('registrationFilter');
      
      if (!monthFilterContainer || !driverFilterContainer || !vehicleFilterContainer || !registrationFilterContainer) {
        console.warn('Filter containers not found');
        return;
      }
      
      // Get all unique values from the table
      const months = new Set();
      const drivers = new Set();
      const vehicles = new Set();
      const registrations = new Set();
      
      const rows = dataTable.querySelectorAll('tbody tr');
      
      rows.forEach(row => {
        // Extract month from date
        const dateCell = row.cells[1]; // Date column
        if (dateCell) {
          const dateText = dateCell.textContent.trim();
          const dateMatch = dateText.match(/(\d{2})\.(\d{2})\.(\d{4})/);
          if (dateMatch) {
            const month = dateMatch[2]; // MM
            const year = dateMatch[3]; // YYYY
            const monthYear = `${month}/${year}`;
            months.add(monthYear);
          }
        }
        
        // Extract driver
        const driverCell = row.querySelector('.driver-name');
        if (driverCell) {
          const driverText = driverCell.textContent.trim();
          if (driverText) {
            drivers.add(driverText);
          }
        }
        
        // Extract vehicle
        const vehicleCell = row.querySelector('.vehicle-badge');
        if (vehicleCell) {
          const vehicleText = vehicleCell.textContent.trim();
          if (vehicleText) {
            vehicles.add(vehicleText);
          }
        }
        
        // Extract registration
        const registrationCell = row.cells[4]; // Registration column
        if (registrationCell) {
          const registrationText = registrationCell.textContent.trim();
          if (registrationText) {
            registrations.add(registrationText);
          }
        }
      });
      
      // Clear existing content
      monthFilterContainer.innerHTML = '';
      driverFilterContainer.innerHTML = '';
      vehicleFilterContainer.innerHTML = '';
      registrationFilterContainer.innerHTML = '';
      
      // Create filter checkboxes for each month (sorted)
      const sortedMonths = Array.from(months).sort((a, b) => {
        const [monthA, yearA] = a.split('/');
        const [monthB, yearB] = b.split('/');
        const dateA = new Date(parseInt(yearA), parseInt(monthA) - 1);
        const dateB = new Date(parseInt(yearB), parseInt(monthB) - 1);
        return dateB - dateA; // Newest first
      });
      
      sortedMonths.forEach(month => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'month');
        checkbox.setAttribute('data-value', month);
        
        const [mm, yyyy] = month.split('/');
        const monthNames = ['Siječanj', 'Veljača', 'Ožujak', 'Travanj', 'Svibanj', 'Lipanj',
                           'Srpanj', 'Kolovoz', 'Rujan', 'Listopad', 'Studeni', 'Prosinac'];
        const monthName = monthNames[parseInt(mm) - 1];
        const displayText = `${monthName} ${yyyy}`;
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + displayText));
        
        monthFilterContainer.appendChild(label);
      });
      
      // Create filter checkboxes for each driver (sorted alphabetically)
      const sortedDrivers = Array.from(drivers).sort();
      sortedDrivers.forEach(driver => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'driver');
        checkbox.setAttribute('data-value', driver);
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + driver));
        
        driverFilterContainer.appendChild(label);
      });
      
      // Create filter checkboxes for each vehicle (sorted alphabetically)
      const sortedVehicles = Array.from(vehicles).sort();
      sortedVehicles.forEach(vehicle => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'vehicle');
        checkbox.setAttribute('data-value', vehicle);
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + vehicle));
        
        vehicleFilterContainer.appendChild(label);
      });
      
      // Create filter checkboxes for each registration (sorted alphabetically)
      const sortedRegistrations = Array.from(registrations).sort();
      sortedRegistrations.forEach(registration => {
        const label = document.createElement('label');
        label.className = 'filter-checkbox';
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.setAttribute('data-filter', 'registration');
        checkbox.setAttribute('data-value', registration);
        
        label.appendChild(checkbox);
        label.appendChild(document.createTextNode(' ' + registration));
        
        registrationFilterContainer.appendChild(label);
      });
      
      // Add event listeners to the new checkboxes
      const filterCheckboxes = document.querySelectorAll('.filter-checkbox input[type="checkbox"]');
      filterCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
          applyFilters();
          updateFilterButtonState();
        });
      });
    }
    
    /**
     * Apply filters to the table
     */
    function applyFilters() {
      if (!dataTable) return;
      
      const filterCheckboxes = document.querySelectorAll('.filter-checkbox input:checked');
      
      console.log('Applying filters, checked filters:', filterCheckboxes.length);
      
      // Build active filters map
      activeFilters = {};
      filterCheckboxes.forEach(checkbox => {
        const filterType = checkbox.getAttribute('data-filter');
        const filterValue = checkbox.getAttribute('data-value');
        
        if (!activeFilters[filterType]) {
          activeFilters[filterType] = [];
        }
        
        activeFilters[filterType].push(filterValue);
      });
      
      console.log('Active filters:', activeFilters);
      
      // Apply filtering based on current view mode
      if (isShowingAll) {
        showAllRows();
      } else {
        showTopTen();
      }
      
      updateExportForm(); // Update export form with current data
      updateViewControls();
      
      const totalMatchingRows = getTotalMatchingRows();
      console.log('Filter applied, total matching rows:', totalMatchingRows);
    }
    
    /**
     * Update export form with current filtered data (always exports ALL matching data)
     */
    function updateExportForm(visibleRows = null) {
      const exportForm = document.getElementById('exportForm');
      if (!exportForm) return;
      
      // Remove existing filtered data inputs
      const existingInputs = exportForm.querySelectorAll('input[name^="filtered_data"]');
      existingInputs.forEach(input => input.remove());
      
      // Always use ALL rows that match filters for export, not just visible ones
      const rowsToExport = allRows.filter(row => rowMatchesFilters(row));
      
      // Add filtered data to form
      rowsToExport.forEach((row, index) => {
        // Extract data from each row
        const rowData = extractRowData(row);
        
        // Create hidden inputs for each field
        Object.keys(rowData).forEach(key => {
          const input = document.createElement('input');
          input.type = 'hidden';
          input.name = `filtered_data[${index}][${key}]`;
          input.value = rowData[key] || '';
          exportForm.appendChild(input);
        });
      });
      
      console.log('Export form updated with', rowsToExport.length, 'total matching rows (not just visible)');
    }
    
    /**
     * Extract data from a table row
     */
    function extractRowData(row) {
      const dateCell = row.cells[1];
      const driverCell = row.querySelector('.driver-name');
      const vehicleCell = row.querySelector('.vehicle-badge');
      const registrationCell = row.cells[4];
      const directionCell = row.cells[5];
      const distanceCell = row.querySelector('.distance-display');
      const fuelCell = row.querySelector('.cost-display');
      
      // Extract date and time
      let datum = '';
      let vrijeme_polaska = '';
      let vrijeme_dolaska = '';
      
      if (dateCell) {
        const dateText = dateCell.textContent.trim();
        const dateMatch = dateText.match(/(\d{2})\.(\d{2})\.(\d{4})/);
        if (dateMatch) {
          datum = `${dateMatch[3]}-${dateMatch[2]}-${dateMatch[1]}`; // Convert to YYYY-MM-DD
        }
        
        const timeMatch = dateText.match(/(\d{2}:\d{2}) - (\d{2}:\d{2})/);
        if (timeMatch) {
          vrijeme_polaska = timeMatch[1];
          vrijeme_dolaska = timeMatch[2];
        }
      }
      
      return {
        datum: datum,
        vrijeme_polaska: vrijeme_polaska,
        vrijeme_dolaska: vrijeme_dolaska,
        vozac: driverCell ? driverCell.textContent.trim() : '',
        vozilo: vehicleCell ? vehicleCell.textContent.trim() : '',
        registracija: registrationCell ? registrationCell.textContent.trim() : '',
        pravac_kretanja: directionCell ? directionCell.textContent.trim() : '',
        prijedeni_km: distanceCell ? distanceCell.textContent.replace(' km', '').trim() : '',
        gorivo_uk_cijena: fuelCell ? fuelCell.textContent.replace(' €', '').replace(',', '.').trim() : ''
      };
    }
      
    /**
     * Initialize delete modal functionality
     */
    function initDeleteModal() {
      const deleteButtons = document.querySelectorAll('.action-btn.delete');
      const deleteModal = document.getElementById('deleteModal');
      const deleteForm = document.getElementById('deleteForm');
      const deleteId = document.getElementById('deleteId');
      const deleteName = document.getElementById('deleteName');
      const modalClose = document.querySelector('.modal-close');
      const modalCancel = document.querySelector('.modal-cancel');
      
      if (!deleteModal) {
        console.warn('Delete modal not found');
        return;
      }
      
      deleteButtons.forEach(button => {
        button.addEventListener('click', function() {
          const id = this.getAttribute('data-id');
          const name = this.getAttribute('data-name');
          
          deleteId.value = id;
          deleteName.textContent = name;
          deleteModal.style.display = 'block';
        });
      });
      
      // Close modal when clicking on X or Cancel
      modalClose.addEventListener('click', function() {
        deleteModal.style.display = 'none';
      });
      
      modalCancel.addEventListener('click', function() {
        deleteModal.style.display = 'none';
      });
      
      // Close modal when clicking outside
      window.addEventListener('click', function(event) {
        if (event.target == deleteModal) {
          deleteModal.style.display = 'none';
        }
      });
    }
    
    /**
     * Initialize export panel functionality
     */
    function initExportPanel() {
      const exportToggle = document.getElementById('exportToggle');
      const exportPanel = document.getElementById('exportPanel');
      const closeExport = document.getElementById('closeExport');
      
      if (!exportPanel || !exportToggle) {
        console.warn('Export panel elements not found');
        return;
      }
      
      exportToggle.addEventListener('click', function() {
        exportPanel.classList.toggle('visible');
        updateExportForm(); // Update form when panel opens
      });
      
      closeExport.addEventListener('click', function() {
        exportPanel.classList.remove('visible');
      });
    }
    
    /**
     * Store all table rows at initialization
     */
    function storeAllRows() {
      const tbody = dataTable.querySelector('tbody');
      allRows = Array.from(tbody.querySelectorAll('tr'));
      console.log('Stored', allRows.length, 'total rows');
    }
    
    /**
     * Initialize show all button functionality
     */
    function initShowAllButton() {
      // Create show all button container
      const tableContainer = document.querySelector('.table-responsive');
      if (!tableContainer) return;
      
      const buttonContainer = document.createElement('div');
      buttonContainer.className = 'table-view-controls';
      buttonContainer.innerHTML = `
        <div class="view-info">
          <span id="viewStatus">Displayed: <strong>10</strong> of <strong>${allRows.length}</strong> travel orders</span>
        </div>
        <button type="button" id="showAllBtn" class="show-all-button">
          <i class="fas fa-list"></i> Show all
        </button>
      `;
      
      // Insert after table
      tableContainer.parentNode.insertBefore(buttonContainer, tableContainer.nextSibling);
      
      // Add event listener
      const showAllBtn = document.getElementById('showAllBtn');
      showAllBtn.addEventListener('click', toggleShowAll);
      
      updateViewControls();
    }
    
    /**
     * Show only top 10 rows (after sorting)
     */
    function showTopTen() {
      if (isShowingAll) return;
      
      const tbody = dataTable.querySelector('tbody');
      const currentRows = Array.from(tbody.querySelectorAll('tr'));
      
      // Hide all rows first
      currentRows.forEach(row => row.style.display = 'none');
      
      // Show only first 10 visible rows
      let visibleCount = 0;
      for (let row of currentRows) {
        if (visibleCount >= 10) break;
        
        // Check if row matches current filters
        if (rowMatchesFilters(row)) {
          row.style.display = '';
          visibleCount++;
        }
      }
      
      updateViewControls();
      console.log('Showing top 10 rows');
    }
    
    /**
     * Show all rows
     */
    function showAllRows() {
      const tbody = dataTable.querySelector('tbody');
      const currentRows = Array.from(tbody.querySelectorAll('tr'));
      
      // Show rows that match current filters
      currentRows.forEach(row => {
        if (rowMatchesFilters(row)) {
          row.style.display = '';
        } else {
          row.style.display = 'none';
        }
      });
      
      updateViewControls();
      console.log('Showing all rows');
    }
    
    /**
     * Toggle between show all and top 10
     */
    function toggleShowAll() {
      isShowingAll = !isShowingAll;
      
      if (isShowingAll) {
        showAllRows();
      } else {
        showTopTen();
      }
    }
    
    /**
     * Update view controls (button text and status)
     */
    function updateViewControls() {
      const showAllBtn = document.getElementById('showAllBtn');
      const viewStatus = document.getElementById('viewStatus');
      
      if (!showAllBtn || !viewStatus) return;
      
      const tbody = dataTable.querySelector('tbody');
      const visibleRows = Array.from(tbody.querySelectorAll('tr')).filter(row => row.style.display !== 'none');
      const totalMatchingRows = getTotalMatchingRows();
      
      if (isShowingAll) {
        showAllBtn.innerHTML = '<i class="fas fa-list-ol"></i> Show top 10';
        viewStatus.innerHTML = `Displayed: <strong>${visibleRows.length}</strong> of <strong>${allRows.length}</strong> travel orders`;
      } else {
        showAllBtn.innerHTML = '<i class="fas fa-list"></i> Show all';
        const showing = Math.min(10, totalMatchingRows);
        viewStatus.innerHTML = `Displayed: <strong>${showing}</strong> of <strong>${totalMatchingRows}</strong> travel orders`;
      }
    }
    
    /**
     * Get total number of rows matching current filters
     */
    function getTotalMatchingRows() {
      return allRows.filter(row => rowMatchesFilters(row)).length;
    }
    
    /**
     * Check if a row matches current active filters
     */
    function rowMatchesFilters(row) {
      // If no active filters, all rows match
      if (Object.keys(activeFilters).length === 0) {
        return true;
      }
      
      // Check each filter type
      for (const filterType in activeFilters) {
        let matches = false;
        
        if (filterType === 'month') {
          const dateCell = row.cells[1];
          if (dateCell) {
            const dateText = dateCell.textContent.trim();
            const dateMatch = dateText.match(/(\d{2})\.(\d{2})\.(\d{4})/);
            if (dateMatch) {
              const month = dateMatch[2];
              const year = dateMatch[3];
              const monthYear = `${month}/${year}`;
              
              matches = activeFilters[filterType].some(value => 
                monthYear === value
              );
            }
          }
        } else if (filterType === 'vehicle') {
          const vehicleBadge = row.querySelector('.vehicle-badge');
          if (vehicleBadge) {
            const vehicleText = vehicleBadge.textContent.trim();
            matches = activeFilters[filterType].some(value => 
              vehicleText === value
            );
          }
        } else if (filterType === 'driver') {
          const driverCell = row.querySelector('.driver-name');
          if (driverCell) {
            const driverText = driverCell.textContent.trim();
            matches = activeFilters[filterType].some(value => 
              driverText === value
            );
          }
        } else if (filterType === 'registration') {
          const registrationCell = row.cells[4];
          if (registrationCell) {
            const registrationText = registrationCell.textContent.trim();
            matches = activeFilters[filterType].some(value => 
              registrationText === value
            );
          }
        }
        
        if (!matches) {
          return false;
        }
      }
      
      return true;
    }
    function setDefaultSorting() {
      // Try to use existing table.js sorting first
      const dateHeader = dataTable.querySelector('thead th:nth-child(2)');
      
      if (dateHeader && dateHeader.classList.contains('sortable')) {
        // Trigger click to sort by date
        dateHeader.click();
        
        // Check if it's ascending, if so click again to make it descending
        setTimeout(() => {
          if (dateHeader.classList.contains('sort-asc')) {
            dateHeader.click(); // Click again to make it descending
          }
        }, 100);
      } else {
        // Fallback: Manual sorting if table.js sorting is not available
        sortTableByDate();
      }
    }
    
    /**
     * Manually sort table rows by date (newest first)
     */
    function sortTableByDate() {
      const tbody = dataTable.querySelector('tbody');
      const rows = Array.from(tbody.querySelectorAll('tr'));
      
      rows.sort((a, b) => {
        const dateA = extractDateFromRow(a);
        const dateB = extractDateFromRow(b);
        
        // Sort descending (newest first)
        return dateB - dateA;
      });
      
      // Clear tbody and re-append sorted rows
      tbody.innerHTML = '';
      rows.forEach(row => tbody.appendChild(row));
    }
    
    /**
     * Extract date from table row for sorting
     */
    function extractDateFromRow(row) {
      const dateCell = row.cells[1]; // Date column
      if (!dateCell) return new Date(0);
      
      const dateText = dateCell.textContent.trim();
      const dateMatch = dateText.match(/(\d{2})\.(\d{2})\.(\d{4})/);
      
      if (dateMatch) {
        const day = dateMatch[1];
        const month = dateMatch[2];
        const year = dateMatch[3];
        return new Date(year, month - 1, day); // month is 0-indexed
      }
      
      return new Date(0); // Default date if parsing fails
    }
    
    /**
     * Update filter button state to show active filters
     */
    function updateFilterButtonState() {
      const filterButton = document.querySelector('.filter-button');
      const activeFilters = document.querySelectorAll('.filter-checkbox input:checked');
      
      if (activeFilters.length > 0) {
        filterButton.classList.add('has-active-filters');
        filterButton.style.position = 'relative';
      } else {
        filterButton.classList.remove('has-active-filters');
      }
    }
    
    // Initialize export form on page load
    updateExportForm();
    updateFilterButtonState();
    
    // Listen for theme changes
    document.addEventListener('themechange', function(event) {
      console.log('Theme changed in putni nalog list page:', event.detail.theme);
      // Update vehicle badge colors when theme changes
      updateVehicleBadgeColors();
    });
  });
//# sourceMappingURL=listtravelorders.8a899f17.js.map