/**
 * DataTables Filter Service - REFACTORED VERSION
 * Minimalne optimizacije - service je bio odličan
 * Poboljšana integracija s ostalim servisima
 */

export class FilterService {
  constructor(options = {}) {
    this.security = options.security || null;
    this.i18n = options.i18n || null;
    this.columnMappings = null;
    this.debug = options.debug || false;
    this.filterPopups = new Map();
    this.eventListeners = new Set();
    
    this.activeFilters = new Map();
    this.originalUniqueValues = new Map();
    this.refreshInProgress = false;
    
    this.useServerSideRefresh = !!options.useServerSideRefresh;
    
    if (this.debug) {
      console.log(`FilterService initialized - refresh mode: ${this.useServerSideRefresh}`);
    }
    
    this.setupFilters = this.setupFilters.bind(this);
    this.refreshUniqueValues = this.refreshUniqueValues.bind(this);
    this.handleFilterIconClick = this.handleFilterIconClick.bind(this);
    this.applyDropdownFilter = this.applyDropdownFilter.bind(this);
    this.clearAllFilters = this.clearAllFilters.bind(this);
    this.updateFilterInfoBar = this.updateFilterInfoBar.bind(this);
  }

  // ==========================================
  // CONFIGURATION & SETUP METHODS (SHARED)
  // ==========================================
  
  setColumnMappings(mappings) {
    this.columnMappings = mappings;
  }
  
  setRefreshMode(mode) {
    this.useServerSideRefresh = mode;
    if (this.debug) {
      console.log(`FilterService switched to refresh mode: ${mode}`);
    }
  }

  /**
   * Main setup method - works for both client and server side
   */
  setupFilters(api, searchColumns, dropdownColumns, mappingArray) {
    // Store API reference for later use
    this.currentApi = api;
    
    this.clearFilterEvents();
    
    // Remove existing filter icons
    $(api.table().header()).find('.dt-filter__icon').remove();
    
    // Setup filter info bar
    this.setupFilterInfoBar(api);
    
    let $filterRow = $(api.table().header()).find('tr.filters');
    if ($filterRow.length === 0) {
      $filterRow = $('<tr class="filters"></tr>');
      $(api.table().header()).append($filterRow);
      
      // Add placeholder cells for all columns
      api.columns().every(function() {
        $filterRow.append('<th></th>');
      });
    }
    
    if (this.debug) {
      console.log(`Setting up filters - checking visibility for columns`);
    }
    
    // Setup filters only for visible columns
    api.columns().every((colIdx) => {
      const column = api.column(colIdx);
      const filterCell = $filterRow.find('th').eq(colIdx);
      
      if (!column.visible()) {
        if (this.debug) {
          console.log(`Hiding filter for hidden column ${colIdx}`);
        }
        filterCell.hide().empty();
        return;
      }
      
      const columnInfo = this.getColumnInfoByIndex(mappingArray, colIdx);
      if (!columnInfo) {
        filterCell.show().empty();
        return;
      }
      
      const columnName = columnInfo.sqlcolumnname;
      const needsSearchBox = searchColumns.includes(colIdx);
      const isDropdown = dropdownColumns.includes(colIdx);
      
      filterCell.show();
      
      if (!needsSearchBox && !isDropdown) {
        filterCell.empty();
        return;
      }
      
      if (this.debug) {
        console.log(`Setting up filters for visible column ${colIdx} (${columnName})`);
      }
      
      let $filterContainer = filterCell;
      if (needsSearchBox && isDropdown) {
        $filterContainer = $('<div class="dt-filter-container"></div>');
        $filterContainer.css({
          'display': 'flex',
          'align-items': 'center',
          'width': '100%',
          'gap': '4px'
        });
        filterCell.empty().append($filterContainer);
      } else {
        filterCell.empty();
      }
      
      if (needsSearchBox) {
        this.setupTextFilter(column, $filterContainer, filterCell, api);
      }
      
      if (isDropdown) {
        this.setupDropdownFilter(api, column, colIdx, columnName, $filterContainer);
      }
    });
    
    // Event listener for closing popups
    $(document).on('click.filterPopup', e => {
      if (!$(e.target).closest('.dt-filter__popup, .dt-filter__icon').length) {
        $('.dt-filter__popup').hide();
      }
    });
    this.eventListeners.add('click.filterPopup');
    
    this.restoreFilterIcons(api);
    this.storeOriginalUniqueValues(api);
    this.enhanceResetButton(api);
    
    // Check for active filters after setup
    setTimeout(() => {
      let hasActiveFilters = false;
      api.columns().every(function() {
        if (this.search()) {
          hasActiveFilters = true;
          return false;
        }
      });
      
      if (hasActiveFilters) {
        if (this.debug) {
          console.log('Active filters detected after setup, refreshing unique values');
        }
        this.refreshUniqueValues(api);
        this.updateFilterInfoBar(api);
      }
    }, 200);
  }

  /**
   * Setup filter info bar to show active filters
   */
  setupFilterInfoBar(api) {
    // Wait for DataTables to fully initialize
    setTimeout(() => {
      if (this.debug) {
        console.log('=== SETTING UP FILTER INFO BAR (DELAYED) ===');
      }
      
      const $wrapper = $(api.table().container()).find('.dataTables_wrapper');
      
      if ($wrapper.length === 0) {
        const $container = $(api.table().container());
        
        // Remove existing info bar
        $container.find('.dt-filters-info').remove();
        
        // Create filter info bar
        const $filterInfo = $(`
          <div class="dt-filters-info">
            <span class="dt-filters-info__header">
              <i class="fas fa-filter"></i> Aktivni filteri:
            </span>
            <div class="dt-filters-info__tags"></div>
            <button class="dt-filters-clear-all" title="Ukloni sve filtere">
              <i class="fas fa-times"></i> Ukloni sve
            </button>
          </div>
        `);
        
        // Insert before table
        $container.prepend($filterInfo);
        if (this.debug) {
          console.log('Filter info added to container');
        }
        
      } else {
        $wrapper.find('.dt-filters-info').remove();
        
        const $filterInfo = $(`
          <div class="dt-filters-info">
            <span class="dt-filters-info__header">
              <i class="fas fa-filter"></i> Aktivni filteri:
            </span>
            <div class="dt-filters-info__tags"></div>
            <button class="dt-filters-clear-all" title="Ukloni sve filtere">
              <i class="fas fa-times"></i> Ukloni sve
            </button>
          </div>
        `);
        
        const $filterControl = $wrapper.find('.dataTables_filter');
        if ($filterControl.length) {
          $filterControl.parent().after($filterInfo);
        } else {
          $wrapper.prepend($filterInfo);
        }
      }
      
      // Add event listener for clear all button
      $('.dt-filters-clear-all').off('click').on('click', () => {
        this.clearAllFilters(api);
      });
      
      // Update info bar initially
      this.updateFilterInfoBar(api);
      
    }, 500);
  }

  /**
   * Update filter info bar with current active filters
   */
  updateFilterInfoBar(api) {
    const $filterInfo = $(api.table().container()).find('.dt-filters-info');
    const $tagsContainer = $filterInfo.find('.dt-filters-info__tags');
    
    if (!$filterInfo.length) return;
    
    $tagsContainer.empty();
    
    let hasActiveFilters = false;
    const activeTags = [];
    
    // Check each column for active filters
    api.columns().every((colIdx) => {
      const column = api.column(colIdx);
      const search = column.search();
      
      if (search) {
        const columnInfo = this.getColumnInfoByIndex(this.columnMappings, colIdx);
        const columnTitle = columnInfo ? columnInfo.headername : `Column ${colIdx}`;
        
        hasActiveFilters = true;
        
        // Parse filter value for display
        let displayValue = search;
        
        // Handle dropdown filters (regex format)
        if (search.startsWith('(') && search.endsWith(')')) {
          const values = search.slice(1, -1).split('|');
          if (values.length > 3) {
            displayValue = `${values.slice(0, 2).join(', ')} +${values.length - 2} više`;
          } else {
            displayValue = values.join(', ');
          }
        }
        
        // Truncate long values
        if (displayValue.length > 30) {
          displayValue = displayValue.substring(0, 27) + '...';
        }
        
        const $tag = $(`
          <div class="dt-filter-tag" data-column="${colIdx}">
            <span class="dt-filter-tag__column">${this.escapeHtml(columnTitle)}:</span>
            <span class="dt-filter-tag__value">${this.escapeHtml(displayValue)}</span>
            <button class="dt-filter-tag__remove" title="Ukloni filter">×</button>
          </div>
        `);
        
        // Add click handler for remove button
        $tag.find('.dt-filter-tag__remove').on('click', () => {
          column.search('').draw();
          
          // Update filter icon
          const $filterIcon = $(api.table().header()).find(`[data-column-idx="${colIdx}"] .dt-filter__icon`);
          $filterIcon.removeClass('filtered');
          
          // Update filter popup if exists
          const popupInfo = this.filterPopups.get(colIdx);
          if (popupInfo && popupInfo.optionsContainer) {
            popupInfo.optionsContainer.find('.dt-filter__checkbox').prop('checked', false);
            if (popupInfo.selectAll) {
              popupInfo.selectAll.prop('checked', false);
            }
          }
          
          // Update filter info bar
          this.updateFilterInfoBar(api);
          
          // Refresh unique values after filter removal
          setTimeout(() => {
            this.refreshUniqueValues(api);
          }, 100);
        });
        
        activeTags.push($tag);
      }
    });
    
    // Add tags to container
    activeTags.forEach($tag => {
      $tagsContainer.append($tag);
    });
    
    // Show/hide filter info bar
    if (hasActiveFilters) {
      $filterInfo.addClass('active');
      this.updateFilterCount(activeTags.length);
    } else {
      $filterInfo.removeClass('active');
      this.updateFilterCount(0);
    }
  }

  /**
   * Update filter count badge (optional)
   */
  updateFilterCount(count) {
    // Remove existing count
    $('.dt-filter-count').remove();
    
    if (count > 0) {
      // Add count badge to card title or wherever appropriate
      const $title = $('.card-title');
      if ($title.length) {
        $title.append(`<span class="dt-filter-count">${count}</span>`);
      }
    }
  }

  setupTextFilter(column, $filterContainer, filterCell, api) {
    const $searchBox = $('<input type="text" class="dt-column-search" placeholder="Search...">');
    
    $searchBox.css({
      'width': '100%'
    });
    
    $filterContainer.append($searchBox);
    
    let searchTimeout = null;
    
    $searchBox.on('keyup', (event) => {
      if (searchTimeout) {
        clearTimeout(searchTimeout);
      }
      
      searchTimeout = setTimeout(() => {
        const searchValue = $(event.target).val();
        column.search(searchValue).draw();
        
        if (searchValue) {
          filterCell.addClass('filtered');
        } else {
          filterCell.removeClass('filtered');
        }
        
        setTimeout(() => {
          this.refreshUniqueValues(api);
        }, 100);
        
        // Update filter info bar
        setTimeout(() => {
          this.updateFilterInfoBar(api);
        }, 150);
      }, 300);
    });
    
    const currentSearch = column.search();
    if (currentSearch) {
      $searchBox.val(currentSearch);
      filterCell.addClass('filtered');
    }
  }

  setupDropdownFilter(api, column, colIdx, columnName, $filterContainer) {
    const $filterIcon = $('<span class="dt-filter__icon"><i class="fas fa-filter"></i></span>');
    $filterIcon.attr('data-column-idx', colIdx);
    
    $filterContainer.append($filterIcon);
    
    const $filterPopup = $('<div class="dt-filter__popup"></div>');
    $filterPopup.attr('data-column-idx', colIdx);
    
    const searchPlaceholder = this.i18n ? this.i18n.translate('search') : 'Search...';
    const $searchInput = $('<input type="text" class="dt-filter__search">').attr('placeholder', searchPlaceholder);
    $filterPopup.append($searchInput);
    
    let $optionsContainer = null;
    let $selectAllCheckbox = null;
    
    let uniqueValues = this.getUniqueValuesForColumn(api, colIdx, columnName);
    
    if (uniqueValues.length) {
      const selectAllLabel = this.i18n ? this.i18n.translate('selectAll') : 'Select All';
      const $selectAllContainer = $('<div class="dt-filter__option"></div>');
      $selectAllCheckbox = $('<input type="checkbox" class="dt-filter__select-all">');
      const $selectAllLabel = $('<label></label>').text(` (${selectAllLabel})`);
      $selectAllContainer.append($selectAllCheckbox).append($selectAllLabel);
      $filterPopup.append($selectAllContainer);
      
      $optionsContainer = $('<div class="dt-filter__options"></div>');
      $filterPopup.append($optionsContainer);
      
      this.populateDropdownOptions($optionsContainer, uniqueValues, $selectAllCheckbox);
      
      $searchInput.on('keyup', function() {
        const searchValue = $(this).val().toLowerCase();
        $optionsContainer.find('.dt-filter__option').each(function() {
          const optionValue = $(this).text().toLowerCase();
          $(this).toggle(optionValue.indexOf(searchValue) > -1);
        });
      });
    } else {
      $filterPopup.append('<div class="dt-filter__empty">No values available for filtering</div>');
    }
    
    const applyLabel = this.i18n ? this.i18n.translate('apply') : 'Apply';
    const $applyButton = $('<button class="dt-filter__apply"></button>').text(applyLabel);
    $filterPopup.append($applyButton);
    
    const resetLabel = this.i18n ? this.i18n.translate('reset') : 'Reset';
    const $resetButton = $('<button class="dt-filter__reset"></button>').text(resetLabel);
    $filterPopup.append($resetButton);
    
    this.filterPopups.set(colIdx, {
      popup: $filterPopup,
      isDropdown: true,
      selectAll: $selectAllCheckbox,
      optionsContainer: $optionsContainer
    });
    
    $('body').append($filterPopup);
    
    $filterIcon.on('click', e => this.handleFilterIconClick(e, api));
    
    $applyButton.on('click', async () => {
      await this.applyDropdownFilter(api, column, $optionsContainer, $filterIcon);
      $filterPopup.hide();
    });
    
    $resetButton.on('click', () => {
      $optionsContainer.find('.dt-filter__checkbox').prop('checked', false);
      if ($selectAllCheckbox) {
        $selectAllCheckbox.prop('checked', false);
      }
      column.search('').draw();
      $filterIcon.removeClass('filtered');
      $filterPopup.hide();
      
      setTimeout(() => {
        this.refreshUniqueValues(api);
      }, 100);
      
      // Update filter info bar
      setTimeout(() => {
        this.updateFilterInfoBar(api);
      }, 150);
    });
  }

  // ==========================================
  // MAIN REFRESH LOGIC (STRATEGY SELECTOR)
  // ==========================================
  
  /**
   * Main refresh method - decides between client-side and server-side
   */
  async refreshUniqueValues(api) {
    if (this.refreshInProgress) {
      return;
    }
    
    this.refreshInProgress = true;
    
    try {
      if (this.useServerSideRefresh) {
        await this.refreshUniqueValuesServerSide(api);
      } else {
        this.refreshUniqueValuesClientSide(api);
      }
    } catch (error) {
      console.error('Error refreshing unique values:', error);
      
      if (this.useServerSideRefresh) {
        if (this.debug) {
          console.log('Server-side refresh failed, falling back to client-side');
        }
        this.refreshUniqueValuesClientSide(api);
      }
    } finally {
      this.refreshInProgress = false;
    }
  }

  // ==========================================
  // SERVER-SIDE METHODS
  // ==========================================
  
  /**
   * SERVER-SIDE: Refresh unique values via AJAX
   */
  async refreshUniqueValuesServerSide(api) {
    if (this.debug) {
      console.log('=== SERVER-SIDE REFRESH START ===');
    }
    
    // Prepare current filter data
    const currentFilters = {};
    const appliedFiltersDebug = [];
    
    api.columns().every(function(index) {
      const search = this.search();
      if (search) {
        currentFilters[index] = search;
        
        const columnHeader = $(this.header()).text();
        appliedFiltersDebug.push({
          index: index,
          column: columnHeader,
          filter: search,
          searchType: search.includes('(') && search.includes(')') ? 'dropdown' : 'text'
        });
      }
    });
    
    if (this.debug) {
      console.log('Current applied filters:', appliedFiltersDebug);
    }
    
    // Add column type information
    const columnsWithTypes = api.settings()[0].aoColumns.map((col, index) => {
      const columnMapping = this.getColumnInfoByIndex(this.columnMappings, index);
      return {
        data: index.toString(),
        search: {
          value: currentFilters[index] || ''
        },
        columnType: columnMapping?.type || 'text',
        columnName: columnMapping?.sqlcolumnname || `col_${index}`,
        isDate: columnMapping?.type === 'date' || columnMapping?.type === 'datetime'
      };
    });
    
    if (this.debug) {
      console.log('Sending column info to server:', columnsWithTypes);
    }
    
    try {
      const response = await $.ajax({
        url: api.ajax.url(),
        type: 'POST',
        data: {
          refreshFilters: 'true',
          columns: columnsWithTypes,
          filterOrder: appliedFiltersDebug.map(f => f.index),
          debug: this.debug
        }
      });
      
      if (response.success && response.uniqueValues) {
        if (this.debug) {
          console.log('Server response - unique values:', response.uniqueValues);
          
          Object.keys(response.uniqueValues).forEach(columnName => {
            const values = response.uniqueValues[columnName];
            if (columnName.includes('date') || columnName.includes('Date')) {
              console.log(`Date column ${columnName} values:`, values.slice(0, 5));
            }
          });
        }
        
        this.updateAllDropdownOptions(response.uniqueValues);
        
        if (this.debug) {
          console.log('=== SERVER-SIDE REFRESH SUCCESS ===');
        }
      } else {
        throw new Error('Server-side refresh failed: ' + (response.error || 'Unknown error'));
      }
    } catch (error) {
      if (this.debug) {
        console.error('=== SERVER-SIDE REFRESH ERROR ===', error);
      }
      throw error;
    }
  }

  // ==========================================
  // CLIENT-SIDE METHODS  
  // ==========================================
  
  /**
   * CLIENT-SIDE: Refresh unique values from local data
   */
  refreshUniqueValuesClientSide(api) {
    if (this.debug) {
      console.log('=== CLIENT-SIDE REFRESH START ===');
    }
    
    const visibleData = api.rows({ search: 'applied' }).data().toArray();
    const newUniqueValues = {};
    
    if (this.debug) {
      console.log(`Processing ${visibleData.length} visible rows`);
    }
    
    this.filterPopups.forEach((popupInfo, colIdx) => {
      if (popupInfo.isDropdown) {
        const columnInfo = this.getColumnInfoByIndex(this.columnMappings, colIdx);
        if (columnInfo) {
          const columnName = columnInfo.sqlcolumnname;
          const columnType = columnInfo.type;
          
          const uniqueSet = new Set();
          visibleData.forEach(row => {
            const value = row[colIdx];
            if (value !== null && value !== undefined && value !== '') {
              // Special handling for dates
              if (columnType === 'date' || columnType === 'datetime') {
                const normalizedDate = this.normalizeDateValue(value.toString());
                if (normalizedDate) {
                  uniqueSet.add(normalizedDate);
                }
              } else {
                uniqueSet.add(value.toString());
              }
            }
          });
          
          newUniqueValues[columnName] = Array.from(uniqueSet);
          
          if (this.debug) {
            console.log(`Client-side column ${columnName} (${columnType}): ${newUniqueValues[columnName].length} unique values`);
            if (columnType === 'date' || columnType === 'datetime') {
              console.log(`Sample date values:`, newUniqueValues[columnName].slice(0, 3));
            }
          }
        }
      }
    });
    
    this.updateAllDropdownOptions(newUniqueValues);
    
    if (this.debug) {
      console.log('=== CLIENT-SIDE REFRESH SUCCESS ===');
    }
  }
  
  /**
   * CLIENT-SIDE: Normalize date values to standard format
   */
  normalizeDateValue(value) {
    if (!value) return null;
    
    // Try to parse different date formats
    const dateFormats = [
      /^(\d{4})-(\d{2})-(\d{2})/, // YYYY-MM-DD
      /^(\d{2})-(\d{2})-(\d{4})/, // DD-MM-YYYY
      /^(\d{2})\.(\d{2})\.(\d{4})/, // DD.MM.YYYY
      /^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2}):(\d{2})/ // YYYY-MM-DD HH:MM:SS
    ];
    
    for (const format of dateFormats) {
      const match = value.match(format);
      if (match) {
        // Return in standard YYYY-MM-DD format
        if (format.source.includes('(\\d{4})-(\\d{2})-(\\d{2})')) {
          // Already in YYYY-MM-DD format
          return value.substring(0, 10);
        } else if (format.source.includes('(\\d{2})-(\\d{2})-(\\d{4})')) {
          // DD-MM-YYYY -> YYYY-MM-DD
          return `${match[3]}-${match[2]}-${match[1]}`;
        }
      }
    }
    
    return value; // Return original if can't parse
  }

  // ==========================================
  // SHARED UTILITY METHODS
  // ==========================================
  
  updateAllDropdownOptions(newUniqueValues) {
    this.filterPopups.forEach((popupInfo, colIdx) => {
      if (popupInfo.isDropdown && popupInfo.optionsContainer) {
        const column = this.getColumnInfoByIndex(this.columnMappings, colIdx);
        if (column && newUniqueValues[column.sqlcolumnname]) {
          this.updateDropdownOptions(
            popupInfo, 
            newUniqueValues[column.sqlcolumnname]
          );
        }
      }
    });
  }
  
  updateDropdownOptions(popupInfo, newValues) {
    const $optionsContainer = popupInfo.optionsContainer;
    const $selectAll = popupInfo.selectAll;
    
    const selectedValues = [];
    $optionsContainer.find('.dt-filter__checkbox:checked').each(function() {
      selectedValues.push($(this).attr('data-value'));
    });
    
    $optionsContainer.empty();
    
    newValues.sort().forEach(value => {
      if (value === null || value === undefined || value === '') return;
      
      const safeValue = this.sanitizeValue(value);
      const $option = $('<div class="dt-filter__option"></div>');
      const $checkbox = $('<input type="checkbox" class="dt-filter__checkbox">');
      $checkbox.attr('data-value', safeValue);
      
      const wasSelected = selectedValues.includes(safeValue);
      $checkbox.prop('checked', wasSelected);
      
      const $label = $('<label></label>').text(' ' + safeValue);
      $option.append($checkbox).append($label);
      $optionsContainer.append($option);
    });
    
    $optionsContainer.off('change', '.dt-filter__checkbox');
    $optionsContainer.on('change', '.dt-filter__checkbox', () => {
      const allChecked = $optionsContainer.find('.dt-filter__checkbox:checked').length === 
                $optionsContainer.find('.dt-filter__checkbox').length;
      if ($selectAll) {
        $selectAll.prop('checked', allChecked);
      }
    });
    
    if ($selectAll) {
      const totalCheckboxes = $optionsContainer.find('.dt-filter__checkbox').length;
      const checkedCheckboxes = $optionsContainer.find('.dt-filter__checkbox:checked').length;
      $selectAll.prop('checked', checkedCheckboxes > 0 && checkedCheckboxes === totalCheckboxes);
    }
  }
  
  storeOriginalUniqueValues(api) {
    this.filterPopups.forEach((popupInfo, colIdx) => {
      if (popupInfo.isDropdown) {
        const columnInfo = this.getColumnInfoByIndex(this.columnMappings, colIdx);
        if (columnInfo) {
          const columnName = columnInfo.sqlcolumnname;
          const uniqueValues = this.getUniqueValuesForColumn(api, colIdx, columnName);
          this.originalUniqueValues.set(columnName, uniqueValues);
        }
      }
    });
  }
  
  async applyDropdownFilter(api, column, $optionsContainer, $filterIcon) {
    const selectedValues = [];
    $optionsContainer.find('.dt-filter__checkbox:checked').each(function() {
      selectedValues.push($(this).attr('data-value'));
    });
    
    const colIdx = parseInt($filterIcon.attr('data-column-idx'));
    const columnInfo = this.getColumnInfoByIndex(this.columnMappings, colIdx);
    
    if (this.debug) {
      console.log(`=== APPLYING FILTER ===`);
      console.log(`Column: ${columnInfo?.sqlcolumnname} (index: ${colIdx})`);
      console.log(`Type: ${columnInfo?.type}`);
      console.log(`Selected values:`, selectedValues);
    }
    
    // Apply filter
    if (selectedValues.length > 0) {
      const regex = selectedValues.join('|');
      column.search(`(${regex})`, true, false).draw();
      $filterIcon.addClass('filtered');
      
      if (this.debug) {
        console.log(`Applied regex: (${regex})`);
      }
    } else {
      column.search('').draw();
      $filterIcon.removeClass('filtered');
      
      if (this.debug) {
        console.log(`Cleared filter for column ${colIdx}`);
      }
    }
    
    // Update active filters map
    if (selectedValues.length > 0) {
      this.activeFilters.set(colIdx, selectedValues);
    } else {
      this.activeFilters.delete(colIdx);
    }
    
    // Update filter info bar
    setTimeout(() => {
      this.updateFilterInfoBar(api);
    }, 100);
    
    // Refresh unique values
    const isServerSide = this.useServerSideRefresh === true || 
                        (this.useServerSideRefresh === 'auto' && api.settings()[0].oFeatures.bServerSide);
    
    const delay = isServerSide ? 500 : 100;
    
    if (this.debug) {
      console.log(`Will refresh unique values in ${delay}ms using ${isServerSide ? 'server-side' : 'client-side'} method`);
    }
    
    setTimeout(() => {
      this.refreshUniqueValues(api);
    }, delay);
  }
  
  clearAllFilters(api) {
    api.columns().every(function() {
      this.search('');
    });
    
    $('.dt-filter__icon').removeClass('filtered');
    this.activeFilters.clear();
    
    this.filterPopups.forEach(popupInfo => {
      if (popupInfo.optionsContainer) {
        popupInfo.optionsContainer.find('.dt-filter__checkbox').prop('checked', false);
      }
      if (popupInfo.selectAll) {
        popupInfo.selectAll.prop('checked', false);
      }
    });
    
    api.draw();
    
    // Update filter info bar
    this.updateFilterInfoBar(api);
    
    const delay = (this.useServerSideRefresh === true || 
                  (this.useServerSideRefresh === 'auto' && api.settings()[0].oFeatures.bServerSide)) 
                  ? 500 : 100;
    setTimeout(() => {
      if (this.useServerSideRefresh === false || 
          (this.useServerSideRefresh === 'auto' && !api.settings()[0].oFeatures.bServerSide)) {
        const originalValues = {};
        this.originalUniqueValues.forEach((values, columnName) => {
          originalValues[columnName] = values;
        });
        this.updateAllDropdownOptions(originalValues);
      } else {
        this.refreshUniqueValues(api);
      }
    }, delay);
  }

  enhanceResetButton(api) {
    $('.dt-button--reset').off('click');
    $('.dt-button--reset').on('click', (e) => {
      e.preventDefault();
      this.clearAllFilters(api);
    });
  }

  populateDropdownOptions($optionsContainer, values, $selectAllCheckbox) {
    $optionsContainer.empty();
    
    values.sort().forEach(value => {
      if (value === null || value === undefined || value === '') return;
      
      const safeValue = this.sanitizeValue(value);
      const $option = $('<div class="dt-filter__option"></div>');
      const $checkbox = $('<input type="checkbox" class="dt-filter__checkbox">');
      $checkbox.attr('data-value', safeValue);
      const $label = $('<label></label>').text(' ' + safeValue);
      
      $option.append($checkbox).append($label);
      $optionsContainer.append($option);
    });
    
    if ($selectAllCheckbox) {
      $selectAllCheckbox.on('change', e => {
        const isChecked = $(e.target).prop('checked');
        $optionsContainer.find('.dt-filter__checkbox').prop('checked', isChecked);
      });
    }
    
    $optionsContainer.on('change', '.dt-filter__checkbox', () => {
      const allChecked = $optionsContainer.find('.dt-filter__checkbox:checked').length === 
                $optionsContainer.find('.dt-filter__checkbox').length;
      if ($selectAllCheckbox) {
        $selectAllCheckbox.prop('checked', allChecked);
      }
    });
  }

  handleFilterIconClick(e, api) {
    e.stopPropagation();
    
    $('.dt-filter__popup').hide();
    
    const $icon = $(e.currentTarget);
    const colIdx = $icon.attr('data-column-idx');
    
    if (!colIdx) {
      console.error('Missing column index on filter icon');
      return;
    }
    
    const popupInfo = this.filterPopups.get(parseInt(colIdx));
    if (!popupInfo) {
      console.error(`Popup information not found for column ${colIdx}`);
      return;
    }
    
    const $popup = popupInfo.popup;
    const column = api.column(colIdx);
    
    const iconPos = $icon.offset();
    $popup.css({
      'top': iconPos.top + $icon.outerHeight() + 5,
      'left': iconPos.left - ($popup.width() / 2) + $icon.width() / 2
    }).show();
    
    $popup.find('.dt-filter__search').focus();
    
    this.restoreFilterState(column, popupInfo);
  }

  getUniqueValuesForColumn(api, colIdx, columnName) {
    let uniqueValues = [];
    
    const response = api.ajax.json();
    if (response && response.uniqueValues && response.uniqueValues[columnName]) {
      uniqueValues = response.uniqueValues[columnName];
      if (this.debug) console.log(`Retrieved unique values from server for ${columnName}:`, uniqueValues.length);
      return uniqueValues;
    }
    
    if (api.settings()[0].json && api.settings()[0].json.uniqueValues && 
        api.settings()[0].json.uniqueValues[columnName]) {
      uniqueValues = api.settings()[0].json.uniqueValues[columnName];
      if (this.debug) console.log(`Retrieved unique values from settings for ${columnName}:`, uniqueValues.length);
      return uniqueValues;
    }
    
    const uniqueSet = new Set();
    
    api.column(colIdx).data().each(function(value) {
      if (value !== null && value !== undefined && value !== '') {
        uniqueSet.add(value.toString());
      }
    });
    
    uniqueValues = Array.from(uniqueSet);
    if (this.debug) console.log(`Extracted unique values from data for ${columnName}:`, uniqueValues.length);
    
    return uniqueValues;
  }

  getColumnInfoByIndex(mappingArray, index) {
    return mappingArray.find(item => item.index === parseInt(index));
  }

  sanitizeValue(value) {
    if (this.security && typeof this.security.sanitize === 'function') {
      return this.security.sanitize(value);
    }
    
    if (value === null || value === undefined) {
      return '';
    }
    
    return String(value)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  escapeHtml(html) {
    // Use SecurityService if available
    if (this.security && typeof this.security.sanitize === 'function') {
      return this.security.sanitize(html);
    }
    
    // Fallback to sanitizeValue
    return this.sanitizeValue(html);
  }

  restoreFilterState(column, popupInfo) {
    const currentSearch = column.search();
    
    if (popupInfo.isDropdown && popupInfo.optionsContainer) {
      if (currentSearch) {
        try {
          const cleanedSearch = currentSearch.replace(/^\(/, '').replace(/\)$/, '');
          const values = cleanedSearch.split('|');
          
          popupInfo.optionsContainer.find('.dt-filter__checkbox').prop('checked', false);
          
          popupInfo.optionsContainer.find('.dt-filter__checkbox').each(function() {
            const checkValue = $(this).attr('data-value');
            
            for (const value of values) {
              const escapedValue = value.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
              const regex = new RegExp(`^${escapedValue}$`);
              
              if (regex.test(checkValue)) {
                $(this).prop('checked', true);
                break;
              }
            }
          });
          
          if (popupInfo.selectAll) {
            const totalCheckboxes = popupInfo.optionsContainer.find('.dt-filter__checkbox').length;
            const checkedCheckboxes = popupInfo.optionsContainer.find('.dt-filter__checkbox:checked').length;
            
            popupInfo.selectAll.prop('checked', checkedCheckboxes > 0 && checkedCheckboxes === totalCheckboxes);
          }
          
          // Refresh unique values if filter is active
          const checkedCheckboxes = popupInfo.optionsContainer.find('.dt-filter__checkbox:checked').length;
          if (checkedCheckboxes > 0 && this.refreshUniqueValues) {
            setTimeout(() => {
              this.refreshUniqueValues(this.currentApi);
            }, 100);
          }
          
        } catch (error) {
          console.error('Error restoring filter state:', error);
        }
      } else {
        popupInfo.optionsContainer.find('.dt-filter__checkbox').prop('checked', false);
        if (popupInfo.selectAll) {
          popupInfo.selectAll.prop('checked', false);
        }
      }
    }
  }

  restoreFilterIcons(table) {
    if (!table || typeof table.columns !== 'function') {
      return;
    }

    try {
      table.columns().every(function() {
        const column = this;
        
        if (!column || typeof column.header !== 'function') {
          return;
        }
        
        const header = column.header();
        if (!header) return;
        
        const $headerCell = $(header);
        const $filterIcon = $headerCell.find('.dt-filter__icon');
        
        if (column.search() && $filterIcon.length) {
          $filterIcon.addClass('filtered');
        }
      });
    } catch (error) {
      console.warn('Error in restoreFilterIcons:', error);
    }
  }

  clearFilterEvents() {
    this.eventListeners.forEach(eventName => {
      $(document).off(eventName);
    });
    
    this.filterPopups.forEach(info => {
      if (info.popup) {
        info.popup.remove();
      }
    });
    
    this.eventListeners.clear();
    this.filterPopups.clear();
  }

  // ==========================================
  // CLEANUP & LIFECYCLE METHODS
  // ==========================================
  
  destroy() {
    this.clearFilterEvents();
    
    // Clear all intervals/timeouts
    this.activeFilters.clear();
    this.originalUniqueValues.clear();
    
    // Reset state
    this.refreshInProgress = false;
    this.columnMappings = null;
  }
}

// Zatvaranje klase FilterService
//# sourceMappingURL=filter-service.js.map