/**
 * DataTables Format Service - POBOLJŠANA VERZIJA s hrvatskim formatiranjem
 * Dodano pravilno formatiranje za hrvatske standarde
 */

export class FormatService {
  /**
   * Creates a new FormatService instance
   * @param {string} locale - Locale code (e.g. 'hr', 'en')
   */
  constructor(locale = 'hr') {
    this.locale = locale;
    this.defaultCurrency = '€';
    
    // Initialize locale settings
    this.initializeLocale();
  }
  
  /**
   * Initialize locale settings
   */
  initializeLocale() {
    // Set locale settings for numbers and currency
    this.numberFormats = {
      'hr': {
        decimalSeparator: ',',
        thousandsSeparator: '.',
        currencySymbol: this.defaultCurrency,
        currencyPosition: 'suffix' // 'prefix' or 'suffix'
      },
      'en': {
        decimalSeparator: '.',
        thousandsSeparator: ',',
        currencySymbol: '$',
        currencyPosition: 'prefix' // 'prefix' or 'suffix'
      }
    };
  }
  
  /**
   * Change locale
   * @param {string} locale - New locale code
   */
  setLocale(locale) {
    if (this.numberFormats[locale]) {
      this.locale = locale;
    } else {
      console.warn(`Locale '${locale}' is not supported. Using default '${this.locale}'.`);
    }
  }
  
  /**
   * Format currency - POBOLJŠANA VERZIJA za hrvatski format
   * @param {number} value - Value to format
   * @param {Object} options - Formatting options
   * @return {string} - Formatted currency value
   */
  formatCurrency(value, options = {}) {
    const floatValue = parseFloat(value);
    if (isNaN(floatValue)) {
      return '';
    }
    
    const settings = this.numberFormats[this.locale] || this.numberFormats['hr'];
    const decimalPlaces = options.decimalPlaces ?? 2;
    const symbol = options.symbol ?? settings.currencySymbol;
    const position = options.position ?? settings.currencyPosition;
    
    // Format number with decimal and thousands separators
    let formattedNumber = this.formatNumber(floatValue, { 
      decimalPlaces: decimalPlaces 
    });
    
    // Add currency symbol with proper spacing for Croatian format
    if (position === 'prefix') {
      return `${symbol}${formattedNumber}`;
    } else {
      return `${formattedNumber} ${symbol}`;
    }
  }
  
  /**
   * Format number - POBOLJŠANA VERZIJA za hrvatski format
   * @param {number} value - Value to format
   * @param {Object} options - Formatting options
   * @return {string} - Formatted number value
   */
  formatNumber(value, options = {}) {
    const floatValue = parseFloat(value);
    if (isNaN(floatValue)) {
      return '';
    }
    
    const settings = this.numberFormats[this.locale] || this.numberFormats['hr'];
    const decimalPlaces = options.decimalPlaces ?? 2;
    
    // Split integer and decimal parts
    const parts = floatValue.toFixed(decimalPlaces).split('.');
    const integerPart = parts[0];
    const decimalPart = parts[1] || '';
    
    // Add thousands separator (Croatian: . for thousands)
    const formattedInteger = integerPart.replace(/\B(?=(\d{3})+(?!\d))/g, settings.thousandsSeparator);
    
    // If no decimal places, return just the integer part
    if (decimalPlaces === 0) {
      return formattedInteger;
    }
    
    // Return integer and decimal part with Croatian decimal separator (,)
    return `${formattedInteger}${settings.decimalSeparator}${decimalPart}`;
  }

  /**
   * Parse currency value from various formats - POBOLJŠANA za hrvatski format
   * @param {string|number} value - Value to parse
   * @return {number} - Parsed numeric value
   */
  parseCurrency(value) {
    if (typeof value === 'number') {
      return isNaN(value) ? 0 : value;
    }

    if (typeof value === 'string') {
      // Handle Croatian format: 1.234,56 € -> 1234.56
      const cleanValue = value
        .replace(/\s*€\s*/, '')           // Remove € symbol
        .replace(/\./g, '')               // Remove thousand separators (Croatian dots)
        .replace(',', '.');               // Replace decimal comma with dot
      
      const numValue = parseFloat(cleanValue);
      return isNaN(numValue) ? 0 : numValue;
    }

    return 0;
  }

  /**
   * Format currency for DataTables - KLJUČNA FUNKCIJA za proper rendering
   * @param {*} data - Cell data
   * @param {string} type - DataTables render type
   * @return {string|number} - Formatted value
   */
  formatCurrencyForDataTables(data, type) {
    if (type === 'display') {
      // For display, return Croatian formatted string
      return this.formatCurrency(data);
    } else if (type === 'type' || type === 'sort') {
      // For sorting, return numeric value
      return this.parseCurrency(data);
    } else {
      // For filtering and other types, return numeric value
      return this.parseCurrency(data);
    }
  }

  /**
   * Parse date from various formats
   * @param {string|Date} value - Date to parse
   * @return {Date|null} - Parsed date or null
   */
  parseDate(value) {
    if (!value) return null;
    if (value instanceof Date) return value;

    const str = String(value).trim();
    
    // YYYY-MM-DD HH:MM:SS format (with seconds)
    let match = str.match(/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2}):(\d{2})/);
    if (match) {
      return new Date(match[1], match[2] - 1, match[3], match[4], match[5], match[6]);
    }
    
    // YYYY-MM-DD HH:MM format (without seconds)
    match = str.match(/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2})/);
    if (match) {
      return new Date(match[1], match[2] - 1, match[3], match[4], match[5]);
    }
    
    // YYYY-MM-DD format (date only)
    match = str.match(/^(\d{4})-(\d{2})-(\d{2})/);
    if (match) {
      return new Date(match[1], match[2] - 1, match[3]);
    }
    
    // DD-MM-YYYY format
    match = str.match(/^(\d{2})-(\d{2})-(\d{4})/);
    if (match) {
      return new Date(match[3], match[2] - 1, match[1]);
    }
    
    // DD.MM.YYYY format
    match = str.match(/^(\d{2})\.(\d{2})\.(\d{4})/);
    if (match) {
      return new Date(match[3], match[2] - 1, match[1]);
    }

    // Fallback to Date constructor
    const date = new Date(value);
    return isNaN(date.getTime()) ? null : date;
  }

  /**
   * Format date for saving to database (YYYY-MM-DD)
   * @param {Date} date - Date object
   * @return {string} - SQL formatted date
   */
  formatDateForSave(date) {
    if (!date || !(date instanceof Date)) return '';
    
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    
    return `${year}-${month}-${day}`;
  }
  
  /**
   * Format date - hrvatski format DD-MM-YYYY
   * @param {string|Date} value - Date to format
   * @param {string} format - Date format type
   * @return {string} - Formatted date
   */
  formatDate(value, format = 'shortDate') {
    const date = this.parseDate(value);
    if (!date) return '';

    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    
    return `${day}-${month}-${year}`;
  }
  
  /**
   * Format datetime - hrvatski format DD-MM-YYYY HH:MM
   * @param {string|Date} value - Datetime to format
   * @return {string} - Formatted datetime
   */
  formatDateTime(value) {
    const date = this.parseDate(value);
    if (!date) return '';

    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    const hour = String(date.getHours()).padStart(2, '0');
    const minute = String(date.getMinutes()).padStart(2, '0');
    
    return `${day}-${month}-${year} ${hour}:${minute}`;
  }

  /**
   * NOVA FUNKCIJA: Validate Croatian currency format
   * @param {string} value - Value to validate
   * @return {boolean} - Is valid Croatian currency format
   */
  isValidCroatianCurrency(value) {
    // Croatian currency format: 1.234,56 € or 1.234,56€ or 1234,56 €
    const pattern = /^\d{1,3}(\.\d{3})*,\d{2}\s*€?$/;
    return pattern.test(value.trim());
  }

  /**
   * NOVA FUNKCIJA: Convert any currency format to Croatian
   * @param {string|number} value - Value to convert
   * @return {string} - Croatian formatted currency
   */
  toCroatianCurrency(value) {
    const numericValue = this.parseCurrency(value);
    return this.formatCurrency(numericValue);
  }
}
//# sourceMappingURL=format-service.js.map