/**
 * DataTables Internationalization Service - REFACTORED VERSION
 * Uklonjen duplikatni kod i nekorišćene funkcije
 * Fokus SAMO na DataTables prijevode i osnovne i18n funkcionalnosti
 */

export class I18nService {
  /**
   * Creates a new I18nService instance
   * @param {string} locale - Locale code (e.g. 'hr', 'en')
   * @param {Object} options - Additional options
   */
  constructor(locale = 'hr', options = {}) {
    this.locale = locale;
    this.fallbackLocale = options.fallbackLocale || 'en';
    this.debug = options.debug || false;
    
    // Load translations
    this.translations = this.getTranslations();
    
    // Add additional translations if available
    if (options.translations) {
      this.addTranslations(options.translations);
    }
    
    if (this.debug) {
      console.log(`I18nService initialized with locale: ${locale}`);
    }
  }
  
  /**
   * Get core translations - SAMO potrebni prijevodi
   * @return {Object} - Object with translations
   */
  getTranslations() {
    return {
      'hr': {
        // DataTables translations - GLAVNA FUNKCIONALNOST
        processing: "Učitavanje...",
        loadingRecords: "Molimo pričekajte...",
        lengthMenu: "_MENU_",
        zeroRecords: "Nema pronađenih zapisa",
        info: "_START_-_END_ / _TOTAL_",
        infoEmpty: "0-0 / 0",
        infoFiltered: "(_MAX_)",
        search: "Traži:",
        paginate: {
          previous: "<",
          next: ">"
        },
        
        // Filters - KORISTI SE U FILTER SERVICE
        selectAll: "Odaberi sve",
        apply: "Primijeni",
        reset: "Poništi",
        
        // Actions - KORISTI SE U COLUMN SERVICE
        edit: "Uredi",
        delete: "Obriši",
        save: "Spremi",
        cancel: "Odustani",
        confirmDelete: "Da li ste sigurni da želite obrisati ovaj zapis?",
        
        // Editing - KORISTI SE U EDITOR SERVICE
        editingError: "Greška pri uređivanju",
        saveSuccess: "Uspješno spremljeno"
      },
      'en': {
        // DataTables translations
        processing: "Loading...",
        loadingRecords: "Please wait...",
        lengthMenu: "_MENU_",
        zeroRecords: "No matching records found",
        info: "_START_-_END_ of _TOTAL_",
        infoEmpty: "0-0 of 0",
        infoFiltered: "(filtered from _MAX_ total records)",
        search: "Search:",
        paginate: {
          previous: "<",
          next: ">"
        },
        
        // Filters
        selectAll: "Select All",
        apply: "Apply",
        reset: "Reset",
        
        // Actions
        edit: "Edit",
        delete: "Delete",
        save: "Save",
        cancel: "Cancel",
        confirmDelete: "Are you sure you want to delete this record?",
        
        // Editing
        editingError: "Editing error",
        saveSuccess: "Successfully saved"
      }
    };
  }
  
  /**
   * Add custom translations - POTREBNO za proširenje
   * @param {Object} translations - Object with translations
   */
  addTranslations(translations) {
    if (!translations || typeof translations !== 'object') {
      return;
    }
    
    // Go through all locales
    Object.keys(translations).forEach(locale => {
      // Create locale if it doesn't exist
      if (!this.translations[locale]) {
        this.translations[locale] = {};
      }
      
      // Merge new translations with existing ones
      this.translations[locale] = {
        ...this.translations[locale],
        ...translations[locale]
      };
    });
  }
  
  /**
   * Change current locale
   * @param {string} locale - Locale code
   */
  setLocale(locale) {
    if (this.translations[locale]) {
      this.locale = locale;
    } else {
      console.warn(`Locale '${locale}' is not supported. Using default '${this.locale}'.`);
    }
  }
  
  /**
   * Translate key - GLAVNA FUNKCIJA
   * @param {string} key - Key to translate (can be nested with '.')
   * @param {Object} params - Parameters for interpolation
   * @return {string} - Translated text
   */
  translate(key, params = {}) {
    // Get translation according to key path
    const translation = this.getByPath(this.translations[this.locale], key) || 
                        this.getByPath(this.translations[this.fallbackLocale], key) || 
                        key;
    
    // If no parameters, return translation as is
    if (Object.keys(params).length === 0) {
      return translation;
    }
    
    // Otherwise interpolate parameters
    return Object.keys(params).reduce((text, param) => {
      const regex = new RegExp(`{${param}}`, 'g');
      return text.replace(regex, params[param]);
    }, translation);
  }
  
  /**
   * Shorter method for translate - CONVENIENCE METODA
   * @param {string} key - Key to translate
   * @param {Object} params - Parameters for interpolation
   * @return {string} - Translated text
   */
  t(key, params = {}) {
    return this.translate(key, params);
  }
  
  /**
   * Get value by path from object - UTILITY
   * @param {Object} obj - Object to get value from
   * @param {string} path - Path (e.g. 'paginate.next')
   * @return {*} - Value at path or undefined
   */
  getByPath(obj, path) {
    return path.split('.').reduce((o, i) => o ? o[i] : undefined, obj);
  }
  
  /**
   * Get DataTables language options - GLAVNA FUNKCIJA ZA DATATABLES
   * @return {Object} - Object with language options for DataTables
   */
  getLanguageOptions() {
    const trans = this.translations[this.locale];
    
    return {
      processing: trans.processing,
      loadingRecords: trans.loadingRecords,
      lengthMenu: trans.lengthMenu,
      zeroRecords: trans.zeroRecords,
      info: trans.info,
      infoEmpty: trans.infoEmpty,
      infoFiltered: trans.infoFiltered,
      search: trans.search,
      paginate: trans.paginate
    };
  }
}
//# sourceMappingURL=i18n-service.js.map