/**
 * DataTables Security Service - REFACTORED VERSION
 * Bez promjena - service je već bio odlično napisan
 * Samo dodani komentari za bolju dokumentaciju
 */

export class SecurityService {
  /**
   * Creates a new SecurityService instance
   * @param {Object} options - Configuration options
   */
  constructor(options = {}) {
    this.csrfToken = null;
    this.tokenRefreshInterval = options.tokenRefreshInterval || 0; // ms, 0 = disabled
    this.csrfHeaderName = options.csrfHeaderName || 'X-CSRF-Token';
    this.csrfInputName = options.csrfInputName || 'csrf_token';
    this.debug = options.debug || false;
    
    // Initialize token
    this.refreshCSRFToken();
    
    // Set interval for refreshing if enabled
    if (this.tokenRefreshInterval > 0) {
      this.refreshIntervalId = setInterval(
        () => this.refreshCSRFToken(), 
        this.tokenRefreshInterval
      );
    }
  }
  
  /**
   * Refresh CSRF token from different sources
   * KLJUČNA FUNKCIJA - koristi se u cijelom sistemu
   */
  refreshCSRFToken() {
    // Try several strategies to get the token
    
    // 1. Get from jQuery AJAX settings (most common)
    if (window.jQuery && $.ajaxSettings && $.ajaxSettings.headers) {
      const csrfHeader = $.ajaxSettings.headers[this.csrfHeaderName];
      if (csrfHeader) {
        this.csrfToken = csrfHeader;
        if (this.debug) console.log(`CSRF token retrieved from jQuery AJAX headers: ${this.truncateToken(this.csrfToken)}`);
        return;
      }
    }
    
    // 2. Get from form input (standardized approach from csrf-jquery.js)
    const inputToken = document.querySelector(`input[name="${this.csrfInputName}"]`)?.value;
    if (inputToken) {
      this.csrfToken = inputToken;
      if (this.debug) console.log(`CSRF token retrieved from form input: ${this.truncateToken(this.csrfToken)}`);
      return;
    }
    
    // 3. Get from window.fetchWithCsrf (for compatibility with csrf-fetch.js)
    if (window.fetchWithCsrf) {
      try {
        const csrfToken = document.querySelector('input[name="csrf_token"]')?.value;
        if (csrfToken) {
          this.csrfToken = csrfToken;
          if (this.debug) console.log(`CSRF token retrieved via fetchWithCsrf approach: ${this.truncateToken(this.csrfToken)}`);
          return;
        }
      } catch (e) {
        console.warn('Error getting CSRF token from fetchWithCsrf approach:', e);
      }
    }
    
    // Warn if token not found
    if (!this.csrfToken) {
      console.warn('CSRF token not found using any known method. This might cause security issues.');
    }
  }
  
  /**
   * Truncate token for safe display in console
   * @param {string} token - Token to truncate
   * @returns {string} - Truncated token
   */
  truncateToken(token) {
    if (!token) return '';
    if (token.length <= 8) return '****';
    return token.substring(0, 4) + '...' + token.substring(token.length - 4);
  }
  
  /**
   * Get current CSRF token
   * GLAVNA FUNKCIJA - koristi se u editor i filter servisima
   * @return {string} - CSRF token
   */
  getCSRFToken() {
    // Automatically refresh token if not set
    if (!this.csrfToken) {
      this.refreshCSRFToken();
    }
    
    return this.csrfToken || '';
  }
  
  /**
   * Add CSRF token to AJAX data
   * @param {Object|FormData} data - Data to be sent
   * @return {Object|FormData} - Data with added CSRF token
   */
  addCSRFToData(data) {
    const token = this.getCSRFToken();
    if (!token) return data;
    
    // For FormData
    if (data instanceof FormData) {
      // Check if token already exists
      let hasToken = false;
      for (const pair of data.entries()) {
        if (pair[0] === this.csrfInputName) {
          hasToken = true;
          break;
        }
      }
      
      if (!hasToken) {
        data.append(this.csrfInputName, token);
      }
      
      return data;
    }
    
    // For plain objects
    if (data && typeof data === 'object' && !Array.isArray(data)) {
      if (!data[this.csrfInputName]) {
        data[this.csrfInputName] = token;
      }
    }
    
    return data;
  }
  
  /**
   * Validate and add token to form
   * @param {HTMLFormElement} form - Form element
   */
  validateFormToken(form) {
    if (!form || !(form instanceof HTMLFormElement)) {
      console.error('Invalid form element provided to validateFormToken');
      return;
    }
    
    const token = this.getCSRFToken();
    if (!token) return;
    
    // Check if token already exists in form
    let tokenInput = form.querySelector(`input[name="${this.csrfInputName}"]`);
    
    if (tokenInput) {
      // If exists, update value
      tokenInput.value = token;
    } else {
      // If doesn't exist, add new input
      tokenInput = document.createElement('input');
      tokenInput.type = 'hidden';
      tokenInput.name = this.csrfInputName;
      tokenInput.value = token;
      form.appendChild(tokenInput);
    }
  }
  
  /**
   * Sanitize string to prevent XSS attacks
   * KLJUČNA FUNKCIJA - koristi se u filter servisu
   * @param {string} str - String to sanitize
   * @return {string} - Sanitized string
   */
  sanitize(str) {
    if (str === null || str === undefined) {
      return '';
    }
    
    if (typeof str !== 'string') {
      str = String(str);
    }
    
    // Basic HTML escape
    return str
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }
  
  /**
   * Sanitize HTML attribute
   * @param {string} attr - Attribute value to sanitize
   * @return {string} - Sanitized attribute value
   */
  sanitizeAttribute(attr) {
    return this.sanitize(attr);
  }
  
  /**
   * Sanitize URL
   * @param {string} url - URL to sanitize
   * @return {string} - Sanitized URL
   */
  sanitizeUrl(url) {
    if (!url) {
      return '';
    }
    
    try {
      // Try to parse URL
      const currentUrl = window.location.href;
      const baseUrl = currentUrl.split('/').slice(0, 3).join('/'); // http(s)://domain.com
      
      // If relative URL, use current domain as base
      const fullUrl = url.startsWith('http') ? url : new URL(url, baseUrl).href;
      
      // Check URL
      const parsedUrl = new URL(fullUrl);
      
      // Allow only http and https schemes
      if (parsedUrl.protocol !== 'http:' && parsedUrl.protocol !== 'https:') {
        if (this.debug) console.warn(`Blocked unsafe URL protocol: ${parsedUrl.protocol}`);
        return '';
      }
      
      return parsedUrl.toString();
    } catch (e) {
      // If URL is invalid, return empty string
      console.error('Invalid URL:', url, e);
      return '';
    }
  }
  
  /**
   * Prepare secure AJAX request
   * @param {Object} options - Options for $.ajax
   * @return {Object} - Options with added security enhancements
   */
  prepareSecureAjaxOptions(options) {
    if (!options || typeof options !== 'object') {
      return options;
    }
    
    // Clone options to avoid modifying original object
    const secureOptions = { ...options };
    
    // Add CSRF token in header if using jQuery AJAX
    if (window.jQuery) {
      const token = this.getCSRFToken();
      if (token) {
        secureOptions.headers = secureOptions.headers || {};
        secureOptions.headers[this.csrfHeaderName] = token;
      }
    }
    
    // Add CSRF to data if using POST method
    if ((secureOptions.type || secureOptions.method || '').toUpperCase() === 'POST') {
      secureOptions.data = this.addCSRFToData(secureOptions.data || {});
    }
    
    // Sanitize URL
    if (secureOptions.url) {
      secureOptions.url = this.sanitizeUrl(secureOptions.url);
      if (!secureOptions.url) {
        console.error('Invalid URL in AJAX request');
        return null; // Return null for failed options
      }
    }
    
    return secureOptions;
  }
  
  /**
   * Clean up resources when destroying
   */
  destroy() {
    // Clear refresh interval
    if (this.refreshIntervalId) {
      clearInterval(this.refreshIntervalId);
      this.refreshIntervalId = null;
    }
  }
  
  /**
   * Generate safe ID
   * @param {string} prefix - Prefix for ID
   * @return {string} - Safe unique ID
   */
  generateSafeId(prefix = 'dt') {
    const randomPart = Math.random().toString(36).substring(2, 10);
    const timestamp = Date.now().toString(36);
    return `${prefix}-${randomPart}-${timestamp}`;
  }
}
//# sourceMappingURL=security-service.js.map