# baseKRIZAN Framework

baseKRIZAN is a modern, lightweight PHP MVC framework focused on security, performance, and developer experience. It features a modular architecture with specialized components for different types of applications including web, API, and console applications.

## Overview

### Requirements
- PHP 8.0 or higher
- MySQL 5.7 or higher
- Composer (for dependency management)
- Redis (optional, for high-performance storage)

### Setting Up
1. Clone the repository
2. Copy `.env.example` to `.env` and configure your environment variables
3. Install dependencies with Composer
4. Set up your database and configure connection details in `.env`
5. Configure web server to point to the `/public` directory
6. Run asset compilation for production: `php bin/console assets:compile`

### Directory Structure
```
/
├── app/
│   ├── baseKRIZAN/        # Framework core
│   │   ├── Assets/        # Asset management system
│   │   │   ├── Strategy/  # Asset handling strategies
│   │   │   └── Processor/ # Asset processing components
│   │   ├── BORNA/         # Security module
│   │   │   ├── Analyzers/ # Security analysis components
│   │   │   ├── Storage/   # Security data storage
│   │   │   └── Enum/      # Security enums and types
│   │   ├── Bootstrap/     # Bootstrap components
│   │   ├── Config/        # Configuration utilities
│   │   ├── Database/      # Database abstractions
│   │   ├── Error/         # Error handling and logging
│   │   ├── Events/        # Event system
│   │   ├── Http/          # HTTP components
│   │   │   └── Middleware/ # Middleware components
│   │   ├── LUKA/          # Digital Twin performance monitoring
│   │   │   └── Storage/   # Metrics storage
│   │   ├── Modules/       # Core module system
│   │   ├── Providers/     # Service providers
│   │   ├── Routing/       # Routing system
│   │   ├── Security/      # Security components
│   │   ├── Services/      # Core services
│   │   │   └── Cache/     # Caching services
│   │   ├── Session/       # Session management
│   │   ├── Template/      # Template system
│   │   └── Validation/    # Validation utilities
│   ├── config/            # Application configuration
│   │   ├── app_asset_resources.php # Asset configuration
│   │   ├── app_basedependencies.php # Base dependencies
│   │   ├── app_baseroutes.php      # Base routes
│   │   └── env_config.php          # Environment configuration
│   ├── Controllers/       # Application controllers
│   ├── Models/            # Data models
│   └── resources/         # Application resources
│       ├── assets/        # Source assets
│       └── views/         # View templates
└── public/                # Web root
    ├── appassets/         # Compiled core assets
    ├── moduliassets/      # Compiled module assets
    └── srcassets/         # Compiled source assets
```

## Core Architecture

### Bootstrap System
- Modular bootstrap process with different modes (`full`, `minimal`, `api`, `console`)
- Service Provider pattern for component initialization
- Environment configuration management with `.env` support

### Request/Response Cycle
- PSR-7 compatible Request and Response classes
- Middleware architecture with prioritization
- RequestClassifier for detecting request types (web, API, asset)
- Entry point system for centralized request handling

### Dependency Injection
- Container implementation with service registration and resolution
- Support for singleton, prototype, and request-scoped dependencies
- Autowiring capabilities for automatic dependency resolution
- Lazy initialization for optimized performance

### Routing
- Route builder with fluent API for creating routes
- Support for route groups with shared attributes (prefix, middleware, auth)
- Route collection and resolving with pattern matching
- Support for route parameters and constraints (regex patterns)
- Named routes with url generation
- Resource-based routing for RESTful endpoints

## Security Features

### Authentication & Authorization
- Session-based authentication with secure defaults
- API token authentication system with rotation
- Permission-based authorization with role checking
- Cross-Site Request Forgery (CSRF) protection with token validation
- Rate limiting with IP rotation detection

### Middleware Security Stack
- `SecurityHeadersMiddleware` for setting secure HTTP headers
- `SanitizationMiddleware` for input filtering
- `ApiAuthMiddleware` for API authentication
- `CsrfMiddleware` for CSRF token validation
- `RateLimiterMiddleware` for rate limiting and IP rotation detection
- `BornaMiddleware` for advanced security rules (firewall)

### Session Management
- Secure session handling with ID regeneration
- Encrypted session storage
- Multiple session storage backends (file, database)
- Configurable cookie security settings (SameSite, HttpOnly, Secure)
- Session garbage collection and cleanup

### BORNA Security Module
Advanced security monitoring and protection:
- Firewall rules engine with pattern recognition (`FirewallRules.php`)
- IP-based blocking and tracking
- Multi-layered security analysis:
  - `AnomalyAnalyzer` for statistical outlier detection
  - `PatternAnalyzer` for known attack signature matching
  - `GeolocationAnalyzer` for location-based restrictions
  - `BehavioralAnalyzer` for suspicious user behavior
  - `IntegrityAnalyzer` for file and system integrity
- Machine learning threat prediction (`ThreatPredictor.php`)
- Configurable security levels with `SecurityLevel` enum
- Comprehensive security event logging and monitoring
- Flexible storage interfaces (Database, Redis)

## Asset Management System

### Asset Path Resolution
- Centralized path management through `AssetPathResolver` singleton
- Support for different asset types (CSS, JS, images, etc.)
- Automatic path resolution with hash-based filenames for cache busting
- Path resolution for different asset sources (core, modules, src)

### Asset Compilation and Processing
- `AssetCompiler` for minification and optimization
- Flexible pipeline architecture with processor chain
- Multiple processor implementations:
  - CSS optimization (CssAutoprefixer, CssMinifier, CssImageOptimizer)
  - JavaScript processing (JsMinifier, JsModuleBundler)
- Modular processing strategy system

### Asset Strategy System
- Strategy pattern for handling different asset types
- Core asset strategy for framework assets
- Module asset strategy for modular assets
- Source asset strategy for application-specific assets
- Common functionality in AbstractAssetStrategy

### Asset Bundling and Versioning
- Asset bundling for production with `AssetBundler`
- Version-controlled manifests with rollback capability
- CDN integration with path transformation
- Asset file watching for development environments

### Resource Loading
- Enhanced template resource loading with `ResourceLoader`
- CSP nonce support for inline scripts and styles
- Automatic JavaScript and CSS resource inclusion
- Modular organization with configurable priorities

## Module System

### Core Modules
- LUKA Digital Twin - Performance monitoring and metrics collection
- BORNA Security - Advanced security monitoring and protection
- Abstract module implementation for easy extension

### Module Management
- Module lifecycle management (register, bootstrap, activate, deactivate)
- Event-based communication between modules
- Dependency resolution for modules
- Module metadata handling with JSON configuration

### Performance Monitoring (LUKA)
- Complete application metrics collection and analysis
- `MetricsCollector` for capturing performance data
- Storage interfaces for metrics data
- Component tracking:
  - `DatabaseHook` for database query profiling
  - `LoggerHook` for log event capture
  - `EventPublisher` for real-time updates via WebSockets
- Comprehensive dashboard with `LukaManager`
- Request and response timing
- Asset loading performance analysis
- WebSocket integration for real-time monitoring

## Other Components

### Template System
- File-based template system with caching
- Template directive processing for dynamic content
- Layout support with content blocks
- Component-based templating
- Template cache optimization with memory cache
- Support for aliasing and custom extensions

### Database Layer
- PDO-based database abstraction
- DatabaseTable model implementation
- Event dispatching for database operations
- Database connection pooling

### Error Handling & Logging
- Centralized error and exception handling through `UnifiedErrorHandler`
- Support for different response formats (HTML, JSON, XML, plain text)
- Development/production mode with appropriate error details
- PSR-3 compatible Logger implementation
- Categorized logging for different components (security, routing, template, etc.)
- Log rotation and cleanup
- Configurable log levels for different environments

### Caching System
- PSR-16 compatible cache implementation
- File-based cache with optimization
- Memory cache for frequently accessed items
- Cache statistics and monitoring
- Garbage collection for expired items

### Storage Systems
- Flexible storage interfaces for different data types
- Database storage implementation (`DatabaseStorage.php`)
- Redis storage for high-performance environments (`RedisStorage.php`)
- File-based storage options
- Storage monitoring and statistics

### Validation
- Robust validation system with rule-based validation
- Custom validation rules
- Error message customization
- Validation result handling and formatting

### Additional Features
- Event system for loose coupling
- Configuration management
- WebSocket integration for real-time communication
- Rate limiting for API protection
- DataTables integration for dynamic tables
- Firebase Integration for push notifications and database synchronization

## Usage Examples

### Defining Routes

```php
// app/config/app_baseroutes.php

$builder->group([
    'middleware' => 'web', 
    'auth' => false
], function($builder) {
    // Home route
    $builder->get('/', ['homeController', 'home']);
    
    // Register routes
    $builder->get('register/user', ['registerController', 'registrationForm']);
    $builder->post('register/user', ['registerController', 'registerUser']);
});
```

### Creating a Controller

```php
namespace Controllers;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class HomeController extends BaseController
{
    public function home(Request $request): Response
    {
        return $this->response()->render(
            'Home/home.html.php',
            ['message' => 'Welcome to baseKRIZAN Framework'],
            'Home Page'
        );
    }
}
```

### Using Middleware

```php
namespace baseKRIZAN\Http\Middleware;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class CustomMiddleware extends Middleware
{
    public function process(Request $request, callable $next): Response
    {
        // Do something before the request is handled
        
        $response = $next($request);
        
        // Do something after the request is handled
        
        return $response;
    }
}
```

### Dependency Injection

```php
// app/config/app_basedependencies.php

$container->singleton('serviceName', function($container) {
    return new MyService(
        $container->get('dependency1'),
        $container->get('dependency2')
    );
});
```

### Template Rendering

```php
// Using the TemplateRenderer

$renderer->render('template/path.html.php', [
    'variable1' => 'value1',
    'variable2' => 'value2'
]);

// Using template directives
// @component(name="alert")
// @include(template="partials/header.html.php")
// @csrf()
```

### Asset Management

```php
// Get path to an asset
$cssPath = AssetManager::getAssetPath('default/css/app_theme.css');
$modulePath = AssetManager::getAssetPath('dashboard.js', 'admin');

// In templates, use ResourceLoader
ResourceLoader::init();
ResourceLoader::loadResource('default');
ResourceLoader::loadResource('datatable');

// Output in template head section
ResourceLoader::generateHeadResources();

// Output at end of body
ResourceLoader::generateBodyResources();
```

### Security Configuration

```php
// Initialize BORNA security module
$bornaManager = BornaManager::getInstance($logger, $container);

// Set security level
$bornaManager->setSecurityLevel(SecurityLevel::HIGH);

// Analyze request for threats
$analysis = $bornaManager->analyzeRequest($requestData);
if ($analysis['score'] > 75) {
    // Block suspicious request
    $bornaManager->blockIP($requestData['ip'], 'High threat score: ' . $analysis['score']);
}

// Check system integrity
$integrityResults = $bornaManager->checkSystemIntegrity();
```

### Performance Monitoring

```php
// Initialize LUKA digital twin
$lukaManager = LukaManager::getInstance($logger, $container);

// Record application event
$lukaManager->recordEvent('order_placed', ['order_id' => $orderId]);

// Track component execution time
$collector = $lukaManager->getMetricsCollector();
$startTime = microtime(true);
// ... component execution
$executionTime = microtime(true) - $startTime;
$collector->recordComponentExecution('payment_processor', $executionTime);

// Get application state information
$appState = $lukaManager->getApplicationState();
```

### Module System

```php
// Registering a module
ModuleManager::getInstance($logger, $container)
    ->loadModules();

// Using ModuleFacade
ModuleFacade::activate('moduleName');
ModuleFacade::isEnabled('moduleName');
```

### Firebase Notifications

```php
// Send a notification through Firebase
$notificationService->createNotification([
    'title' => 'New message',
    'message' => 'You have a new message',
    'type' => 'message'
], [$userId]);

// Sync with Firebase Realtime Database
$firebaseRealtimeService->syncNotifications($userId, $notifications);
```

## License

This project is licensed under the MIT License.

## About

baseKRIZAN is a personal project developed by Tomislav Križan as a learning exercise and demonstration of modern PHP development practices.